'use strict';

Object.defineProperty(exports, '__esModule', { value: true });

var e2eSelectors = require('@grafana/e2e-selectors');
var uuid = require('uuid');

const getBaseUrl = () => e2e.env("BASE_URL") || e2e.config().baseUrl || "http://localhost:3000";
const fromBaseUrl = (url = "") => new URL(url, getBaseUrl()).href;
const getDashboardUid = (url) => {
  const matches = new URL(url).pathname.match(/\/d\/([^/]+)/);
  if (!matches) {
    throw new Error(`Couldn't parse uid from ${url}`);
  } else {
    return matches[1];
  }
};

const selectOption = (config) => {
  const fullConfig = {
    clickToOpen: true,
    forceClickOption: false,
    ...config
  };
  const { clickToOpen, container, forceClickOption, optionText } = fullConfig;
  container.within(() => {
    if (clickToOpen) {
      e2e().get('[class$="-input-suffix"]').click();
    }
  });
  return e2e.components.Select.option().filter((_, { textContent }) => {
    if (textContent === null) {
      return false;
    } else if (typeof optionText === "string") {
      return textContent.includes(optionText);
    } else {
      return optionText.test(textContent);
    }
  }).scrollIntoView().click({ force: forceClickOption });
};

const setTimeRange = ({ from, to, zone }) => {
  e2e.components.TimePicker.openButton().click();
  if (zone) {
    e2e().contains("button", "Change time settings").click();
    if (e2e.components.TimeZonePicker.containerV2) {
      selectOption({
        clickToOpen: true,
        container: e2e.components.TimeZonePicker.containerV2(),
        optionText: zone
      });
    } else {
      selectOption({
        clickToOpen: true,
        container: e2e.components.TimeZonePicker.container(),
        optionText: zone
      });
    }
  }
  e2e.components.TimePicker.absoluteTimeRangeTitle().click();
  e2e.components.TimePicker.fromField().clear().type(from);
  e2e.components.TimePicker.toField().clear().type(to);
  e2e.components.TimePicker.applyTimeRange().click();
};

const setDashboardTimeRange = (config) => setTimeRange(config);

const addDashboard = (config) => {
  const fullConfig = {
    annotations: [],
    title: `e2e-${uuid.v4()}`,
    variables: [],
    ...config,
    timeRange: {
      from: "2020-01-01 00:00:00",
      to: "2020-01-01 06:00:00",
      zone: "Coordinated Universal Time",
      ...config == null ? void 0 : config.timeRange
    }
  };
  const { annotations, timeRange, title, variables } = fullConfig;
  e2e().logToConsole("Adding dashboard with title:", title);
  e2e.pages.AddDashboard.visit();
  if (annotations.length > 0 || variables.length > 0) {
    e2e.components.PageToolbar.item("Dashboard settings").click();
    addAnnotations(annotations);
    fullConfig.variables = addVariables(variables);
    e2e.components.BackButton.backArrow().should("be.visible").click({ force: true });
  }
  setDashboardTimeRange(timeRange);
  e2e.components.PageToolbar.item("Save dashboard").click();
  e2e.pages.SaveDashboardAsModal.newName().clear().type(title, { force: true });
  e2e.pages.SaveDashboardAsModal.save().click();
  e2e.flows.assertSuccessNotification();
  e2e().logToConsole("Added dashboard with title:", title);
  return e2e().url().should("contain", "/d/").then((url) => {
    const uid = getDashboardUid(url);
    e2e.getScenarioContext().then(({ addedDashboards }) => {
      e2e.setScenarioContext({
        addedDashboards: [...addedDashboards, { title, uid }]
      });
    });
    return e2e().wrap(
      {
        config: fullConfig,
        uid
      },
      { log: false }
    );
  });
};
const addAnnotation = (config, isFirst) => {
  if (isFirst) {
    if (e2e.pages.Dashboard.Settings.Annotations.List.addAnnotationCTAV2) {
      e2e.pages.Dashboard.Settings.Annotations.List.addAnnotationCTAV2().click();
    } else {
      e2e.pages.Dashboard.Settings.Annotations.List.addAnnotationCTA().click();
    }
  } else {
    cy.contains("New query").click();
  }
  const { dataSource, dataSourceForm, name } = config;
  selectOption({
    container: e2e.components.DataSourcePicker.container(),
    optionText: dataSource
  });
  e2e.pages.Dashboard.Settings.Annotations.Settings.name().clear().type(name);
  if (dataSourceForm) {
    dataSourceForm();
  }
};
const addAnnotations = (configs) => {
  if (configs.length > 0) {
    e2e.pages.Dashboard.Settings.General.sectionItems("Annotations").click();
  }
  return configs.forEach((config, i) => addAnnotation(config, i === 0));
};
const VARIABLE_HIDE_LABEL = "Label";
const VARIABLE_HIDE_NOTHING = "";
const VARIABLE_HIDE_VARIABLE = "Variable";
const VARIABLE_TYPE_AD_HOC_FILTERS = "Ad hoc filters";
const VARIABLE_TYPE_CONSTANT = "Constant";
const VARIABLE_TYPE_DATASOURCE = "Datasource";
const VARIABLE_TYPE_QUERY = "Query";
const addVariable = (config, isFirst) => {
  const fullConfig = {
    hide: VARIABLE_HIDE_NOTHING,
    type: VARIABLE_TYPE_QUERY,
    ...config
  };
  if (isFirst) {
    if (e2e.pages.Dashboard.Settings.Variables.List.addVariableCTAV2) {
      e2e.pages.Dashboard.Settings.Variables.List.addVariableCTAV2().click();
    } else {
      e2e.pages.Dashboard.Settings.Variables.List.addVariableCTA().click();
    }
  } else {
    e2e.pages.Dashboard.Settings.Variables.List.newButton().click();
  }
  const { constantValue, dataSource, label, name, query, regex, type, variableQueryForm } = fullConfig;
  if (type !== VARIABLE_TYPE_QUERY) {
    e2e.pages.Dashboard.Settings.Variables.Edit.General.generalTypeSelectV2().should("be.visible").within(() => {
      e2e.components.Select.singleValue().should("have.text", "Query").parent().click();
    });
    e2e.pages.Dashboard.Settings.Variables.Edit.General.generalTypeSelectV2().find("input").type(`${type}{enter}`);
  }
  if (label) {
    e2e.pages.Dashboard.Settings.Variables.Edit.General.generalLabelInputV2().type(label);
  }
  e2e.pages.Dashboard.Settings.Variables.Edit.General.generalNameInputV2().clear().type(name);
  if (dataSource && (type === VARIABLE_TYPE_AD_HOC_FILTERS || type === VARIABLE_TYPE_DATASOURCE || type === VARIABLE_TYPE_QUERY)) {
    e2e.pages.Dashboard.Settings.Variables.Edit.QueryVariable.queryOptionsDataSourceSelect().should("be.visible").within(() => {
      e2e.components.DataSourcePicker.inputV2().type(`${dataSource}{enter}`);
    });
  }
  if (constantValue && type === VARIABLE_TYPE_CONSTANT) {
    e2e.pages.Dashboard.Settings.Variables.Edit.ConstantVariable.constantOptionsQueryInputV2().type(constantValue);
  }
  if (type === VARIABLE_TYPE_QUERY) {
    if (query) {
      e2e.pages.Dashboard.Settings.Variables.Edit.QueryVariable.queryOptionsQueryInput().type(query);
    }
    if (regex) {
      e2e.pages.Dashboard.Settings.Variables.Edit.QueryVariable.queryOptionsRegExInputV2().type(regex);
    }
    if (variableQueryForm) {
      variableQueryForm(fullConfig);
    }
  }
  e2e().focused().blur();
  e2e.pages.Dashboard.Settings.Variables.Edit.General.previewOfValuesOption().should("exist").within((previewOfValues) => {
    if (type === VARIABLE_TYPE_CONSTANT) {
      expect(previewOfValues.text()).equals(constantValue);
    }
  });
  e2e.pages.Dashboard.Settings.Variables.Edit.General.submitButton().click();
  e2e.pages.Dashboard.Settings.Variables.Edit.General.applyButton().click();
  return fullConfig;
};
const addVariables = (configs) => {
  if (configs.length > 0) {
    e2e.components.Tab.title("Variables").click();
  }
  return configs.map((config, i) => addVariable(config, i === 0));
};

const addDataSource = (config) => {
  const fullConfig = {
    basicAuth: false,
    basicAuthPassword: "",
    basicAuthUser: "",
    expectedAlertMessage: "Data source is working",
    form: () => {
    },
    name: `e2e-${uuid.v4()}`,
    skipTlsVerify: false,
    type: "TestData",
    ...config
  };
  const {
    basicAuth,
    basicAuthPassword,
    basicAuthUser,
    expectedAlertMessage,
    form,
    name,
    skipTlsVerify,
    type,
    timeout,
    awaitHealth
  } = fullConfig;
  if (awaitHealth) {
    e2e().intercept(/health/).as("health");
  }
  e2e().logToConsole("Adding data source with name:", name);
  e2e.pages.AddDataSource.visit();
  e2e.pages.AddDataSource.dataSourcePluginsV2(type).scrollIntoView().should("be.visible").click();
  e2e.pages.DataSource.name().clear();
  e2e.pages.DataSource.name().type(name);
  if (basicAuth) {
    e2e().contains("label", "Basic auth").scrollIntoView().click();
    e2e().contains(".gf-form-group", "Basic Auth Details").should("be.visible").scrollIntoView().within(() => {
      if (basicAuthUser) {
        e2e().get("[placeholder=user]").type(basicAuthUser);
      }
      if (basicAuthPassword) {
        e2e().get("[placeholder=Password]").type(basicAuthPassword);
      }
    });
  }
  if (skipTlsVerify) {
    e2e().contains("label", "Skip TLS Verify").scrollIntoView().click();
  }
  form();
  e2e.pages.DataSource.saveAndTest().click();
  if (awaitHealth) {
    e2e().wait("@health", { timeout: timeout ?? e2e.config().defaultCommandTimeout });
  }
  e2e.pages.DataSource.alert().should("exist").contains(expectedAlertMessage, {
    timeout: timeout ?? e2e.config().defaultCommandTimeout
  });
  e2e().logToConsole("Added data source with name:", name);
  return e2e().url().then(() => {
    e2e.getScenarioContext().then(({ addedDataSources }) => {
      e2e.setScenarioContext({
        addedDataSources: [...addedDataSources, { name }]
      });
    });
    return e2e().wrap(
      {
        config: fullConfig
      },
      { log: false }
    );
  });
};

const scenarioContext = {
  addedDashboards: [],
  addedDataSources: [],
  get lastAddedDashboard() {
    return lastProperty(this.addedDashboards, "title");
  },
  get lastAddedDashboardUid() {
    return lastProperty(this.addedDashboards, "uid");
  },
  get lastAddedDataSource() {
    return lastProperty(this.addedDataSources, "name");
  },
  get lastAddedDataSourceId() {
    return lastProperty(this.addedDataSources, "id");
  }
};
const lastProperty = (items, key) => {
  var _a;
  return ((_a = items[items.length - 1]) == null ? void 0 : _a[key]) ?? "";
};
const getScenarioContext = () => e2e().wrap(
  {
    getScenarioContext: () => ({ ...scenarioContext })
  },
  { log: false }
).invoke({ log: false }, "getScenarioContext");
const setScenarioContext = (newContext) => e2e().wrap(
  {
    setScenarioContext: () => {
      Object.entries(newContext).forEach(([key, value]) => {
        scenarioContext[key] = value;
      });
    }
  },
  { log: false }
).invoke({ log: false }, "setScenarioContext");

const configurePanel = (config) => getScenarioContext().then(({ lastAddedDashboardUid }) => {
  const fullConfig = {
    chartData: {
      method: "POST",
      route: "/api/ds/query"
    },
    dashboardUid: lastAddedDashboardUid,
    matchScreenshot: false,
    saveDashboard: true,
    screenshotName: "panel-visualization",
    visitDashboardAtStart: true,
    ...config
  };
  const {
    chartData,
    dashboardUid,
    dataSourceName,
    isEdit,
    matchScreenshot,
    panelTitle,
    queriesForm,
    screenshotName,
    timeRange,
    visitDashboardAtStart,
    visualizationName,
    timeout
  } = fullConfig;
  if (visitDashboardAtStart) {
    e2e.flows.openDashboard({ uid: dashboardUid });
  }
  if (isEdit) {
    e2e.components.Panels.Panel.title(panelTitle).click();
    e2e.components.Panels.Panel.headerItems("Edit").click();
  } else {
    try {
      e2e.components.PageToolbar.itemButton("Add panel button").should("be.visible");
      e2e.components.PageToolbar.itemButton("Add panel button").click();
    } catch (e) {
      e2e.components.PageToolbar.item("Show more items").click();
      e2e.components.PageToolbar.item("Add panel button").last().click();
    }
    e2e.pages.AddDashboard.itemButton("Add new visualization menu item").should("be.visible");
    e2e.pages.AddDashboard.itemButton("Add new visualization menu item").click();
  }
  if (timeRange) {
    setDashboardTimeRange(timeRange);
  }
  e2e().intercept(chartData.method, chartData.route).as("chartData");
  if (dataSourceName) {
    selectOption({
      container: e2e.components.DataSourcePicker.container(),
      optionText: dataSourceName
    });
  }
  e2e().wait(2e3);
  const changeTitle = panelTitle && !isEdit;
  if (changeTitle || visualizationName) {
    if (changeTitle && panelTitle) {
      e2e.components.PanelEditor.OptionsPane.fieldLabel("Panel options Title").type(`{selectall}${panelTitle}`);
    }
    if (visualizationName) {
      e2e.components.PluginVisualization.item(visualizationName).scrollIntoView().click();
      e2e().wait(2e3);
    }
  } else {
    closeOptions();
  }
  if (queriesForm) {
    queriesForm(fullConfig);
    e2e().wait("@chartData");
    e2e().wait(1e3);
  }
  e2e.components.RefreshPicker.runButtonV2().first().click({ force: true });
  e2e().wait("@chartData");
  e2e().wait(timeout ?? e2e.config().defaultCommandTimeout);
  if (matchScreenshot) {
    let visualization;
    visualization = e2e.components.Panels.Panel.containerByTitle(panelTitle).find(".panel-content");
    visualization.scrollIntoView().screenshot(screenshotName);
    e2e().compareScreenshots(screenshotName);
  }
  return e2e().wrap({ config: fullConfig }, { log: false });
});
const closeOptions = () => e2e.components.PanelEditor.toggleVizOptions().click();
const VISUALIZATION_ALERT_LIST = "Alert list";
const VISUALIZATION_BAR_GAUGE = "Bar gauge";
const VISUALIZATION_CLOCK = "Clock";
const VISUALIZATION_DASHBOARD_LIST = "Dashboard list";
const VISUALIZATION_GAUGE = "Gauge";
const VISUALIZATION_GRAPH = "Graph";
const VISUALIZATION_HEAT_MAP = "Heatmap";
const VISUALIZATION_LOGS = "Logs";
const VISUALIZATION_NEWS = "News";
const VISUALIZATION_PIE_CHART = "Pie Chart";
const VISUALIZATION_PLUGIN_LIST = "Plugin list";
const VISUALIZATION_POLYSTAT = "Polystat";
const VISUALIZATION_STAT = "Stat";
const VISUALIZATION_TABLE = "Table";
const VISUALIZATION_TEXT = "Text";
const VISUALIZATION_WORLD_MAP = "Worldmap Panel";

const addPanel = (config) => getScenarioContext().then(
  ({ lastAddedDataSource }) => configurePanel({
    dataSourceName: lastAddedDataSource,
    panelTitle: `e2e-${uuid.v4()}`,
    ...config,
    isEdit: false
  })
);

const assertSuccessNotification = () => {
  if (e2e.components.Alert.alertV2) {
    e2e.components.Alert.alertV2("success").should("exist");
  } else {
    e2e.components.Alert.alert("success").should("exist");
  }
};

const deleteDashboard = ({ quick = false, title, uid }) => {
  e2e().logToConsole("Deleting dashboard with uid:", uid);
  if (quick) {
    quickDelete$1(uid);
  } else {
    uiDelete$1(uid, title);
  }
  e2e().logToConsole("Deleted dashboard with uid:", uid);
  e2e.getScenarioContext().then(({ addedDashboards }) => {
    e2e.setScenarioContext({
      addedDashboards: addedDashboards.filter((dashboard) => {
        return dashboard.title !== title && dashboard.uid !== uid;
      })
    });
  });
};
const quickDelete$1 = (uid) => {
  e2e().request("DELETE", fromBaseUrl(`/api/dashboards/uid/${uid}`));
};
const uiDelete$1 = (uid, title) => {
  e2e.pages.Dashboard.visit(uid);
  e2e.components.PageToolbar.item("Dashboard settings").click();
  e2e.pages.Dashboard.Settings.General.deleteDashBoard().click();
  e2e.pages.ConfirmModal.delete().click();
  e2e.flows.assertSuccessNotification();
  e2e.pages.Dashboards.visit();
  if (e2e.components.Search.dashboardItems) {
    e2e.components.Search.dashboardItems().each((item) => e2e().wrap(item).should("not.contain", title));
  } else {
    e2e().get('[aria-label^="Dashboard search item "]').each((item) => e2e().wrap(item).should("not.contain", title));
  }
};

const deleteDataSource = ({ id, name, quick = false }) => {
  e2e().logToConsole("Deleting data source with name:", name);
  if (quick) {
    quickDelete(name);
  } else {
    uiDelete(name);
  }
  e2e().logToConsole("Deleted data source with name:", name);
  e2e.getScenarioContext().then(({ addedDataSources }) => {
    e2e.setScenarioContext({
      addedDataSources: addedDataSources.filter((dataSource) => {
        return dataSource.id !== id && dataSource.name !== name;
      })
    });
  });
};
const quickDelete = (name) => {
  e2e().request("DELETE", fromBaseUrl(`/api/datasources/name/${name}`));
};
const uiDelete = (name) => {
  e2e.pages.DataSources.visit();
  e2e.pages.DataSources.dataSources(name).click();
  e2e.pages.DataSource.delete().click();
  e2e.pages.ConfirmModal.delete().click();
  e2e.pages.DataSources.visit();
  e2e().get('[aria-label^="Data source list item "]').each((item) => e2e().wrap(item).should("not.contain", name));
};

const editPanel = (config) => configurePanel({
  ...config,
  isEdit: true
});

const DEFAULT_USERNAME = "admin";
const DEFAULT_PASSWORD = "admin";
const loginApi = (username, password) => {
  cy.request({
    method: "POST",
    url: fromBaseUrl("/login"),
    body: {
      user: username,
      password
    }
  });
};
const loginUi = (username, password) => {
  e2e().logToConsole("Logging in with username:", username);
  e2e.pages.Login.visit();
  e2e.pages.Login.username().should("be.visible").type(username);
  e2e.pages.Login.password().type(password);
  e2e.pages.Login.submit().click();
  if (password === DEFAULT_PASSWORD) {
    e2e.pages.Login.skip().should("be.visible").click();
  }
  e2e().get(".login-page").should("not.exist");
};
const login = (username = DEFAULT_USERNAME, password = DEFAULT_PASSWORD, loginViaApi = true) => {
  if (loginViaApi) {
    loginApi(username, password);
  } else {
    loginUi(username, password);
  }
  e2e().logToConsole("Logged in with username:", username);
};

const openDashboard = (config) => getScenarioContext().then(({ lastAddedDashboardUid }) => {
  const fullConfig = {
    uid: lastAddedDashboardUid,
    ...config
  };
  const { timeRange, uid, queryParams } = fullConfig;
  e2e.pages.Dashboard.visit(uid, queryParams);
  if (timeRange) {
    setDashboardTimeRange(timeRange);
  }
  return e2e().wrap({ config: fullConfig }, { log: false });
});

var PanelMenuItems = /* @__PURE__ */ ((PanelMenuItems2) => {
  PanelMenuItems2["Edit"] = "Edit";
  PanelMenuItems2["Inspect"] = "Inspect";
  PanelMenuItems2["More"] = "More...";
  PanelMenuItems2["Extensions"] = "Extensions";
  return PanelMenuItems2;
})(PanelMenuItems || {});
const openPanelMenuItem = (menu, panelTitle = "Panel Title") => {
  detectPanelType(panelTitle, (isAngularPanel) => {
    if (isAngularPanel) {
      e2e.components.Panels.Panel.title(panelTitle).should("be.visible").click();
      e2e.components.Panels.Panel.headerItems(menu).should("be.visible").click();
    } else {
      e2e.components.Panels.Panel.menu(panelTitle).click({ force: true });
      e2e.components.Panels.Panel.menuItems(menu).should("be.visible").click();
    }
  });
};
const openPanelMenuExtension = (extensionTitle, panelTitle = "Panel Title") => {
  const menuItem = "Extensions" /* Extensions */;
  detectPanelType(panelTitle, (isAngularPanel) => {
    if (isAngularPanel) {
      e2e.components.Panels.Panel.title(panelTitle).should("be.visible").click();
      e2e.components.Panels.Panel.headerItems(menuItem).should("be.visible").parent().parent().invoke("addClass", "open");
      e2e.components.Panels.Panel.headerItems(extensionTitle).should("be.visible").click();
    } else {
      e2e.components.Panels.Panel.menu(panelTitle).click({ force: true });
      e2e.components.Panels.Panel.menuItems(menuItem).trigger("mouseover", { force: true });
      e2e.components.Panels.Panel.menuItems(extensionTitle).click({ force: true });
    }
  });
};
function detectPanelType(panelTitle, detected) {
  e2e.components.Panels.Panel.title(panelTitle).then((el) => {
    const isAngularPanel = el.find("plugin-component.ng-scope").length > 0;
    if (isAngularPanel) {
      Cypress.log({
        name: "detectPanelType",
        displayName: "detector",
        message: "Angular panel detected, will use legacy selectors."
      });
    }
    detected(isAngularPanel);
  });
}

const revertAllChanges = () => {
  e2e.getScenarioContext().then(({ addedDashboards, addedDataSources }) => {
    addedDashboards.forEach((dashboard) => e2e.flows.deleteDashboard({ ...dashboard, quick: true }));
    addedDataSources.forEach((dataSource) => e2e.flows.deleteDataSource({ ...dataSource, quick: true }));
  });
};

const saveDashboard = () => {
  e2e.components.PageToolbar.item("Save dashboard").click();
  e2e.pages.SaveDashboardModal.save().click();
  e2e.flows.assertSuccessNotification();
};

const importDashboard = (dashboardToImport, queryTimeout, skipPanelValidation) => {
  e2e().visit(fromBaseUrl("/dashboard/import"));
  e2e.components.DashboardImportPage.textarea().should("be.visible").click().invoke("val", JSON.stringify(dashboardToImport));
  e2e.components.DashboardImportPage.submit().should("be.visible").click();
  e2e.components.ImportDashboardForm.name().should("be.visible").click().clear().type(dashboardToImport.title);
  e2e.components.ImportDashboardForm.submit().should("be.visible").click();
  e2e().wait(queryTimeout || 6e3);
  e2e().url().should("contain", "/d/").then((url) => {
    const uid = getDashboardUid(url);
    e2e.getScenarioContext().then(({ addedDashboards }) => {
      e2e.setScenarioContext({
        addedDashboards: [...addedDashboards, { title: dashboardToImport.title, uid }]
      });
    });
    expect(dashboardToImport.uid).to.equal(uid);
  });
  if (!skipPanelValidation) {
    dashboardToImport.panels.forEach((panel) => {
      e2e.components.Panels.Panel.menu(panel.title).click({ force: true });
      e2e.components.Panels.Panel.menuItems("Inspect").should("be.visible").click();
      e2e.components.Tab.title("JSON").should("be.visible").click();
      e2e.components.PanelInspector.Json.content().should("be.visible").contains("Panel JSON").click({ force: true });
      e2e.components.Select.option().should("be.visible").contains("Panel data").click();
      e2e.components.CodeEditor.container().should("be.visible").contains(/"state": "(Done|Streaming)"/);
      e2e.components.Drawer.General.close().click();
    });
  }
};

const importDashboards = async (dirPath, queryTimeout, skipPanelValidation) => {
  e2e().getJSONFilesFromDir(dirPath).then((jsonFiles) => {
    jsonFiles.forEach((file) => {
      importDashboard(file, queryTimeout || 6e3, skipPanelValidation);
    });
  });
};

var flows = /*#__PURE__*/Object.freeze({
  __proto__: null,
  VISUALIZATION_ALERT_LIST: VISUALIZATION_ALERT_LIST,
  VISUALIZATION_BAR_GAUGE: VISUALIZATION_BAR_GAUGE,
  VISUALIZATION_CLOCK: VISUALIZATION_CLOCK,
  VISUALIZATION_DASHBOARD_LIST: VISUALIZATION_DASHBOARD_LIST,
  VISUALIZATION_GAUGE: VISUALIZATION_GAUGE,
  VISUALIZATION_GRAPH: VISUALIZATION_GRAPH,
  VISUALIZATION_HEAT_MAP: VISUALIZATION_HEAT_MAP,
  VISUALIZATION_LOGS: VISUALIZATION_LOGS,
  VISUALIZATION_NEWS: VISUALIZATION_NEWS,
  VISUALIZATION_PIE_CHART: VISUALIZATION_PIE_CHART,
  VISUALIZATION_PLUGIN_LIST: VISUALIZATION_PLUGIN_LIST,
  VISUALIZATION_POLYSTAT: VISUALIZATION_POLYSTAT,
  VISUALIZATION_STAT: VISUALIZATION_STAT,
  VISUALIZATION_TABLE: VISUALIZATION_TABLE,
  VISUALIZATION_TEXT: VISUALIZATION_TEXT,
  VISUALIZATION_WORLD_MAP: VISUALIZATION_WORLD_MAP,
  addDashboard: addDashboard,
  VARIABLE_HIDE_LABEL: VARIABLE_HIDE_LABEL,
  VARIABLE_HIDE_NOTHING: VARIABLE_HIDE_NOTHING,
  VARIABLE_HIDE_VARIABLE: VARIABLE_HIDE_VARIABLE,
  VARIABLE_TYPE_AD_HOC_FILTERS: VARIABLE_TYPE_AD_HOC_FILTERS,
  VARIABLE_TYPE_CONSTANT: VARIABLE_TYPE_CONSTANT,
  VARIABLE_TYPE_DATASOURCE: VARIABLE_TYPE_DATASOURCE,
  VARIABLE_TYPE_QUERY: VARIABLE_TYPE_QUERY,
  addDataSource: addDataSource,
  addPanel: addPanel,
  assertSuccessNotification: assertSuccessNotification,
  deleteDashboard: deleteDashboard,
  deleteDataSource: deleteDataSource,
  editPanel: editPanel,
  login: login,
  openDashboard: openDashboard,
  PanelMenuItems: PanelMenuItems,
  openPanelMenuItem: openPanelMenuItem,
  openPanelMenuExtension: openPanelMenuExtension,
  revertAllChanges: revertAllChanges,
  saveDashboard: saveDashboard,
  selectOption: selectOption,
  importDashboard: importDashboard,
  importDashboards: importDashboards
});

const Selector = {
  fromAriaLabel: (selector) => `[aria-label="${selector}"]`,
  fromDataTestId: (selector) => `[data-testid="${selector}"]`,
  fromSelector: (selector) => selector
};

const processSelectors = (e2eObjects, selectors) => {
  const logOutput = (data) => e2e().logToConsole("Retrieving Selector:", data);
  const keys = Object.keys(selectors);
  for (let index = 0; index < keys.length; index++) {
    const key = keys[index];
    const value = selectors[key];
    if (key === "url") {
      e2eObjects["visit"] = (args, queryParams) => {
        let parsedUrl = "";
        if (typeof value === "string") {
          parsedUrl = fromBaseUrl(value);
        }
        if (typeof value === "function" && args) {
          parsedUrl = fromBaseUrl(value(args));
        }
        e2e().logToConsole("Visiting", parsedUrl);
        if (queryParams) {
          return e2e().visit({ url: parsedUrl, qs: queryParams });
        } else {
          return e2e().visit(parsedUrl);
        }
      };
      continue;
    }
    if (typeof value === "string") {
      e2eObjects[key] = (options) => {
        logOutput(value);
        const selector = value.startsWith("data-testid") ? Selector.fromDataTestId(value) : Selector.fromAriaLabel(value);
        return e2e().get(selector, options);
      };
      continue;
    }
    if (typeof value === "function") {
      e2eObjects[key] = function(textOrOptions, options) {
        if (arguments.length === 0) {
          const selector = value(void 0);
          logOutput(selector);
          return e2e().get(selector);
        }
        if (arguments.length === 1) {
          if (typeof textOrOptions === "string") {
            const selectorText = value(textOrOptions);
            const selector2 = selectorText.startsWith("data-testid") ? Selector.fromDataTestId(selectorText) : Selector.fromAriaLabel(selectorText);
            logOutput(selector2);
            return e2e().get(selector2);
          }
          const selector = value(void 0);
          logOutput(selector);
          return e2e().get(selector, textOrOptions);
        }
        if (arguments.length === 2 && typeof textOrOptions === "string") {
          const text = textOrOptions;
          const selectorText = value(text);
          const selector = text.startsWith("data-testid") ? Selector.fromDataTestId(selectorText) : Selector.fromAriaLabel(selectorText);
          logOutput(selector);
          return e2e().get(selector, options);
        }
      };
      continue;
    }
    if (typeof value === "object") {
      e2eObjects[key] = processSelectors({}, value);
    }
  }
  return e2eObjects;
};
const e2eFactory = ({ selectors }) => {
  const e2eObjects = {};
  processSelectors(e2eObjects, selectors);
  return { ...e2eObjects };
};

const benchmark = ({
  name,
  skipScenario = false,
  repeat,
  duration,
  appStats,
  dashboard
}) => {
  if (skipScenario) {
    describe(name, () => {
      it.skip(name, () => {
      });
    });
  }
  describe(name, () => {
    before(() => {
      e2e.flows.login(e2e.env("USERNAME"), e2e.env("PASSWORD"));
    });
    beforeEach(() => {
      e2e.flows.importDashboards(dashboard.folder, 1e3, dashboard.skipPanelValidation);
      Cypress.Cookies.preserveOnce("grafana_session");
    });
    afterEach(() => e2e.flows.revertAllChanges());
    after(() => {
      e2e().clearCookies();
    });
    Array(repeat).fill(0).map((_, i) => {
      const testName = `${name}-${i}`;
      return it(testName, () => {
        e2e.flows.openDashboard();
        e2e().wait(dashboard.delayAfterOpening);
        if (appStats) {
          const startCollecting = appStats.startCollecting;
          if (startCollecting) {
            e2e().window().then((win) => startCollecting(win));
          }
          e2e().startBenchmarking(testName);
          e2e().wait(duration);
          e2e().window().then((win) => {
            e2e().stopBenchmarking(testName, appStats.collect(win));
          });
        } else {
          e2e().startBenchmarking(testName);
          e2e().wait(duration);
          e2e().stopBenchmarking(testName, {});
        }
      });
    });
  });
};

const e2eScenario = ({
  describeName,
  itName,
  scenario,
  skipScenario = false,
  addScenarioDataSource = false,
  addScenarioDashBoard = false,
  loginViaApi = true
}) => {
  describe(describeName, () => {
    if (skipScenario) {
      it.skip(itName, () => scenario());
    } else {
      before(() => e2e.flows.login(e2e.env("USERNAME"), e2e.env("PASSWORD"), loginViaApi));
      beforeEach(() => {
        Cypress.Cookies.preserveOnce("grafana_session");
        if (addScenarioDataSource) {
          e2e.flows.addDataSource();
        }
        if (addScenarioDashBoard) {
          e2e.flows.addDashboard();
        }
      });
      afterEach(() => e2e.flows.revertAllChanges());
      after(() => e2e().clearCookies());
      it(itName, () => scenario());
      it("temporary", () => {
      });
    }
  });
};

const undo = () => {
  switch (Cypress.platform) {
    case "darwin" /* osx */:
      return "{cmd}z";
    default:
      return "{ctrl}z";
  }
};

var typings = /*#__PURE__*/Object.freeze({
  __proto__: null,
  undo: undo
});

const e2eObject = {
  env: (args) => Cypress.env(args),
  config: () => Cypress.config(),
  blobToBase64String: (blob) => Cypress.Blob.blobToBase64String(blob),
  imgSrcToBlob: (url) => Cypress.Blob.imgSrcToBlob(url),
  scenario: (args) => e2eScenario(args),
  benchmark,
  pages: e2eFactory({ selectors: e2eSelectors.selectors.pages }),
  typings,
  components: e2eFactory({ selectors: e2eSelectors.selectors.components }),
  flows,
  getScenarioContext,
  setScenarioContext,
  getSelectors: (selectors2) => e2eFactory({ selectors: selectors2 })
};
const e2e = Object.assign(() => cy, e2eObject);

exports.e2e = e2e;
//# sourceMappingURL=index.js.map
