import { FieldType, TIME_SERIES_TIME_FIELD_NAME, TIME_SERIES_VALUE_FIELD_NAME } from '../types/dataFrame.js';
import '@grafana/schema';
import '../datetime/moment_wrapper.js';
import '../types/datasource.js';
import 'lodash';
import '../types/legacyEvents.js';
import { formatLabels } from '../utils/labels.js';

function getFrameDisplayName(frame, index) {
  if (frame.name) {
    return frame.name;
  }
  const valuesWithLabels = [];
  for (const field of frame.fields) {
    if (field.labels && Object.keys(field.labels).length > 0) {
      valuesWithLabels.push(field);
    }
  }
  if (valuesWithLabels.length === 1) {
    return formatLabels(valuesWithLabels[0].labels);
  }
  if (index === void 0) {
    return frame.fields.filter((f) => f.type !== FieldType.time).map((f) => getFieldDisplayName(f, frame)).join(", ");
  }
  if (frame.refId) {
    return `Series (${frame.refId})`;
  }
  return `Series (${index})`;
}
function getFieldDisplayName(field, frame, allFrames) {
  var _a, _b;
  const existingTitle = (_a = field.state) == null ? void 0 : _a.displayName;
  const multipleFrames = Boolean(allFrames && allFrames.length > 1);
  if (existingTitle && multipleFrames === ((_b = field.state) == null ? void 0 : _b.multipleFrames)) {
    return existingTitle;
  }
  const displayName = calculateFieldDisplayName(field, frame, allFrames);
  field.state = field.state || {};
  field.state.displayName = displayName;
  field.state.multipleFrames = multipleFrames;
  return displayName;
}
function calculateFieldDisplayName(field, frame, allFrames) {
  var _a, _b, _c;
  const hasConfigTitle = ((_a = field.config) == null ? void 0 : _a.displayName) && ((_b = field.config) == null ? void 0 : _b.displayName.length);
  let displayName = hasConfigTitle ? field.config.displayName : field.name;
  if (hasConfigTitle) {
    return displayName;
  }
  if (frame && ((_c = field.config) == null ? void 0 : _c.displayNameFromDS)) {
    return field.config.displayNameFromDS;
  }
  if (field.type === FieldType.time && !field.labels) {
    return displayName != null ? displayName : TIME_SERIES_TIME_FIELD_NAME;
  }
  let parts = [];
  let frameNamesDiffer = false;
  if (allFrames && allFrames.length > 1) {
    for (let i = 1; i < allFrames.length; i++) {
      const frame2 = allFrames[i];
      if (frame2.name !== allFrames[i - 1].name) {
        frameNamesDiffer = true;
        break;
      }
    }
  }
  let frameNameAdded = false;
  let labelsAdded = false;
  if (frameNamesDiffer && (frame == null ? void 0 : frame.name)) {
    parts.push(frame.name);
    frameNameAdded = true;
  }
  if (field.name && field.name !== TIME_SERIES_VALUE_FIELD_NAME) {
    parts.push(field.name);
  }
  if (field.labels && frame) {
    let singleLabelName = getSingleLabelName(allFrames != null ? allFrames : [frame]);
    if (!singleLabelName) {
      let allLabels = formatLabels(field.labels);
      if (allLabels) {
        parts.push(allLabels);
        labelsAdded = true;
      }
    } else if (field.labels[singleLabelName]) {
      parts.push(field.labels[singleLabelName]);
      labelsAdded = true;
    }
  }
  if (frame && !frameNameAdded && !labelsAdded && field.name === TIME_SERIES_VALUE_FIELD_NAME) {
    if (frame.name && frame.name.length > 0) {
      parts.push(frame.name);
      frameNameAdded = true;
    }
  }
  if (parts.length) {
    displayName = parts.join(" ");
  } else if (field.name) {
    displayName = field.name;
  } else {
    displayName = TIME_SERIES_VALUE_FIELD_NAME;
  }
  if (displayName === field.name) {
    displayName = getUniqueFieldName(field, frame);
  }
  return displayName;
}
function getUniqueFieldName(field, frame) {
  let dupeCount = 0;
  let foundSelf = false;
  if (frame) {
    for (let i = 0; i < frame.fields.length; i++) {
      const otherField = frame.fields[i];
      if (field === otherField) {
        foundSelf = true;
        if (dupeCount > 0) {
          dupeCount++;
          break;
        }
      } else if (field.name === otherField.name) {
        dupeCount++;
        if (foundSelf) {
          break;
        }
      }
    }
  }
  if (dupeCount) {
    return `${field.name} ${dupeCount}`;
  }
  return field.name;
}
function getSingleLabelName(frames) {
  let singleName = null;
  for (let i = 0; i < frames.length; i++) {
    const frame = frames[i];
    for (const field of frame.fields) {
      if (!field.labels) {
        continue;
      }
      for (const labelKey in field.labels) {
        if (singleName === null) {
          singleName = labelKey;
        } else if (labelKey !== singleName) {
          return null;
        }
      }
    }
  }
  return singleName;
}

export { getFieldDisplayName, getFrameDisplayName };
//# sourceMappingURL=fieldState.js.map
