import { cloneDeep, isNumber, unset, set, get } from 'lodash';
import { useRef, useMemo } from 'react';
import usePrevious from 'react-use/lib/usePrevious';
import { VariableFormatID } from '@grafana/schema';
import { FieldType } from '../types/dataFrame.js';
import { guessFieldTypeForField } from '../dataframe/processDataFrame.js';
import '../datetime/moment_wrapper.js';
import '../types/datasource.js';
import { FieldColorModeId } from '../types/fieldColor.js';
import '../types/legacyEvents.js';
import '../datetime/rangeutil.js';
import '../datetime/timezones.js';
import '../datetime/formats.js';
import 'moment-timezone';
import 'date-fns';
import 'date-fns/add';
import 'date-fns/intervalToDuration';
import { compareArrayValues, compareDataFrameStructures } from '../dataframe/frameComparisons.js';
import { asHexString } from '../themes/colorManipulator.js';
import { fieldMatchers } from '../transformations/matchers.js';
import '../transformations/transformers/calculateField.js';
import '../transformations/transformers/concat.js';
import '../transformations/transformers/convertFieldType.js';
import '../transformations/transformers/ensureColumns.js';
import '../transformations/transformers/filter.js';
import '../transformations/transformers/filterByName.js';
import '../transformations/transformers/filterByRefId.js';
import '../transformations/transformers/filterByValue.js';
import '../transformations/transformers/groupBy.js';
import '../transformations/transformers/groupingToMatrix.js';
import '../transformations/transformers/histogram.js';
import '../transformations/transformers/joinByField.js';
import '../transformations/transformers/labelsToFields.js';
import '../transformations/transformers/limit.js';
import '../transformations/transformers/merge.js';
import '../transformations/transformers/noop.js';
import '../transformations/transformers/order.js';
import '../transformations/transformers/organize.js';
import '../transformations/transformers/reduce.js';
import '../transformations/transformers/rename.js';
import '../transformations/transformers/renameByRegex.js';
import '../transformations/transformers/seriesToRows.js';
import '../transformations/transformers/sortBy.js';
import { ReducerID, reduceField } from '../transformations/fieldReducer.js';
import 'rxjs';
import 'rxjs/operators';
import '../transformations/standardTransformersRegistry.js';
import '../transformations/matchers/nameMatcher.js';
import 'fast_array_intersect';
import 'papaparse';
import { getDisplayProcessor, getRawDisplayProcessor } from './displayProcessor.js';
import { getFrameDisplayName } from './fieldState.js';
import { standardFieldConfigEditorRegistry } from './standardFieldConfigEditorRegistry.js';
import './fieldColor.js';
import { mapInternalLinkToExplore } from '../utils/dataLinks.js';
import { getFieldDisplayValuesProxy } from './getFieldDisplayValuesProxy.js';
import { getTemplateProxyForField } from './templateProxies.js';
import '../utils/binaryOperators.js';
import { locationUtil } from '../utils/location.js';

var __defProp = Object.defineProperty;
var __defProps = Object.defineProperties;
var __getOwnPropDescs = Object.getOwnPropertyDescriptors;
var __getOwnPropSymbols = Object.getOwnPropertySymbols;
var __hasOwnProp = Object.prototype.hasOwnProperty;
var __propIsEnum = Object.prototype.propertyIsEnumerable;
var __defNormalProp = (obj, key, value) => key in obj ? __defProp(obj, key, { enumerable: true, configurable: true, writable: true, value }) : obj[key] = value;
var __spreadValues = (a, b) => {
  for (var prop in b || (b = {}))
    if (__hasOwnProp.call(b, prop))
      __defNormalProp(a, prop, b[prop]);
  if (__getOwnPropSymbols)
    for (var prop of __getOwnPropSymbols(b)) {
      if (__propIsEnum.call(b, prop))
        __defNormalProp(a, prop, b[prop]);
    }
  return a;
};
var __spreadProps = (a, b) => __defProps(a, __getOwnPropDescs(b));
function findNumericFieldMinMax(data) {
  let min = null;
  let max = null;
  const reducers = [ReducerID.min, ReducerID.max];
  for (const frame of data) {
    for (const field of frame.fields) {
      if (field.type === FieldType.number) {
        const stats = reduceField({ field, reducers });
        const statsMin = stats[ReducerID.min];
        const statsMax = stats[ReducerID.max];
        if (min === null || statsMin < min) {
          min = statsMin;
        }
        if (max === null || statsMax > max) {
          max = statsMax;
        }
      }
    }
  }
  return { min, max, delta: (max != null ? max : 0) - (min != null ? min : 0) };
}
function applyFieldOverrides(options) {
  var _a;
  if (!options.data) {
    return [];
  }
  const source = options.fieldConfig;
  if (!source) {
    return options.data;
  }
  const fieldConfigRegistry = (_a = options.fieldConfigRegistry) != null ? _a : standardFieldConfigEditorRegistry;
  let seriesIndex = 0;
  let globalRange = void 0;
  const override = [];
  if (source.overrides) {
    for (const rule of source.overrides) {
      const info = fieldMatchers.get(rule.matcher.id);
      if (info) {
        override.push({
          match: info.get(rule.matcher.options),
          properties: rule.properties
        });
      }
    }
  }
  return options.data.map((originalFrame, index) => {
    var _a2, _b;
    const newFrame = __spreadValues({}, originalFrame);
    newFrame.fields = newFrame.fields.map((field) => {
      return __spreadProps(__spreadValues({}, field), {
        config: cloneDeep(field.config),
        state: __spreadValues({}, field.state)
      });
    });
    const scopedVars = {
      __series: { text: "Series", value: { name: getFrameDisplayName(newFrame, index) } }
      // might be missing
    };
    for (const field of newFrame.fields) {
      const config = field.config;
      field.state.scopedVars = __spreadProps(__spreadValues({}, scopedVars), {
        __field: {
          text: "Field",
          value: getTemplateProxyForField(field, newFrame, options.data)
        }
      });
      const context = {
        field,
        data: options.data,
        dataFrameIndex: index,
        replaceVariables: options.replaceVariables,
        fieldConfigRegistry
      };
      setFieldConfigDefaults(config, source.defaults, context);
      for (const rule of override) {
        if (rule.match(field, newFrame, options.data)) {
          for (const prop of rule.properties) {
            setDynamicConfigValue(config, prop, context);
          }
        }
      }
      let type = field.type;
      if (!type || type === FieldType.other) {
        const t = guessFieldTypeForField(field);
        if (t) {
          type = t;
        }
      }
      let range = void 0;
      if (field.type === FieldType.number) {
        if (!globalRange && (!isNumber(config.min) || !isNumber(config.max))) {
          globalRange = findNumericFieldMinMax(options.data);
        }
        const min = (_a2 = config.min) != null ? _a2 : globalRange.min;
        const max = (_b = config.max) != null ? _b : globalRange.max;
        range = { min, max, delta: max - min };
      }
      field.state.seriesIndex = seriesIndex;
      field.state.range = range;
      field.type = type;
      if (field.type !== FieldType.time) {
        seriesIndex++;
      }
      field.display = getDisplayProcessor({
        field,
        theme: options.theme,
        timeZone: options.timeZone
      });
      if (field.config.unit !== "dateTimeFromNow") {
        field.display = cachingDisplayProcessor(field.display, 2500);
      }
      field.getLinks = getLinksSupplier(
        newFrame,
        field,
        field.state.scopedVars,
        context.replaceVariables,
        options.timeZone
      );
    }
    return newFrame;
  });
}
function cachingDisplayProcessor(disp, maxCacheSize = 2500) {
  const caches = /* @__PURE__ */ new Map();
  for (let i = -1; i <= 15; i++) {
    caches.set(i, /* @__PURE__ */ new Map());
  }
  return (value, decimals) => {
    let cache = caches.get(decimals != null ? decimals : -1);
    let v = cache.get(value);
    if (!v) {
      if (cache.size === maxCacheSize) {
        cache.clear();
      }
      v = disp(value, decimals);
      if (v.color) {
        v.color = asHexString(v.color);
      }
      cache.set(value, v);
    }
    return v;
  };
}
function setDynamicConfigValue(config, value, context) {
  const reg = context.fieldConfigRegistry;
  const item = reg.getIfExists(value.id);
  if (!item) {
    return;
  }
  const val = item.process(value.value, context, item.settings);
  const remove = val === void 0 || val === null;
  if (remove) {
    if (item.isCustom && config.custom) {
      unset(config.custom, item.path);
    } else {
      unset(config, item.path);
    }
  } else {
    if (item.isCustom) {
      if (!config.custom) {
        config.custom = {};
      }
      set(config.custom, item.path, val);
    } else {
      set(config, item.path, val);
    }
  }
}
function setFieldConfigDefaults(config, defaults, context) {
  if (config.links && defaults.links) {
    config.links = [...config.links, ...defaults.links];
  }
  for (const fieldConfigProperty of context.fieldConfigRegistry.list()) {
    if (fieldConfigProperty.isCustom && !config.custom) {
      config.custom = {};
    }
    processFieldConfigValue(
      fieldConfigProperty.isCustom ? config.custom : config,
      fieldConfigProperty.isCustom ? defaults.custom : defaults,
      fieldConfigProperty,
      context
    );
  }
  validateFieldConfig(config);
}
function processFieldConfigValue(destination, source, fieldConfigProperty, context) {
  const currentConfig = get(destination, fieldConfigProperty.path);
  if (currentConfig === null || currentConfig === void 0) {
    const item = context.fieldConfigRegistry.getIfExists(fieldConfigProperty.id);
    if (!item) {
      return;
    }
    if (item && item.shouldApply(context.field)) {
      const val = item.process(get(source, item.path), context, item.settings);
      if (val !== void 0 && val !== null) {
        set(destination, item.path, val);
      }
    }
  }
}
function validateFieldConfig(config) {
  const { thresholds } = config;
  if (!config.color) {
    if (thresholds) {
      config.color = {
        mode: FieldColorModeId.Thresholds
      };
    }
  } else if (!config.color.mode) {
    delete config.color;
  }
  if (config.hasOwnProperty("min") && config.hasOwnProperty("max") && config.min > config.max) {
    const tmp = config.max;
    config.max = config.min;
    config.min = tmp;
  }
}
const getLinksSupplier = (frame, field, fieldScopedVars, replaceVariables, timeZone) => (config) => {
  if (!field.config.links || field.config.links.length === 0) {
    return [];
  }
  return field.config.links.map((link) => {
    var _a;
    let dataFrameVars = {};
    let dataContext = { value: { frame, field } };
    if (config.valueRowIndex !== void 0 && !isNaN(config.valueRowIndex)) {
      dataContext.value.rowIndex = config.valueRowIndex;
      const fieldsProxy = getFieldDisplayValuesProxy({
        frame,
        rowIndex: config.valueRowIndex,
        timeZone
      });
      dataFrameVars = {
        __data: {
          value: {
            name: frame.name,
            refId: frame.refId,
            fields: fieldsProxy
          },
          text: "Data"
        }
      };
    } else {
      dataContext.value.calculatedValue = config.calculatedValue;
    }
    const variables = __spreadProps(__spreadValues(__spreadValues({}, fieldScopedVars), dataFrameVars), {
      __dataContext: dataContext
    });
    if (link.onClick) {
      return {
        href: link.url,
        title: replaceVariables(link.title || "", variables),
        target: link.targetBlank ? "_blank" : void 0,
        onClick: (evt, origin) => {
          link.onClick({
            origin: origin != null ? origin : field,
            e: evt,
            replaceVariables: (v) => replaceVariables(v, variables)
          });
        },
        origin: field
      };
    }
    if (link.internal) {
      return mapInternalLinkToExplore({
        link,
        internalLink: link.internal,
        scopedVars: variables,
        field,
        range: (_a = link.internal.range) != null ? _a : {},
        replaceVariables
      });
    }
    let href = link.onBuildUrl ? link.onBuildUrl({
      origin: field,
      replaceVariables
    }) : link.url;
    if (href) {
      href = locationUtil.assureBaseUrl(href.replace(/\n/g, ""));
      href = replaceVariables(href, variables, VariableFormatID.UriEncode);
      href = locationUtil.processUrl(href);
    }
    const info = {
      href,
      title: replaceVariables(link.title || "", variables),
      target: link.targetBlank ? "_blank" : void 0,
      origin: field
    };
    return info;
  });
};
function applyRawFieldOverrides(data) {
  if (!data || data.length === 0) {
    return [];
  }
  const newData = [...data];
  const processor = getRawDisplayProcessor();
  for (let frameIndex = 0; frameIndex < newData.length; frameIndex++) {
    const newFrame = __spreadValues({}, newData[frameIndex]);
    const newFields = [...newFrame.fields];
    for (let fieldIndex = 0; fieldIndex < newFields.length; fieldIndex++) {
      newFields[fieldIndex] = __spreadProps(__spreadValues({}, newFields[fieldIndex]), {
        display: processor
      });
    }
    newData[frameIndex] = __spreadProps(__spreadValues({}, newFrame), {
      fields: newFields
    });
  }
  return newData;
}
function useFieldOverrides(plugin, fieldConfig, data, timeZone, theme, replace) {
  const fieldConfigRegistry = plugin == null ? void 0 : plugin.fieldConfigRegistry;
  const structureRev = useRef(0);
  const prevSeries = usePrevious(data == null ? void 0 : data.series);
  return useMemo(() => {
    if (!fieldConfigRegistry || !fieldConfig || !data) {
      return;
    }
    const series = data == null ? void 0 : data.series;
    if (data.structureRev == null && series && prevSeries && !compareArrayValues(series, prevSeries, compareDataFrameStructures)) {
      structureRev.current++;
    }
    return __spreadProps(__spreadValues({
      structureRev: structureRev.current
    }, data), {
      series: applyFieldOverrides({
        data: series,
        fieldConfig,
        fieldConfigRegistry,
        replaceVariables: replace,
        theme,
        timeZone
      })
    });
  }, [fieldConfigRegistry, fieldConfig, data, prevSeries, timeZone, theme, replace]);
}

export { applyFieldOverrides, applyRawFieldOverrides, findNumericFieldMinMax, getLinksSupplier, setDynamicConfigValue, setFieldConfigDefaults, useFieldOverrides, validateFieldConfig };
//# sourceMappingURL=fieldOverrides.js.map
