import React__default, { memo, useRef, useState, useMemo, useEffect, useCallback } from 'react';
import { useTable, useFilters, useSortBy, useAbsoluteLayout, useResizeColumns, useExpanded, usePagination } from 'react-table';
import { VariableSizeList } from 'react-window';
import { ReducerID } from '@grafana/data';
import { TableCellHeight } from '@grafana/schema';
import { useTheme2 } from '../../themes/ThemeContext.js';
import 'memoize-one';
import '@emotion/react';
import 'tinycolor2';
import { CustomScrollbar } from '../CustomScrollbar/CustomScrollbar.js';
import { Pagination } from '../Pagination/Pagination.js';
import { FooterRow } from './FooterRow.js';
import { HeaderRow } from './HeaderRow.js';
import { TableCell } from './TableCell.js';
import { useResetVariableListSizeCache, useFixScrollbarContainer } from './hooks.js';
import { useTableStateReducer, getInitialState } from './reducer.js';
import { useTableStyles } from './styles.js';
import { getColumns, sortNumber, sortCaseInsensitive, getFooterItems, createFooterCalculationValues, EXPANDER_WIDTH } from './utils.js';

var __defProp = Object.defineProperty;
var __defProps = Object.defineProperties;
var __getOwnPropDescs = Object.getOwnPropertyDescriptors;
var __getOwnPropSymbols = Object.getOwnPropertySymbols;
var __hasOwnProp = Object.prototype.hasOwnProperty;
var __propIsEnum = Object.prototype.propertyIsEnumerable;
var __defNormalProp = (obj, key, value) => key in obj ? __defProp(obj, key, { enumerable: true, configurable: true, writable: true, value }) : obj[key] = value;
var __spreadValues = (a, b) => {
  for (var prop in b || (b = {}))
    if (__hasOwnProp.call(b, prop))
      __defNormalProp(a, prop, b[prop]);
  if (__getOwnPropSymbols)
    for (var prop of __getOwnPropSymbols(b)) {
      if (__propIsEnum.call(b, prop))
        __defNormalProp(a, prop, b[prop]);
    }
  return a;
};
var __spreadProps = (a, b) => __defProps(a, __getOwnPropDescs(b));
const COLUMN_MIN_WIDTH = 150;
const FOOTER_ROW_HEIGHT = 36;
const Table = memo((props) => {
  const {
    ariaLabel,
    data,
    subData,
    height,
    onCellFilterAdded,
    width,
    columnMinWidth = COLUMN_MIN_WIDTH,
    noHeader,
    resizable = true,
    initialSortBy,
    footerOptions,
    showTypeIcons,
    footerValues,
    enablePagination,
    cellHeight = TableCellHeight.Sm
  } = props;
  const listRef = useRef(null);
  const tableDivRef = useRef(null);
  const variableSizeListScrollbarRef = useRef(null);
  const theme = useTheme2();
  const tableStyles = useTableStyles(theme, cellHeight);
  const headerHeight = noHeader ? 0 : tableStyles.rowHeight;
  const [footerItems, setFooterItems] = useState(footerValues);
  const footerHeight = useMemo(() => {
    const EXTENDED_ROW_HEIGHT = FOOTER_ROW_HEIGHT;
    let length = 0;
    if (!footerItems) {
      return 0;
    }
    for (const fv of footerItems) {
      if (Array.isArray(fv) && fv.length > length) {
        length = fv.length;
      }
    }
    if (length > 1) {
      return EXTENDED_ROW_HEIGHT * length;
    }
    return EXTENDED_ROW_HEIGHT;
  }, [footerItems]);
  const memoizedData = useMemo(() => {
    if (!data.fields.length) {
      return [];
    }
    return Array(data.length).fill(0);
  }, [data]);
  const isCountRowsSet = Boolean(
    (footerOptions == null ? void 0 : footerOptions.countRows) && footerOptions.reducer && footerOptions.reducer.length && footerOptions.reducer[0] === ReducerID.count
  );
  const memoizedColumns = useMemo(
    () => getColumns(data, width, columnMinWidth, !!(subData == null ? void 0 : subData.length), footerItems, isCountRowsSet),
    [data, width, columnMinWidth, footerItems, subData, isCountRowsSet]
  );
  const stateReducer = useTableStateReducer(props);
  const options = useMemo(
    () => ({
      columns: memoizedColumns,
      data: memoizedData,
      disableResizing: !resizable,
      stateReducer,
      initialState: getInitialState(initialSortBy, memoizedColumns),
      autoResetFilters: false,
      sortTypes: {
        number: sortNumber,
        // the builtin number type on react-table does not handle NaN values
        "alphanumeric-insensitive": sortCaseInsensitive
        // should be replace with the builtin string when react-table is upgraded, see https://github.com/tannerlinsley/react-table/pull/3235
      }
    }),
    [initialSortBy, memoizedColumns, memoizedData, resizable, stateReducer]
  );
  const {
    getTableProps,
    headerGroups,
    footerGroups,
    rows,
    prepareRow,
    totalColumnsWidth,
    page,
    state,
    gotoPage,
    setPageSize,
    pageOptions
  } = useTable(options, useFilters, useSortBy, useAbsoluteLayout, useResizeColumns, useExpanded, usePagination);
  const extendedState = state;
  useEffect(() => {
    if (!footerOptions) {
      setFooterItems(footerValues);
    }
  }, [footerValues, footerOptions]);
  useEffect(() => {
    var _a, _b, _c;
    if (!footerOptions) {
      return;
    }
    if (!footerOptions.show) {
      setFooterItems(void 0);
      return;
    }
    if (isCountRowsSet) {
      const footerItemsCountRows = [];
      footerItemsCountRows[0] = (_c = (_b = (_a = headerGroups[0]) == null ? void 0 : _a.headers[0]) == null ? void 0 : _b.filteredRows.length.toString()) != null ? _c : data.length.toString();
      setFooterItems(footerItemsCountRows);
      return;
    }
    const footerItems2 = getFooterItems(
      headerGroups[0].headers,
      createFooterCalculationValues(rows),
      footerOptions,
      theme
    );
    setFooterItems(footerItems2);
  }, [footerOptions, theme, state.filters, data]);
  let listHeight = height - (headerHeight + footerHeight);
  if (enablePagination) {
    listHeight -= tableStyles.cellHeight;
  }
  const pageSize = Math.round(listHeight / tableStyles.rowHeight) - 1;
  useEffect(() => {
    if (pageSize <= 0) {
      return;
    }
    setPageSize(pageSize);
  }, [pageSize, setPageSize]);
  useResetVariableListSizeCache(extendedState, listRef, data);
  useFixScrollbarContainer(variableSizeListScrollbarRef, tableDivRef);
  const renderSubTable = useCallback(
    (rowIndex) => {
      var _a, _b;
      if (state.expanded[rowIndex]) {
        const rowSubData = subData == null ? void 0 : subData.find((frame) => {
          var _a2, _b2;
          return ((_b2 = (_a2 = frame.meta) == null ? void 0 : _a2.custom) == null ? void 0 : _b2.parentRowIndex) === rowIndex;
        });
        if (rowSubData) {
          const noHeader2 = !!((_b = (_a = rowSubData.meta) == null ? void 0 : _a.custom) == null ? void 0 : _b.noHeader);
          const subTableStyle = {
            height: tableStyles.rowHeight * (rowSubData.length + (noHeader2 ? 0 : 1)),
            // account for the header with + 1
            background: theme.colors.emphasize(theme.colors.background.primary, 0.015),
            paddingLeft: EXPANDER_WIDTH,
            position: "absolute",
            bottom: 0
          };
          return /* @__PURE__ */ React__default.createElement("div", { style: subTableStyle }, /* @__PURE__ */ React__default.createElement(
            Table,
            {
              data: rowSubData,
              width: width - EXPANDER_WIDTH,
              height: tableStyles.rowHeight * (rowSubData.length + 1),
              noHeader: noHeader2
            }
          ));
        }
      }
      return null;
    },
    [state.expanded, subData, tableStyles.rowHeight, theme.colors, width]
  );
  const RenderRow = useCallback(
    ({ index: rowIndex, style }) => {
      let row = rows[rowIndex];
      if (enablePagination) {
        row = page[rowIndex];
      }
      prepareRow(row);
      return /* @__PURE__ */ React__default.createElement("div", __spreadProps(__spreadValues({}, row.getRowProps({ style })), { className: tableStyles.row }), renderSubTable(rowIndex), row.cells.map((cell, index) => /* @__PURE__ */ React__default.createElement(
        TableCell,
        {
          key: index,
          tableStyles,
          cell,
          onCellFilterAdded,
          columnIndex: index,
          columnCount: row.cells.length
        }
      )));
    },
    [onCellFilterAdded, page, enablePagination, prepareRow, rows, tableStyles, renderSubTable]
  );
  const onNavigate = useCallback(
    (toPage) => {
      gotoPage(toPage - 1);
    },
    [gotoPage]
  );
  const itemCount = enablePagination ? page.length : rows.length;
  let paginationEl = null;
  if (enablePagination) {
    const itemsRangeStart = state.pageIndex * state.pageSize + 1;
    let itemsRangeEnd = itemsRangeStart + state.pageSize - 1;
    const isSmall = width < 550;
    if (itemsRangeEnd > data.length) {
      itemsRangeEnd = data.length;
    }
    paginationEl = /* @__PURE__ */ React__default.createElement("div", { className: tableStyles.paginationWrapper }, /* @__PURE__ */ React__default.createElement(
      Pagination,
      {
        currentPage: state.pageIndex + 1,
        numberOfPages: pageOptions.length,
        showSmallVersion: isSmall,
        onNavigate
      }
    ), isSmall ? null : /* @__PURE__ */ React__default.createElement("div", { className: tableStyles.paginationSummary }, itemsRangeStart, " - ", itemsRangeEnd, " of ", data.length, " rows"));
  }
  const getItemSize = (index) => {
    var _a, _b;
    if (state.expanded[index]) {
      const rowSubData = subData == null ? void 0 : subData.find((frame) => {
        var _a2, _b2;
        return ((_b2 = (_a2 = frame.meta) == null ? void 0 : _a2.custom) == null ? void 0 : _b2.parentRowIndex) === index;
      });
      if (rowSubData) {
        const noHeader2 = !!((_b = (_a = rowSubData.meta) == null ? void 0 : _a.custom) == null ? void 0 : _b.noHeader);
        return tableStyles.rowHeight * (rowSubData.length + 1 + (noHeader2 ? 0 : 1));
      }
    }
    return tableStyles.rowHeight;
  };
  const handleScroll = (event) => {
    const { scrollTop } = event.target;
    if (listRef.current !== null) {
      listRef.current.scrollTo(scrollTop);
    }
  };
  return /* @__PURE__ */ React__default.createElement("div", __spreadProps(__spreadValues({}, getTableProps()), { className: tableStyles.table, "aria-label": ariaLabel, role: "table", ref: tableDivRef }), /* @__PURE__ */ React__default.createElement(CustomScrollbar, { hideVerticalTrack: true }, /* @__PURE__ */ React__default.createElement("div", { className: tableStyles.tableContentWrapper(totalColumnsWidth) }, !noHeader && /* @__PURE__ */ React__default.createElement(HeaderRow, { headerGroups, showTypeIcons, tableStyles }), itemCount > 0 ? /* @__PURE__ */ React__default.createElement("div", { ref: variableSizeListScrollbarRef }, /* @__PURE__ */ React__default.createElement(CustomScrollbar, { onScroll: handleScroll, hideHorizontalTrack: true }, /* @__PURE__ */ React__default.createElement(
    VariableSizeList,
    {
      key: tableStyles.rowHeight,
      height: listHeight,
      itemCount,
      itemSize: getItemSize,
      width: "100%",
      ref: listRef,
      style: { overflow: void 0 }
    },
    RenderRow
  ))) : /* @__PURE__ */ React__default.createElement("div", { style: { height: height - headerHeight }, className: tableStyles.noData }, "No data"), footerItems && /* @__PURE__ */ React__default.createElement(
    FooterRow,
    {
      isPaginationVisible: Boolean(enablePagination),
      footerValues: footerItems,
      footerGroups,
      totalColumnsWidth,
      tableStyles
    }
  ))), paginationEl);
});
Table.displayName = "Table";

export { Table };
//# sourceMappingURL=Table.js.map
