import { useState, useEffect } from 'react';
import useAsync from 'react-use/lib/useAsync';
import { toDataFrame, FieldCache, LogsSortOrder } from '@grafana/data';

const getRowContexts = async (getRowContext, row, limit, logsSortOrder) => {
  const promises = [
    getRowContext(row, {
      limit
    }),
    getRowContext(row, {
      // The start time is inclusive so we will get the one row we are using as context entry
      limit: limit + 1,
      direction: "FORWARD"
    })
  ];
  const results = await Promise.all(promises.map((p) => p.catch((e) => e)));
  const data = results.map((result) => {
    const dataResult = result;
    if (!dataResult.data) {
      return [];
    }
    const data2 = [];
    for (let index = 0; index < dataResult.data.length; index++) {
      const dataFrame = toDataFrame(dataResult.data[index]);
      const fieldCache = new FieldCache(dataFrame);
      const timestampField = fieldCache.getFieldByName("ts");
      const idField = fieldCache.getFieldByName("id");
      for (let fieldIndex = 0; fieldIndex < timestampField.values.length; fieldIndex++) {
        if (idField) {
          if (idField.values.get(fieldIndex) === row.uid) {
            continue;
          }
        } else {
          if (parseInt(timestampField.values.get(fieldIndex), 10) === row.timeEpochMs) {
            continue;
          }
        }
        const lineField = dataFrame.fields.filter((field) => field.name === "line")[0];
        const line = lineField.values.get(fieldIndex);
        data2.push(line);
      }
    }
    return logsSortOrder === LogsSortOrder.Ascending ? data2.reverse() : data2;
  });
  const errors = results.map((result) => {
    const errorResult = result;
    if (!errorResult.message) {
      return "";
    }
    return errorResult.message;
  });
  return {
    data: logsSortOrder === LogsSortOrder.Ascending ? data.reverse() : data,
    errors: logsSortOrder === LogsSortOrder.Ascending ? errors.reverse() : errors
  };
};
const LogRowContextProvider = ({ getRowContext, row, children, logsSortOrder }) => {
  const [limit, setLimit] = useState(10);
  const [result, setResult] = useState(null);
  const [hasMoreContextRows, setHasMoreContextRows] = useState({
    before: true,
    after: true
  });
  const { value } = useAsync(async () => {
    return await getRowContexts(getRowContext, row, limit, logsSortOrder);
  }, [limit]);
  useEffect(() => {
    if (value) {
      setResult((currentResult) => {
        let hasMoreLogsBefore = true, hasMoreLogsAfter = true;
        if (currentResult) {
          const currentResultBefore = currentResult.data[0];
          const currentResultAfter = currentResult.data[1];
          const valueBefore = value.data[0];
          const valueAfter = value.data[1];
          if (!valueBefore || currentResultBefore.length === valueBefore.length) {
            hasMoreLogsBefore = false;
          }
          if (!valueAfter || currentResultAfter.length === valueAfter.length) {
            hasMoreLogsAfter = false;
          }
        }
        setHasMoreContextRows({
          before: hasMoreLogsBefore,
          after: hasMoreLogsAfter
        });
        return value;
      });
    }
  }, [value]);
  return children({
    result: {
      before: result ? result.data[0] : [],
      after: result ? result.data[1] : []
    },
    errors: {
      before: result ? result.errors[0] : void 0,
      after: result ? result.errors[1] : void 0
    },
    hasMoreContextRows,
    updateLimit: () => setLimit(limit + 10),
    limit
  });
};

export { LogRowContextProvider, getRowContexts };
//# sourceMappingURL=LogRowContextProvider.js.map
