import { render, screen, within } from '@testing-library/react';
import userEvent from '@testing-library/user-event';
import React from 'react';
import { Router } from 'react-router-dom';

import { locationService } from '@grafana/runtime';

import { defaultSettings } from '../config';

import { ConfigPage, Props } from './ConfigPage';

const setup = (propOverrides?: Props) => {
  const props = {
    navModel: { main: { text: 'Custom branding' }, node: { text: 'Custom branding' } },
    ...propOverrides,
  };

  render(
    <Router history={locationService.getHistory()}>
      <ConfigPage {...props} />
    </Router>
  );
};

beforeEach(() => {
  jest.clearAllMocks();
});

jest.mock('app/core/config', () => ({
  config: {
    buildInfo: {},
    licenseInfo: {},
    featureToggles: {
      accesscontrol: true,
    },
    bootData: { navTree: [], user: {} },
  },
  getConfig: () => {},
}));

let backendGetMock = () => Promise.resolve(defaultSettings);

jest.mock('@grafana/runtime', () => ({
  ...jest.requireActual('@grafana/runtime'),
  config: {
    featureToggles: {},
    licenseInfo: {},
    buildInfo: {},
  },
  getBackendSrv: () => {
    return {
      get: backendGetMock,
    };
  },
}));

const mockUpdate = jest.fn();
window.fetch = mockUpdate;

describe('ConfigPage', () => {
  it('should render the page with the sections', async () => {
    setup();
    expect(await screen.findByText(/login page/i)).toBeInTheDocument();
    //First section should be expanded
    expect(screen.getByRole('textbox', { name: /login logo/i })).toBeInTheDocument();
    expect(screen.getByText(/footer/i)).toBeInTheDocument();
    expect(screen.queryByRole('textbox', { name: /link text/i })).not.toBeInTheDocument();
    expect(screen.getByText(/other/i)).toBeInTheDocument();
    expect(screen.queryByRole('textbox', { name: /application title/i })).not.toBeInTheDocument();
  });

  it('should expand collapsed sections', async () => {
    setup();
    await userEvent.click(await screen.findByRole('button', { name: /^footer/i }));
    expect(screen.getAllByRole('textbox', { name: /link text/i })).toHaveLength(3);

    await userEvent.click(screen.getByRole('button', { name: /^other/i }));
    expect(screen.getByRole('textbox', { name: /application title/i })).toBeInTheDocument();
  });

  it('should save data from inputs', async () => {
    setup();
    // Wait for the page to load
    await screen.findByText('Login page');
    await typeText(/login logo/i, 'https://test-logo.com');
    await typeText(/login title/i, 'Custom title');
    await typeText(/login subtitle/i, 'Custom subtitle');
    const loginBackgroundPicker = screen.getByTestId('resource-picker-loginBackground');
    await selectResourceType(loginBackgroundPicker, /custom color/i);
    await typeText(/login background/i, '#FFFFFF');
    const loginBoxBackgroundPicker = screen.getByTestId('resource-picker-loginBoxBackground');
    await selectResourceType(loginBoxBackgroundPicker, /upload file/i);
    await uploadFile(loginBoxBackgroundPicker, 'testImage.png');

    // Expand "Other" section
    await userEvent.click(screen.getByRole('button', { name: /^other/i }));
    await typeText(/application title/i, 'Custom app');
    const favIcon = screen.getByTestId('resource-picker-favIcon');
    await typeText(/fav icon/i, 'https://favicon.com');
    await selectResourceType(favIcon, /upload file/i);
    await uploadFile(favIcon, 'favIcon.png');
    const menuLogo = screen.getByTestId('resource-picker-menuLogo');
    await selectResourceType(menuLogo, /upload file/i);
    await uploadFile(menuLogo, 'menuLogoCustom.png');
    await typeText(/loading logo/i, 'Custom loading logo');

    await userEvent.click(screen.getByRole('button', { name: 'Save' }));

    const testData: FormData = mockUpdate.mock.calls[0][1].body;
    const config = JSON.parse(testData.get('config') as string);
    const files = testData.getAll('files');
    expect(files).toMatchObject([{ path: 'testImage.png' }, { path: 'favIcon.png' }, { path: 'menuLogoCustom.png' }]);
    expect(config).toEqual({
      footerLinks: null,
      loginTitle: 'Custom title',
      loginSubtitle: 'Custom subtitle',
      loginLogo: 'https://test-logo.com',
      loginBackground: 'rgb(255, 255, 255)',
      loginBoxBackground: 'testImage.png',
      appTitle: 'Custom app',
      favIcon: 'favIcon.png',
      menuLogo: 'menuLogoCustom.png',
      loadingLogo: 'Custom loading logo',
      appleTouchIcon: '',
    });
  });
  it('should show correct data for editing', async () => {
    const config = {
      footerLinks: null,
      loginTitle: 'Custom title',
      loginSubtitle: 'Custom subtitle',
      loginLogo: 'https://test-logo.com',
      loginBackground: 'rgb(255, 255, 255)',
      loginBoxBackground: 'testImage.png',
      appTitle: 'Custom app',
      favIcon: 'https://favicon.com',
      menuLogo: 'menuLogoCustom.png',
      loadingLogo: '',
      appleTouchIcon: '',
    };
    backendGetMock = () => Promise.resolve(config);
    setup();
    expect(await screen.findByRole('textbox', { name: /login logo/i })).toHaveValue(config.loginLogo);
    expect(screen.getByRole('textbox', { name: /login title/i })).toHaveValue(config.loginTitle);
    expect(screen.getByRole('textbox', { name: /login subtitle/i })).toHaveValue(config.loginSubtitle);
    expect(screen.getByRole('textbox', { name: /login background/i })).toHaveValue(config.loginBackground);
    expect(
      within(screen.getByTestId('resource-picker-loginBoxBackground')).getByRole('radio', { name: /upload file/i })
    ).toBeChecked();
  });

  it('should restore default links on button click', async () => {
    const data = {
      footerLinks: [{ text: 'Custom link', url: 'example.com' }],
      loginTitle: 'Custom title',
      loginSubtitle: 'Custom subtitle',
      loginLogo: 'https://test-logo.com',
      loginBackground: 'rgb(255, 255, 255)',
      loginBoxBackground: 'testImage.png',
      appTitle: 'Custom app',
      favIcon: 'https://favicon.com',
      menuLogo: 'menuLogoCustom.png',
      loadingLogo: '',
      appleTouchIcon: '',
    };
    backendGetMock = () => Promise.resolve(data);
    setup();
    await userEvent.click(await screen.findByRole('button', { name: /^footer/i }));
    await userEvent.click(screen.getByRole('button', { name: 'Restore default links' }));
    await userEvent.click(screen.getByRole('button', { name: 'Confirm Modal Danger Button' }));
    await userEvent.click(screen.getByRole('button', { name: 'Save' }));

    const testData: FormData = mockUpdate.mock.calls[0][1].body;
    const config = testData.get('config') as string;
    expect(JSON.parse(config).footerLinks).toEqual(null);
  });
});

const selectResourceType = async (picker: HTMLElement, name: RegExp | string) => {
  await userEvent.click(within(picker).getByRole('radio', { name }));
};

const typeText = async (name: RegExp | string, text: string) => {
  await userEvent.type(screen.getByRole('textbox', { name }), text);
};

const uploadFile = async (picker: HTMLElement, fileName: string) => {
  await userEvent.upload(
    within(picker).getByTestId('dropzone').querySelector('input[type="file"]')!,
    new File([JSON.stringify({ ping: true })], fileName, { type: 'image/png' })
  );
};
