import React from 'react';
import { css } from '@emotion/css';
import { capitalize } from 'lodash';
import { GrafanaTheme2, PluginSignatureType } from '@grafana/data';
import { useStyles2, Icon, Badge, IconName } from '@grafana/ui';

const SIGNATURE_ICONS: Record<string, IconName> = {
  [PluginSignatureType.grafana]: 'grafana',
  [PluginSignatureType.commercial]: 'shield',
  [PluginSignatureType.community]: 'shield',
  DEFAULT: 'shield-exclamation',
};

type Props = {
  signatureType?: PluginSignatureType;
  signatureOrg?: string;
};

// Shows more information about a valid signature
export function PluginSignatureDetailsBadge({ signatureType, signatureOrg = '' }: Props): React.ReactElement | null {
  const styles = useStyles2(getStyles);

  if (!signatureType && !signatureOrg) {
    return null;
  }

  const signatureTypeText = signatureType === PluginSignatureType.grafana ? 'Grafana Labs' : capitalize(signatureType);
  const signatureIcon = SIGNATURE_ICONS[signatureType || ''] || SIGNATURE_ICONS.DEFAULT;

  return (
    <>
      <DetailsBadge>
        <strong className={styles.strong}>Level:&nbsp;</strong>
        <Icon size="xs" name={signatureIcon} />
        &nbsp;
        {signatureTypeText}
      </DetailsBadge>

      <DetailsBadge>
        <strong className={styles.strong}>Signed by:</strong> {signatureOrg}
      </DetailsBadge>
    </>
  );
}

export const DetailsBadge: React.FC = ({ children }) => {
  const styles = useStyles2(getStyles);

  return <Badge color="green" className={styles.badge} text={<>{children}</>} />;
};

const getStyles = (theme: GrafanaTheme2) => ({
  badge: css`
    background-color: ${theme.colors.background.canvas};
    border-color: ${theme.colors.border.strong};
    color: ${theme.colors.text.secondary};
    margin-left: ${theme.spacing()};
  `,
  strong: css`
    color: ${theme.colors.text.primary};
  `,
  icon: css`
    margin-right: ${theme.spacing(0.5)};
  `,
});
