import React, { FC } from 'react';
import { TimeOfDayPicker, Select, RadioButtonGroup, Field } from '@grafana/ui';
import { dateTime, DateTime, SelectableValue } from '@grafana/data';
import { daysOfWeek, SchedulingFrequency, SchedulingOptions } from '../types';
import { TimeZonePicker } from '@grafana/ui/src/components/TimePicker/TimeZonePicker';

interface Props {
  schedulingOptions: SchedulingOptions;
  onModeChange: (mode: SchedulingFrequency) => void;
  onDayOfWeekChange: (day: SelectableValue<string>) => void;
  onTimeOfDayChange: (time: DateTime) => void;
  onTimeZoneChange: (timeZone: string) => void;
}

const frequencyOptions: SelectableValue[] = [
  { label: 'Weekly', value: SchedulingFrequency.Weekly },
  { label: 'Daily', value: SchedulingFrequency.Daily },
  { label: 'Hourly', value: SchedulingFrequency.Hourly },
  { label: 'Never', value: SchedulingFrequency.Never },
];

export const ReportScheduling: FC<Props> = ({
  onModeChange,
  onDayOfWeekChange,
  onTimeOfDayChange,
  onTimeZoneChange,
  schedulingOptions,
}) => {
  const { frequency, hour, minute, timeZone } = schedulingOptions;

  const selectedDay = daysOfWeek.find(day => schedulingOptions.day === day.value);

  return (
    <>
      <Field label="Frequency">
        <RadioButtonGroup options={frequencyOptions} onChange={onModeChange} value={frequency} />
      </Field>
      {frequency === SchedulingFrequency.Weekly && (
        <>
          <Field label="Day">
            <Select onChange={onDayOfWeekChange} options={daysOfWeek} value={selectedDay} />
          </Field>
          <Field label="Time">
            <TimeOfDayPicker
              onChange={onTimeOfDayChange}
              minuteStep={10}
              value={dateTime(new Date(`January 1, 1970 ${hour}:${minute}`))}
            />
          </Field>
        </>
      )}
      {frequency === SchedulingFrequency.Daily && (
        <Field label="Time">
          <TimeOfDayPicker
            minuteStep={10}
            onChange={onTimeOfDayChange}
            value={dateTime(new Date(`January 1, 1970 ${hour}:${minute}`))}
          />
        </Field>
      )}
      {frequency === SchedulingFrequency.Hourly && (
        <Field label="At minute">
          <TimeOfDayPicker
            showHour={false}
            minuteStep={10}
            onChange={onTimeOfDayChange}
            value={dateTime(new Date(`January 1, 1970 ${hour}:${minute}`))}
          />
        </Field>
      )}
      {frequency !== SchedulingFrequency.Never && (
        <Field label="Time zone">
          <TimeZonePicker value={timeZone} onChange={onTimeZoneChange} />
        </Field>
      )}
    </>
  );
};
