import React from 'react';
import { shallow, mount } from 'enzyme';
import { act } from 'react-dom/test-utils';
import { NavModel } from '@grafana/data';
import { ReportPage, Props } from './ReportPage';
import { initialState, updateReportProp, clearReportState } from './state/reducers';
import { mockToolkitActionCreator, mockToolkitActionCreatorWithoutPayload } from '../../../test/core/redux/mocks';

const blankReport = initialState.report;
const testReport = {
  ...blankReport,
  name: 'Test report',
  dashboardId: 1,
  dashboardName: 'Test dashboard',
  recipients: 'test@me.com',
};

const setup = (propOverrides?: Partial<Props>, renderMethod = shallow) => {
  const props: Props = {
    report: blankReport,
    navModel: { node: {}, main: {} } as NavModel,
    reportId: undefined,
    isLoading: false,

    createReport: jest.fn(),
    updateReport: jest.fn(),
    loadReport: jest.fn(),
    sendTestMail: jest.fn(),
    clearReportState: mockToolkitActionCreatorWithoutPayload(clearReportState),
    updateReportProp: mockToolkitActionCreator(updateReportProp),
  };

  Object.assign(props, propOverrides);

  const wrapper = renderMethod(<ReportPage {...props} />);
  const instance = wrapper.instance() as ReportPage;

  return {
    wrapper,
    instance,
  };
};

describe('ReportPage', () => {
  it('should render New report page when reportId is undefined', () => {
    const { wrapper } = setup();
    // 2 elements since enzyme returns dom element and it's text
    expect(wrapper.findWhere(comp => comp.text() === 'New report')).toHaveLength(2);
  });

  it("should trigger 'create' prop on save when New report page is rendered ", () => {
    const mockCreate = jest.fn();
    const { wrapper } = setup({ createReport: mockCreate, report: testReport });
    wrapper.find('Form').simulate('submit', { preventDefault() {} });
    expect(mockCreate).toHaveBeenCalledTimes(1);
  });

  it('should render Edit [reportName] page when reportId is present', () => {
    const mockLoad = jest.fn();
    const { wrapper } = setup({ loadReport: mockLoad, report: { ...testReport, id: 1 }, reportId: 1 });
    // 2 elements since enzyme returns dom element and it's text
    expect(wrapper.findWhere(comp => comp.text() === 'Edit Test report')).toHaveLength(2);

    // Should also call loadReport on mount
    expect(mockLoad).toHaveBeenCalledTimes(1);
  });

  it("should trigger 'update' prop on save when New report page is rendered ", () => {
    const mockUpdate = jest.fn();
    const { wrapper } = setup({ updateReport: mockUpdate, report: { ...testReport, id: 1 }, reportId: 1 });
    wrapper.find('Form').simulate('submit', { preventDefault() {} });
    expect(mockUpdate).toHaveBeenCalledTimes(1);
  });

  it('should trigger updateReportProp on input fields change event', () => {
    const mockUpdateProp = mockToolkitActionCreator(updateReportProp);
    const { wrapper } = setup({ updateReportProp: mockUpdateProp });

    const dashboard = wrapper
      .find('Form')
      .shallow()
      .find('[name="dashboardId"]')
      .at(0);

    // @ts-ignore
    dashboard.props().onSelected({ id: 1, label: 'test' });
    expect(mockUpdateProp).toHaveBeenCalledWith({ ...blankReport, dashboardId: 1, dashboardName: 'test' });
  });

  it('should correctly set the form values for Edit page', async () => {
    const report = {
      id: 20,
      userId: 1,
      orgId: 1,
      dashboardId: 37,
      dashboardName: 'Datasource tests - Elasticsearch comparison',
      name: 'Test Report',
      recipients: 'me@test.com',
      replyTo: '',
      message:
        'Hi all, \nplease find enclosed a PDF status report. If you have any questions, feel free to get in touch with me!\nHave a great day. \nBest,',
      schedule: {
        frequency: 'weekly',
        day: 'monday',
        hour: 12,
        minute: 0,
        timeZone: 'Europe/Helsinki',
      },
      options: {
        landscape: true,
      },
    };

    //@ts-ignore
    const { wrapper } = setup({ report, reportId: report.id }, mount);
    // Since the form has an async Select, which updates state on mount, it needs to be wrapped in act
    await act(async () => {
      wrapper.update();
    });

    const textFields = ['name', 'replyTo', 'recipients', 'message'];

    textFields.forEach(name => {
      expect(
        wrapper
          .find('Form')
          .find({ name })
          .at(0)
          .prop('defaultValue')
        // @ts-ignore
      ).toEqual(report[name]);
    });

    expect(
      wrapper
        .find('Form')
        .find({ name: 'dashboardId' })
        .at(0)
        .text()
    ).toEqual(report.dashboardName);
  });
});
