import React from 'react';
import { mount, ReactWrapper } from 'enzyme';
import { PresenceIndicators, PresenceIndicatorsProps } from './PresenceIndicators';
import { DashboardModel } from 'app/features/dashboard/state';
import { mockRecentUsers } from './__mocks__/recentUsersMocks';
import { getRecentUsers } from './api';

beforeEach(() => {
  jest.clearAllMocks();
});

jest.mock('./api', () => {
  return {
    getRecentUsers: jest.fn((dashboardId: number) => Promise.resolve(mockRecentUsers.slice(0, dashboardId))),
  };
});
jest.mock('app/core/services/context_srv', () => {
  return {
    contextSrv: { user: { id: 1 } },
  };
});

const setup = async (dashboardId: number): Promise<ReactWrapper> => {
  const props: PresenceIndicatorsProps = {
    dashboard: new DashboardModel({ id: dashboardId }),
    openDrawer: jest.fn(),
  };

  let wrapper = await mount(<PresenceIndicators {...props} />);
  return wrapper;
};

describe('Render', () => {
  it('should render component - no dashboard', async () => {
    const wrapper = await setup(0);
    expect(getRecentUsers).toHaveBeenCalledTimes(0);

    wrapper.update();
    expect(wrapper.find({ 'aria-label': 'Presence indicators container' }).children()).toHaveLength(0);
  });

  it('should render component - only current user', async () => {
    const wrapper = await setup(1);
    expect(getRecentUsers).toHaveBeenCalledTimes(1);
    expect(getRecentUsers).toHaveBeenCalledWith(1);

    wrapper.update();
    expect(wrapper.find({ 'aria-label': 'Presence indicators container' }).children()).toHaveLength(0);
  });

  it('should render component - few users (all should be displayed)', async () => {
    const wrapper = await setup(3);
    expect(getRecentUsers).toHaveBeenCalledTimes(1);
    expect(getRecentUsers).toHaveBeenCalledWith(3);

    wrapper.update();
    expect(wrapper.find({ 'aria-label': 'Avatar icon' })).toHaveLength(2);
    expect(wrapper.exists({ 'aria-label': 'More users icon' })).toEqual(false);
  });

  it('should render component - more users (more icon should be displayed)', async () => {
    const wrapper = await setup(6);

    expect(getRecentUsers).toHaveBeenCalledTimes(1);
    expect(getRecentUsers).toHaveBeenCalledWith(6);

    wrapper.update();
    expect(wrapper.find({ 'aria-label': 'Avatar icon' })).toHaveLength(3);
    expect(wrapper.exists({ 'aria-label': 'More users icon' })).toEqual(true);
  });
});
