import React from 'react';
import ReactDOM from 'react-dom';
import store from 'app/core/store';
import { config } from 'app/core/config';
import { contextSrv } from 'app/core/services/context_srv';
import { dateTime } from '@grafana/data';
import { HasExpired, ExpiresSoon, MaxUsersReached } from './LicenseWarning';
import { OrgRole } from 'app/types';
import { refreshLicenseStats, ActiveUserStats } from './state/api';

const SHOW_EXPIRE_WARNING_DAYS = 30;
const DISMISS_WARNING_FOR_DAYS = 5;
/* The auto dissmiss cannot be too low as it will also trigger the snooze */
const WARNING_CLOSE_TIMEOUT_SEC = 3600;
const LICENSE_WARNING_DISMISS_UNTIL_KEY = 'grafana.licence.warning.dismissUntil';

let warningContainer: HTMLElement | null = null;

interface LicensingSettings {
  activeAdminsAndEditors?: number;
  activeViewers?: number;
  includedAdmins?: number;
  includedViewers?: number;
  slug?: string;
}

export function initLicenseWarnings() {
  const dismissUntil = store.get(LICENSE_WARNING_DISMISS_UNTIL_KEY);
  const hasDismissed = dismissUntil && dismissUntil > dateTime().valueOf();
  const isAdmin = contextSrv.hasRole(OrgRole.Admin);
  const showExpireSoon = isAdmin && willExpireSoon() && !hasDismissed;

  if (
    !isRenderingPanel() &&
    (isExpired() || showExpireSoon || numberOfActiveAdminsReached() || numberOfActiveViewersReached())
  ) {
    setTimeout(renderLicenseWarning, 1000);
  }
}

export function onCloseWarning() {
  const dismissTill = dateTime()
    .add(DISMISS_WARNING_FOR_DAYS, 'd')
    .valueOf();
  store.set(LICENSE_WARNING_DISMISS_UNTIL_KEY, dismissTill);
  getMainView().removeChild(warningContainer);
}

export async function onRefreshWarning() {
  const activeUserStats: ActiveUserStats = await refreshLicenseStats().catch(err => null);
  const settings = (config as any).licensing as LicensingSettings;

  if (activeUserStats) {
    settings.activeAdminsAndEditors = activeUserStats.active_admins_and_editors;
    settings.activeViewers = activeUserStats.active_viewers;
    getMainView().removeChild(warningContainer);
  }

  initLicenseWarnings();
}

function renderLicenseWarning() {
  warningContainer = document.createElement('div');
  const maxAdminsReached = numberOfActiveAdminsReached();
  const maxViewersReached = numberOfActiveViewersReached();

  if (isExpired()) {
    ReactDOM.render(<HasExpired />, warningContainer);
  } else if (maxAdminsReached || maxViewersReached) {
    const settings = (config as any).licensing as LicensingSettings;
    const isAdmin = contextSrv.hasRole(OrgRole.Admin);

    ReactDOM.render(
      <MaxUsersReached
        activeUsers={maxAdminsReached ? settings.activeAdminsAndEditors : settings.activeViewers}
        maxUsers={maxAdminsReached ? settings.includedAdmins : settings.includedViewers}
        type={maxAdminsReached ? 'admins / editors' : 'viewers'}
        slug={settings.slug}
        onRefreshWarning={isAdmin ? onRefreshWarning : null}
      />,
      warningContainer
    );
  } else if (willExpireSoon()) {
    const expiresIn = willExpireInDays();
    ReactDOM.render(<ExpiresSoon days={expiresIn} onCloseWarning={onCloseWarning} />, warningContainer);

    // out hide expire warning in case it' a TV monitor with admin permissions
    setTimeout(onCloseWarning, 1000 * WARNING_CLOSE_TIMEOUT_SEC);
  }

  getMainView().appendChild(warningContainer);
}

function getMainView(): Element {
  return document.getElementsByClassName('main-view')[0];
}

function isExpired(): boolean {
  const { expiry } = config.licenseInfo;
  return !expiry || dateTime(expiry * 1000) < dateTime();
}

function willExpireSoon(): boolean {
  const { expiry } = config.licenseInfo;
  return dateTime(expiry * 1000) < dateTime().add(SHOW_EXPIRE_WARNING_DAYS, 'd');
}

function willExpireInDays(): number {
  const { expiry } = config.licenseInfo;
  return Math.ceil((expiry - dateTime().unix()) / 3600 / 24);
}

function numberOfActiveAdminsReached(): boolean {
  const settings = (config as any).licensing as LicensingSettings;
  return settings.includedAdmins !== -1 && settings.activeAdminsAndEditors > settings.includedAdmins;
}

function numberOfActiveViewersReached(): boolean {
  const settings = (config as any).licensing as LicensingSettings;
  return settings.includedViewers !== -1 && settings.activeViewers > settings.includedViewers;
}

function isSoloPanel(): boolean {
  const soloPanelPattern = /\/d-solo\//;
  const path = window.location.pathname;
  return soloPanelPattern.test(path);
}

function isRenderingPanel(): boolean {
  return isSoloPanel();
}
