import React, { FC } from 'react';
import { css } from 'emotion';
import { config } from 'app/core/config';
import { stylesFactory, Alert } from '@grafana/ui';

export const getStyles = stylesFactory(() => {
  const { theme } = config;

  return {
    container: css`
      position: absolute;
      bottom: 0;
      width: 100%;
      display: flex;
      flex-direction: column;
      justify-content: center;
      z-index: 1;
      box-shadow: 0 0 20px ${theme.colors.dropdownShadow};
    `,
  };
});

interface Props {
  title: string;
  subTitle: JSX.Element;
  severity: 'info' | 'error';
  onButtonClick?(): void;
  buttonText?: string;
}

const Warning: FC<Props> = ({ title, subTitle, severity, onButtonClick, buttonText }) => {
  const styles = getStyles();

  return (
    <div className={styles.container}>
      <div className="page-container">
        <Alert title={title} buttonText={buttonText} onButtonClick={onButtonClick} severity={severity}>
          {subTitle}
        </Alert>
      </div>
    </div>
  );
};

export const HasExpired: FC = () => {
  return (
    <Warning
      title="Your Grafana Enterprise license has expired"
      subTitle={
        <>
          <a href="https://grafana.com/docs/grafana/latest/enterprise/license-expiration/" target="_blank">
            Some features
          </a>{' '}
          have been disabled{' '}
        </>
      }
      severity="error"
    />
  );
};

interface ExpiresSoonProps {
  days: number;
  onCloseWarning?(): void;
}

export const ExpiresSoon: FC<ExpiresSoonProps> = ({ days, onCloseWarning }) => {
  return (
    <Warning
      onButtonClick={onCloseWarning}
      title="Your Grafana Enterprise license will expire soon"
      subTitle={
        <>
          {days} days remaining, after which{' '}
          <a href="https://grafana.com/docs/grafana/latest/enterprise/license-expiration/" target="_blank">
            Enterprise features will be disabled.
          </a>
        </>
      }
      severity="info"
      buttonText="Dismiss"
    />
  );
};

interface MaxUsersReachedProps {
  activeUsers: number;
  maxUsers: number;
  type: string;
  onRefreshWarning(): void;
  slug?: string;
}

export const MaxUsersReached: FC<MaxUsersReachedProps> = ({ activeUsers, maxUsers, type, slug, onRefreshWarning }) => {
  return (
    <Warning
      onButtonClick={onRefreshWarning}
      title={'You have reached the number of maximum active ' + type}
      subTitle={
        <>
          Your license enables {maxUsers} active {type} on this Grafana Enterprise instance but you have {activeUsers}.
          Please,{' '}
          {slug ? (
            <a href={'https://grafana.com/orgs/' + slug + '/licenses'} target="_blank">
              upgrade your license.
            </a>
          ) : (
            'upgrade your license.'
          )}
        </>
      }
      severity="error"
      buttonText="Refresh"
    />
  );
};
