import { jsxs, jsx } from 'react/jsx-runtime';
import classNames from 'classnames';
import { PureComponent } from 'react';
import '../../../../utils/dom.mjs';
import '../../../../utils/colors.mjs';
import { validate, EventsWithValidation, hasValidationEvent } from '../../../../utils/validate.mjs';
import 'slate';
import 'lodash';
import '../../../../utils/logger.mjs';

class Input extends PureComponent {
  constructor() {
    super(...arguments);
    this.state = {
      error: null
    };
    this.validatorAsync = (validationRules) => {
      return (evt) => {
        const errors = validate(evt.target.value, validationRules);
        this.setState((prevState) => {
          return { ...prevState, error: errors ? errors[0] : null };
        });
      };
    };
    this.populateEventPropsWithStatus = (restProps, validationEvents) => {
      const inputElementProps = { ...restProps };
      if (!validationEvents) {
        return inputElementProps;
      }
      Object.keys(EventsWithValidation).forEach((eventName) => {
        if (hasValidationEvent(eventName, validationEvents) || restProps[eventName]) {
          inputElementProps[eventName] = async (evt) => {
            evt.persist();
            if (hasValidationEvent(eventName, validationEvents)) {
              await this.validatorAsync(validationEvents[eventName]).apply(this, [evt]);
            }
            if (restProps[eventName]) {
              restProps[eventName].apply(null, [evt, this.status]);
            }
          };
        }
      });
      return inputElementProps;
    };
  }
  get status() {
    return this.state.error ? "invalid" /* Invalid */ : "valid" /* Valid */;
  }
  get isInvalid() {
    return this.status === "invalid" /* Invalid */;
  }
  render() {
    const { validationEvents, className, hideErrorMessage, inputRef, ...restProps } = this.props;
    const { error } = this.state;
    const inputClassName = classNames("gf-form-input", { invalid: this.isInvalid }, className);
    const inputElementProps = this.populateEventPropsWithStatus(restProps, validationEvents);
    return /* @__PURE__ */ jsxs("div", { style: { flexGrow: 1 }, children: [
      /* @__PURE__ */ jsx("input", { ...inputElementProps, ref: inputRef, className: inputClassName }),
      error && !hideErrorMessage && /* @__PURE__ */ jsx("span", { children: error })
    ] });
  }
}
Input.defaultProps = {
  className: ""
};

export { Input };
//# sourceMappingURL=Input.mjs.map
