import { map, uniq, chain } from 'lodash';
import { getPrometheusTime } from './language_utils.mjs';
import { PrometheusLabelNamesRegex, PrometheusLabelNamesRegexWithMatch, PrometheusLabelValuesRegex, PrometheusMetricNamesRegex, PrometheusQueryResultRegex } from './migrations/variableMigration.mjs';
import { isValidLegacyName, escapeForUtf8Support } from './utf8_support.mjs';

class PrometheusMetricFindQuery {
  constructor(datasource, query) {
    this.datasource = datasource;
    this.query = query;
    this.datasource = datasource;
    this.query = query;
  }
  process(timeRange) {
    const labelNamesRegex = PrometheusLabelNamesRegex;
    const labelNamesRegexWithMatch = PrometheusLabelNamesRegexWithMatch;
    const labelValuesRegex = PrometheusLabelValuesRegex;
    const metricNamesRegex = PrometheusMetricNamesRegex;
    const queryResultRegex = PrometheusQueryResultRegex;
    const labelNamesQuery = this.query.match(labelNamesRegex);
    const labelNamesMatchQuery = this.query.match(labelNamesRegexWithMatch);
    if (labelNamesMatchQuery) {
      const selector = `{__name__=~".*${labelNamesMatchQuery[1]}.*"}`;
      return this.datasource.languageProvider.getSeriesLabels(timeRange, selector, []).then(
        (results) => results.map((result) => ({
          text: result
        }))
      );
    }
    if (labelNamesQuery) {
      return this.datasource.getTagKeys({ filters: [], timeRange });
    }
    const labelValuesQuery = this.query.match(labelValuesRegex);
    if (labelValuesQuery) {
      const filter = labelValuesQuery[1];
      const label = labelValuesQuery[2];
      if (isFilterDefined(filter)) {
        return this.labelValuesQuery(label, timeRange, filter);
      } else {
        return this.labelValuesQuery(label, timeRange);
      }
    }
    const metricNamesQuery = this.query.match(metricNamesRegex);
    if (metricNamesQuery) {
      return this.metricNameQuery(metricNamesQuery[1], timeRange);
    }
    const queryResultQuery = this.query.match(queryResultRegex);
    if (queryResultQuery) {
      return this.queryResultQuery(queryResultQuery[1], timeRange);
    }
    const expressions = ["label_values()", "metrics()", "query_result()"];
    if (!expressions.includes(this.query)) {
      return this.metricNameAndLabelsQuery(this.query, timeRange);
    }
    return Promise.resolve([]);
  }
  labelValuesQuery(label, range, metric) {
    const start = getPrometheusTime(range.from, false);
    const end = getPrometheusTime(range.to, true);
    const params = { ...metric && { "match[]": metric }, start: start.toString(), end: end.toString() };
    let escapedLabel = label;
    if (!isValidLegacyName(label)) {
      escapedLabel = escapeForUtf8Support(label);
    }
    if (!metric || this.datasource.hasLabelsMatchAPISupport()) {
      const url = `/api/v1/label/${escapedLabel}/values`;
      return this.datasource.metadataRequest(url, params).then((result) => {
        return map(result.data.data, (value) => {
          return { text: value };
        });
      });
    } else {
      const url = `/api/v1/series`;
      return this.datasource.metadataRequest(url, params).then((result) => {
        const _labels = map(result.data.data, (metric2) => {
          return metric2[label] || "";
        }).filter((label2) => {
          return label2 !== "";
        });
        return uniq(_labels).map((metric2) => {
          return {
            text: metric2,
            expandable: true
          };
        });
      });
    }
  }
  metricNameQuery(metricFilterPattern, range) {
    const start = getPrometheusTime(range.from, false);
    const end = getPrometheusTime(range.to, true);
    const params = {
      start: start.toString(),
      end: end.toString()
    };
    const url = `/api/v1/label/__name__/values`;
    return this.datasource.metadataRequest(url, params).then((result) => {
      return chain(result.data.data).filter((metricName) => {
        const r = new RegExp(metricFilterPattern);
        return r.test(metricName);
      }).map((matchedMetricName) => {
        return {
          text: matchedMetricName,
          expandable: true
        };
      }).value();
    });
  }
  queryResultQuery(query, range) {
    const url = "/api/v1/query";
    const params = {
      query,
      time: getPrometheusTime(range.to, true).toString()
    };
    return this.datasource.metadataRequest(url, params).then((result) => {
      switch (result.data.data.resultType) {
        case "scalar":
        // [ <unix_time>, "<scalar_value>" ]
        case "string":
          return [
            {
              text: result.data.data.result[1] || "",
              expandable: false
            }
          ];
        case "vector":
          return map(result.data.data.result, (metricData) => {
            let text = metricData.metric.__name__ || "";
            delete metricData.metric.__name__;
            text += "{" + map(metricData.metric, (v, k) => {
              return k + '="' + v + '"';
            }).join(",") + "}";
            text += " " + metricData.value[1] + " " + metricData.value[0] * 1e3;
            return {
              text,
              expandable: true
            };
          });
        default:
          throw Error(`Unknown/Unhandled result type: [${result.data.data.resultType}]`);
      }
    });
  }
  metricNameAndLabelsQuery(query, range) {
    const start = getPrometheusTime(range.from, false);
    const end = getPrometheusTime(range.to, true);
    const params = {
      "match[]": query,
      start: start.toString(),
      end: end.toString()
    };
    const url = `/api/v1/series`;
    const self = this;
    return this.datasource.metadataRequest(url, params).then((result) => {
      return map(result.data.data, (metric) => {
        return {
          text: self.datasource.getOriginalMetricName(metric),
          expandable: true
        };
      });
    });
  }
}
function isFilterDefined(filter) {
  return filter && filter.split(" ").join("") !== "{}";
}

export { PrometheusMetricFindQuery };
//# sourceMappingURL=metric_find_query.mjs.map
