import { cloneDeep, pickBy, identity, isNumber, omit } from 'lodash';
import { VizOrientation, fieldReducers, ReducerID, ThresholdsMode, convertOldAngularValueMappings, FieldColorModeId, validateFieldConfig, sortThresholds } from '@grafana/data';
import { LegendDisplayMode } from '@grafana/schema';

const optionsToKeep = ["reduceOptions", "orientation"];
function sharedSingleStatPanelChangedHandler(panel, prevPluginId, prevOptions) {
  let options = panel.options;
  panel.fieldConfig = panel.fieldConfig || {
    defaults: {},
    overrides: []
  };
  if (prevPluginId === "singlestat" && prevOptions.angular) {
    return migrateFromAngularSinglestat(panel, prevOptions);
  } else if (prevPluginId === "graph") {
    return migrateFromGraphPanel(panel, prevOptions);
  }
  for (const k of optionsToKeep) {
    if (prevOptions.hasOwnProperty(k)) {
      options[k] = cloneDeep(prevOptions[k]);
    }
  }
  return options;
}
function migrateFromGraphPanel(panel, prevOptions) {
  var _a;
  const graphOptions = prevOptions.angular;
  const options = {
    orientation: VizOrientation.Auto,
    reduceOptions: {
      values: false,
      calcs: []
    },
    legend: {
      displayMode: LegendDisplayMode.List,
      showLegend: true,
      placement: "bottom",
      calcs: []
    }
  };
  if (((_a = graphOptions.xaxis) == null ? void 0 : _a.mode) === "series") {
    panel.fieldConfig = {
      ...panel.fieldConfig,
      defaults: {
        ...panel.fieldConfig.defaults,
        color: { mode: "palette-classic" }
      }
    };
    if (graphOptions.xaxis.values) {
      options.reduceOptions.calcs = getReducerForMigration(graphOptions.xaxis.values);
    }
    const legendConfig = graphOptions.legend;
    if (legendConfig) {
      if (legendConfig.show) {
        options.legend.displayMode = legendConfig.alignAsTable ? LegendDisplayMode.Table : LegendDisplayMode.List;
      } else {
        options.legend.showLegend = false;
      }
      if (legendConfig.rightSide) {
        options.legend.placement = "right";
      }
      if (legendConfig.values) {
        const enabledLegendValues = pickBy(legendConfig, identity);
        options.legend.calcs = getReducersFromLegend(enabledLegendValues);
      }
      if (legendConfig.sideWidth) {
        options.legend.width = legendConfig.sideWidth;
      }
    }
  }
  return options;
}
function migrateFromAngularSinglestat(panel, prevOptions) {
  const prevPanel = prevOptions.angular;
  const reducer = fieldReducers.getIfExists(prevPanel.valueName);
  const options = {
    reduceOptions: {
      calcs: [reducer ? reducer.id : ReducerID.mean]
    },
    orientation: VizOrientation.Horizontal
  };
  const defaults = {};
  if (prevPanel.format) {
    defaults.unit = prevPanel.format;
  }
  if (prevPanel.tableColumn) {
    options.reduceOptions.fields = `/^${prevPanel.tableColumn}$/`;
  }
  if (prevPanel.nullPointMode) {
    defaults.nullValueMode = prevPanel.nullPointMode;
  }
  if (prevPanel.nullText) {
    defaults.noValue = prevPanel.nullText;
  }
  if (prevPanel.decimals || prevPanel.decimals === 0) {
    defaults.decimals = prevPanel.decimals;
  }
  if (prevPanel.thresholds && prevPanel.colors) {
    const levels = prevPanel.thresholds.split(",").map((strVale) => {
      return Number(strVale.trim());
    });
    const thresholds = [];
    for (const color of prevPanel.colors) {
      const idx = thresholds.length - 1;
      if (idx >= 0) {
        thresholds.push({ value: levels[idx], color });
      } else {
        thresholds.push({ value: -Infinity, color });
      }
    }
    defaults.thresholds = {
      mode: ThresholdsMode.Absolute,
      steps: thresholds
    };
  }
  const mappings = convertOldAngularValueMappings(prevPanel, defaults.thresholds);
  if (mappings && mappings.length) {
    defaults.mappings = mappings;
  }
  if (prevPanel.gauge && prevPanel.gauge.show) {
    defaults.min = prevPanel.gauge.minValue;
    defaults.max = prevPanel.gauge.maxValue;
  }
  panel.fieldConfig.defaults = defaults;
  return options;
}
function sharedSingleStatMigrationHandler(panel) {
  var _a, _b;
  if (!panel.options) {
    return {
      reduceOptions: {
        calcs: [ReducerID.mean]
      },
      orientation: VizOrientation.Horizontal
    };
  }
  const previousVersion = parseFloat(panel.pluginVersion || "6.1");
  let options = panel.options;
  if (previousVersion < 6.2) {
    options = migrateFromValueOptions(options);
  }
  if (previousVersion < 6.3) {
    options = moveThresholdsAndMappingsToField(options);
  }
  const { fieldOptions } = options;
  if (previousVersion < 6.6 && fieldOptions) {
    if (fieldOptions && fieldOptions.override) {
      const { override, ...rest } = options.fieldOptions;
      options = {
        ...options,
        fieldOptions: {
          ...rest,
          overrides: []
        }
      };
    }
    let thresholds = (_a = fieldOptions == null ? void 0 : fieldOptions.defaults) == null ? void 0 : _a.thresholds;
    if (thresholds) {
      delete fieldOptions.defaults.thresholds;
    } else {
      thresholds = fieldOptions == null ? void 0 : fieldOptions.thresholds;
      delete fieldOptions.thresholds;
    }
    if (thresholds) {
      fieldOptions.defaults.thresholds = {
        mode: ThresholdsMode.Absolute,
        steps: thresholds
      };
    }
    const { defaults } = fieldOptions;
    if (defaults.color && typeof defaults.color === "string") {
      defaults.color = {
        mode: FieldColorModeId.Fixed,
        fixedColor: defaults.color
      };
    }
    validateFieldConfig(defaults);
  }
  if (previousVersion < 7) {
    panel.fieldConfig = panel.fieldConfig || { defaults: {}, overrides: [] };
    panel.fieldConfig = {
      defaults: fieldOptions && fieldOptions.defaults ? { ...panel.fieldConfig.defaults, ...fieldOptions.defaults } : panel.fieldConfig.defaults,
      overrides: fieldOptions && fieldOptions.overrides ? [...panel.fieldConfig.overrides, ...fieldOptions.overrides] : panel.fieldConfig.overrides
    };
    if (fieldOptions) {
      options.reduceOptions = {
        values: fieldOptions.values,
        limit: fieldOptions.limit,
        calcs: fieldOptions.calcs
      };
    }
    delete options.fieldOptions;
  }
  if (previousVersion < 7.1) {
    const oldTitle = panel.fieldConfig.defaults.title;
    if (oldTitle !== void 0 && oldTitle !== null) {
      panel.fieldConfig.defaults.displayName = oldTitle;
      delete panel.fieldConfig.defaults.title;
    }
  }
  if (previousVersion < 8) {
    const config = (_b = panel.fieldConfig) == null ? void 0 : _b.defaults;
    let unit = config == null ? void 0 : config.unit;
    if (unit === "percent") {
      if (!isNumber(config.min)) {
        config.min = 0;
      }
      if (!isNumber(config.max)) {
        config.max = 100;
      }
    } else if (unit === "percentunit") {
      if (!isNumber(config.min)) {
        config.min = 0;
      }
      if (!isNumber(config.max)) {
        config.max = 1;
      }
    }
  }
  return options;
}
function moveThresholdsAndMappingsToField(old) {
  const { fieldOptions } = old;
  if (!fieldOptions) {
    return old;
  }
  const { mappings, ...rest } = old.fieldOptions;
  let thresholds = void 0;
  if (old.thresholds) {
    thresholds = {
      mode: ThresholdsMode.Absolute,
      steps: migrateOldThresholds(old.thresholds)
    };
  }
  return {
    ...old,
    fieldOptions: {
      ...rest,
      defaults: {
        ...fieldOptions.defaults,
        mappings,
        thresholds
      }
    }
  };
}
function migrateFromValueOptions(old) {
  const { valueOptions } = old;
  if (!valueOptions) {
    return old;
  }
  const fieldOptions = {};
  const fieldDefaults = {};
  fieldOptions.mappings = old.valueMappings;
  fieldOptions.thresholds = old.thresholds;
  fieldOptions.defaults = fieldDefaults;
  fieldDefaults.unit = valueOptions.unit;
  fieldDefaults.decimals = valueOptions.decimals;
  if (valueOptions.stat) {
    const reducer = fieldReducers.get(valueOptions.stat);
    if (reducer) {
      fieldOptions.calcs = [reducer.id];
    }
  }
  fieldDefaults.min = old.minValue;
  fieldDefaults.max = old.maxValue;
  const newOptions = {
    ...old,
    fieldOptions
  };
  return omit(newOptions, "valueMappings", "thresholds", "valueOptions", "minValue", "maxValue");
}
function migrateOldThresholds(thresholds) {
  if (!thresholds || !thresholds.length) {
    return void 0;
  }
  const copy = thresholds.map((t) => {
    return {
      // Drops 'index'
      value: t.value === null ? -Infinity : t.value,
      color: t.color
    };
  });
  sortThresholds(copy);
  copy[0].value = -Infinity;
  return copy;
}
function convertOldAngularValueMapping(panel) {
  return convertOldAngularValueMappings(panel);
}
function getReducersFromLegend(obj) {
  const ids = [];
  for (const key in obj) {
    const reducer = fieldReducers.getIfExists(key);
    if (reducer) {
      ids.push(reducer.id);
    }
  }
  return ids;
}
function getReducerForMigration(reducers) {
  const transformReducers = [];
  reducers == null ? void 0 : reducers.forEach((reducer) => {
    if (!Object.values(ReducerID).includes(reducer)) {
      if (reducer === "current") {
        transformReducers.push(ReducerID.lastNotNull);
      } else if (reducer === "total") {
        transformReducers.push(ReducerID.sum);
      } else if (reducer === "avg") {
        transformReducers.push(ReducerID.mean);
      }
    } else {
      transformReducers.push(reducer);
    }
  });
  return reducers ? transformReducers : [ReducerID.sum];
}

export { convertOldAngularValueMapping, migrateFromValueOptions, migrateOldThresholds, moveThresholdsAndMappingsToField, sharedSingleStatMigrationHandler, sharedSingleStatPanelChangedHandler };
//# sourceMappingURL=SingleStatBaseOptions.mjs.map
