import { set } from 'lodash';
import { FieldConfigOptionsRegistry } from '../field/FieldConfigOptionsRegistry.mjs';
import { GrafanaPlugin } from '../types/plugin.mjs';
import { PanelOptionsEditorBuilder } from '../utils/OptionsUIBuilders.mjs';
import { deprecationWarning } from '../utils/deprecationWarning.mjs';
import { createFieldConfigRegistry } from './registryFactories.mjs';

class PanelPlugin extends GrafanaPlugin {
  constructor(panel) {
    super();
    this._fieldConfigDefaults = {
      defaults: {},
      overrides: []
    };
    this._initConfigRegistry = () => {
      return new FieldConfigOptionsRegistry();
    };
    this.dataSupport = {
      annotations: false,
      alertStates: false
    };
    this.panel = panel;
  }
  get defaults() {
    let result = this._defaults || {};
    if (!this._defaults && this.optionsSupplier) {
      const builder = new PanelOptionsEditorBuilder();
      this.optionsSupplier(builder, { data: [] });
      for (const item of builder.getItems()) {
        if (item.defaultValue != null) {
          set(result, item.path, item.defaultValue);
        }
      }
    }
    return result;
  }
  get fieldConfigDefaults() {
    const configDefaults = this._fieldConfigDefaults.defaults;
    configDefaults.custom = {};
    for (const option of this.fieldConfigRegistry.list()) {
      if (option.defaultValue === undefined) {
        continue;
      }
      set(configDefaults, option.id, option.defaultValue);
    }
    return {
      defaults: {
        ...configDefaults
      },
      overrides: this._fieldConfigDefaults.overrides
    };
  }
  /**
   * @deprecated setDefaults is deprecated in favor of setPanelOptions
   */
  setDefaults(defaults) {
    deprecationWarning("PanelPlugin", "setDefaults", "setPanelOptions");
    this._defaults = defaults;
    return this;
  }
  get fieldConfigRegistry() {
    if (!this._fieldConfigRegistry) {
      this._fieldConfigRegistry = this._initConfigRegistry();
    }
    return this._fieldConfigRegistry;
  }
  /**
   * @deprecated setEditor is deprecated in favor of setPanelOptions
   */
  setEditor(editor) {
    deprecationWarning("PanelPlugin", "setEditor", "setPanelOptions");
    this.editor = editor;
    return this;
  }
  setNoPadding() {
    this.noPadding = true;
    return this;
  }
  /**
   * This function is called before the panel first loads if
   * the current version is different than the version that was saved.
   *
   * This is a good place to support any changes to the options model
   */
  setMigrationHandler(handler) {
    this.onPanelMigration = handler;
    return this;
  }
  /**
   * This function is called when the visualization was changed. This
   * passes in the panel model for previous visualisation options inspection
   * and panel model updates.
   *
   * This is useful for supporting PanelModel API updates when changing
   * between Angular and React panels.
   */
  setPanelChangeHandler(handler) {
    this.onPanelTypeChanged = handler;
    return this;
  }
  /**
   * Enables panel options editor creation
   *
   * @example
   * ```typescript
   *
   * import { ShapePanel } from './ShapePanel';
   *
   * interface ShapePanelOptions {}
   *
   * export const plugin = new PanelPlugin<ShapePanelOptions>(ShapePanel)
   *   .setPanelOptions(builder => {
   *     builder
   *       .addSelect({
   *         id: 'shape',
   *         name: 'Shape',
   *         description: 'Select shape to render'
   *         settings: {
   *           options: [
   *             {value: 'circle', label: 'Circle' },
   *             {value: 'square', label: 'Square },
   *             {value: 'triangle', label: 'Triangle }
   *            ]
   *         },
   *       })
   *   })
   * ```
   *
   * @public
   **/
  setPanelOptions(builder) {
    this.optionsSupplier = builder;
    return this;
  }
  /**
   * This is used while building the panel options editor.
   *
   * @internal
   */
  getPanelOptionsSupplier() {
    var _a;
    return (_a = this.optionsSupplier) != null ? _a : () => {
    };
  }
  /**
   * Tells Grafana if the plugin should subscribe to annotation and alertState results.
   *
   * @example
   * ```typescript
   *
   * import { ShapePanel } from './ShapePanel';
   *
   * interface ShapePanelOptions {}
   *
   * export const plugin = new PanelPlugin<ShapePanelOptions>(ShapePanel)
   *     .useFieldConfig({})
   *     ...
   *     ...
   *     .setDataSupport({
   *       annotations: true,
   *       alertStates: true,
   *     });
   * ```
   *
   * @public
   **/
  setDataSupport(support) {
    this.dataSupport = { ...this.dataSupport, ...support };
    return this;
  }
  /**
   * Allows specifying which standard field config options panel should use and defining default values
   *
   * @example
   * ```typescript
   *
   * import { ShapePanel } from './ShapePanel';
   *
   * interface ShapePanelOptions {}
   *
   * // when plugin should use all standard options
   * export const plugin = new PanelPlugin<ShapePanelOptions>(ShapePanel)
   *  .useFieldConfig();
   *
   * // when plugin should only display specific standard options
   * // note, that options will be displayed in the order they are provided
   * export const plugin = new PanelPlugin<ShapePanelOptions>(ShapePanel)
   *  .useFieldConfig({
   *    standardOptions: [FieldConfigProperty.Min, FieldConfigProperty.Max]
   *   });
   *
   * // when standard option's default value needs to be provided
   * export const plugin = new PanelPlugin<ShapePanelOptions>(ShapePanel)
   *  .useFieldConfig({
   *    standardOptions: [FieldConfigProperty.Min, FieldConfigProperty.Max],
   *    standardOptionsDefaults: {
   *      [FieldConfigProperty.Min]: 20,
   *      [FieldConfigProperty.Max]: 100
   *    }
   *  });
   *
   * // when custom field config options needs to be provided
   * export const plugin = new PanelPlugin<ShapePanelOptions>(ShapePanel)
   *  .useFieldConfig({
   *    useCustomConfig: builder => {
   *      builder
   *       .addNumberInput({
   *         id: 'shapeBorderWidth',
   *         name: 'Border width',
   *         description: 'Border width of the shape',
   *         settings: {
   *           min: 1,
   *           max: 5,
   *         },
   *       })
   *       .addSelect({
   *         id: 'displayMode',
   *         name: 'Display mode',
   *         description: 'How the shape shout be rendered'
   *         settings: {
   *         options: [{value: 'fill', label: 'Fill' }, {value: 'transparent', label: 'Transparent }]
   *       },
   *     })
   *   },
   *  });
   *
   * ```
   *
   * @public
   */
  useFieldConfig(config = {}) {
    this._initConfigRegistry = () => createFieldConfigRegistry(config, this.meta.name);
    return this;
  }
  /**
   * Sets function that can return visualization examples and suggestions.
   * @alpha
   */
  setSuggestionsSupplier(supplier) {
    this.suggestionsSupplier = supplier;
    return this;
  }
  /**
   * Returns the suggestions supplier
   * @alpha
   */
  getSuggestionsSupplier() {
    return this.suggestionsSupplier;
  }
  hasPluginId(pluginId) {
    return this.meta.id === pluginId;
  }
}

export { PanelPlugin };
//# sourceMappingURL=PanelPlugin.mjs.map
