import { get } from 'lodash';
import { lastValueFrom } from 'rxjs';
import { usePluginContext } from '@grafana/data';
import { config } from '../config.mjs';
import { getBackendSrv } from '../services/backendSrv.mjs';
import '../services/LocationService.mjs';
import '../services/appEvents.mjs';
import '../services/SidecarService_EXPERIMENTAL.mjs';
import '../services/SidecarContext_EXPERIMENTAL.mjs';
import 'react';
import '../services/ScopesContext.mjs';

const baseURL = `/apis/userstorage.grafana.app/v0alpha1/namespaces/${config.namespace}/user-storage`;
async function apiRequest(requestOptions) {
  try {
    const { data: responseData, ...meta } = await lastValueFrom(
      getBackendSrv().fetch({
        ...requestOptions,
        url: baseURL + requestOptions.url,
        data: requestOptions.body
      })
    );
    return { data: responseData, meta };
  } catch (error) {
    return requestOptions.manageError ? requestOptions.manageError(error) : { error };
  }
}
class UserStorage {
  constructor(service) {
    this.service = service;
    this.userUID = config.bootData.user.uid === "" ? config.bootData.user.id.toString() : config.bootData.user.uid;
    this.resourceName = `${service}:${this.userUID}`;
    this.canUseUserStorage = config.featureToggles.userStorageAPI === true && config.bootData.user.isSignedIn;
  }
  async init() {
    if (this.storageSpec !== undefined) {
      return;
    }
    const userStorage = await apiRequest({
      url: `/${this.resourceName}`,
      method: "GET",
      showErrorAlert: false
    });
    if ("error" in userStorage) {
      if (get(userStorage, "error.status") !== 404) {
        console.error("Failed to get user storage", userStorage.error);
      }
      this.storageSpec = null;
    } else {
      this.storageSpec = userStorage.data.spec;
    }
  }
  async getItem(key) {
    if (!this.canUseUserStorage) {
      return localStorage.getItem(this.resourceName);
    }
    await this.init();
    if (!this.storageSpec) {
      return localStorage.getItem(this.resourceName);
    }
    return this.storageSpec.data[key];
  }
  async setItem(key, value) {
    if (!this.canUseUserStorage) {
      localStorage.setItem(key, value);
      return;
    }
    const newData = { data: { [key]: value } };
    await this.init();
    if (!this.storageSpec) {
      await apiRequest({
        url: `/`,
        method: "POST",
        body: {
          metadata: { name: this.resourceName, labels: { user: this.userUID, service: this.service } },
          spec: newData
        }
      });
      this.storageSpec = newData;
      return;
    }
    this.storageSpec.data[key] = value;
    await apiRequest({
      headers: { "Content-Type": "application/merge-patch+json" },
      url: `/${this.resourceName}`,
      method: "PATCH",
      body: { spec: newData }
    });
  }
}
function usePluginUserStorage() {
  const context = usePluginContext();
  if (!context) {
    throw new Error(`No PluginContext found. The usePluginUserStorage() hook can only be used from a plugin.`);
  }
  return new UserStorage(context == null ? undefined : context.meta.id);
}

export { usePluginUserStorage };
//# sourceMappingURL=userStorage.mjs.map
