import { jsxs, jsx, Fragment } from 'react/jsx-runtime';
import { cx } from '@emotion/css';
import { useVirtualizer } from '@tanstack/react-virtual';
import { useMultipleSelection, useCombobox } from 'downshift';
import { useState, useMemo, useCallback } from 'react';
import '@grafana/data';
import { useStyles2 } from '../../themes/ThemeContext.mjs';
import 'micro-memoize';
import '@emotion/react';
import 'tinycolor2';
import '../../utils/skeleton.mjs';
import { t } from '../../utils/i18n.mjs';
import { Checkbox } from '../Forms/Checkbox.mjs';
import { Icon } from '../Icon/Icon.mjs';
import { Box } from '../Layout/Box/Box.mjs';
import { Stack } from '../Layout/Stack/Stack.mjs';
import { Portal } from '../Portal/Portal.mjs';
import { ScrollContainer } from '../ScrollContainer/ScrollContainer.mjs';
import { Text } from '../Text/Text.mjs';
import { Tooltip } from '../Tooltip/Tooltip.mjs';
import { VIRTUAL_OVERSCAN_ITEMS } from './Combobox.mjs';
import { NotFoundError } from './MessageRows.mjs';
import { OptionListItem } from './OptionListItem.mjs';
import { SuffixIcon } from './SuffixIcon.mjs';
import { ValuePill } from './ValuePill.mjs';
import { itemToString } from './filter.mjs';
import { getComboboxStyles, MENU_OPTION_HEIGHT, MENU_OPTION_HEIGHT_DESCRIPTION } from './getComboboxStyles.mjs';
import { getMultiComboboxStyles } from './getMultiComboboxStyles.mjs';
import { ALL_OPTION_VALUE } from './types.mjs';
import { useComboboxFloat } from './useComboboxFloat.mjs';
import { useMeasureMulti, MAX_SHOWN_ITEMS } from './useMeasureMulti.mjs';
import { useMultiInputAutoSize } from './useMultiInputAutoSize.mjs';
import { useOptions } from './useOptions.mjs';

const MultiCombobox = (props) => {
  const {
    placeholder,
    onChange,
    value,
    width,
    enableAllOption,
    invalid,
    disabled,
    minWidth,
    maxWidth,
    isClearable,
    createCustomValue = false
  } = props;
  const styles = useStyles2(getComboboxStyles);
  const [inputValue, setInputValue] = useState("");
  const allOptionItem = useMemo(() => {
    return {
      label: inputValue === "" ? t("multicombobox.all.title", "All") : t("multicombobox.all.title-filtered", "All (filtered)"),
      // Type casting needed to make this work when T is a number
      value: ALL_OPTION_VALUE
    };
  }, [inputValue]);
  const { options: baseOptions, updateOptions, asyncLoading } = useOptions(props.options, createCustomValue);
  const options = useMemo(() => {
    const addAllOption = enableAllOption && baseOptions.length > 1;
    return addAllOption ? [allOptionItem, ...baseOptions] : baseOptions;
  }, [baseOptions, enableAllOption, allOptionItem]);
  const loading = props.loading || asyncLoading;
  const selectedItems = useMemo(() => {
    if (!value) {
      return [];
    }
    return getSelectedItemsFromValue(value, typeof props.options !== "function" ? props.options : baseOptions);
  }, [value, props.options, baseOptions]);
  const { measureRef, counterMeasureRef, suffixMeasureRef, shownItems } = useMeasureMulti(
    selectedItems,
    width,
    disabled
  );
  const isOptionSelected = useCallback(
    (item) => selectedItems.some((opt) => opt.value === item.value),
    [selectedItems]
  );
  const { getSelectedItemProps, getDropdownProps, setSelectedItems, addSelectedItem, removeSelectedItem, reset } = useMultipleSelection({
    selectedItems,
    // initially selected items,
    onStateChange: ({ type, selectedItems: newSelectedItems }) => {
      switch (type) {
        case useMultipleSelection.stateChangeTypes.SelectedItemKeyDownBackspace:
        case useMultipleSelection.stateChangeTypes.SelectedItemKeyDownDelete:
        case useMultipleSelection.stateChangeTypes.DropdownKeyDownBackspace:
        case useMultipleSelection.stateChangeTypes.FunctionRemoveSelectedItem:
        case useMultipleSelection.stateChangeTypes.FunctionAddSelectedItem:
        case useMultipleSelection.stateChangeTypes.FunctionSetSelectedItems:
        case useMultipleSelection.stateChangeTypes.FunctionReset:
          onChange(newSelectedItems != null ? newSelectedItems : []);
          break;
      }
    },
    stateReducer: (state, actionAndChanges) => {
      const { changes } = actionAndChanges;
      return {
        ...changes,
        /**
         * TODO: Fix Hack!
         * This prevents the menu from closing when the user unselects an item in the dropdown at the expense
         * of breaking keyboard navigation.
         *
         * Downshift isn't really designed to keep selected items in the dropdown menu, so when you unselect an item
         * in a multiselect, the stateReducer tries to move focus onto another item which causes the menu to be closed.
         * This only seems to happen when you deselect the last item in the selectedItems list.
         *
         * Check out:
         *  - FunctionRemoveSelectedItem in the useMultipleSelection reducer https://github.com/downshift-js/downshift/blob/master/src/hooks/useMultipleSelection/reducer.js#L75
         *  - The activeIndex useEffect in useMultipleSelection https://github.com/downshift-js/downshift/blob/master/src/hooks/useMultipleSelection/index.js#L68-L72
         *
         * Forcing the activeIndex to -999 both prevents the useEffect that changes the focus from triggering (value never changes)
         * and prevents the if statement in useMultipleSelection from focusing anything.
         */
        activeIndex: -999
      };
    }
  });
  const {
    getToggleButtonProps,
    //getLabelProps,
    isOpen,
    highlightedIndex,
    getMenuProps,
    getInputProps,
    getItemProps
  } = useCombobox({
    items: options,
    itemToString,
    inputValue,
    selectedItem: null,
    stateReducer: (state, actionAndChanges) => {
      const { type } = actionAndChanges;
      let { changes } = actionAndChanges;
      const menuBeingOpened = state.isOpen === false && changes.isOpen === true;
      if (menuBeingOpened && changes.inputValue === state.inputValue) {
        changes = {
          ...changes,
          inputValue: ""
        };
      }
      switch (type) {
        case useCombobox.stateChangeTypes.InputKeyDownEnter:
        case useCombobox.stateChangeTypes.ItemClick:
          return {
            ...changes,
            isOpen: true,
            highlightedIndex: state.highlightedIndex
          };
        case useCombobox.stateChangeTypes.InputBlur:
          setInputValue("");
        default:
          return changes;
      }
    },
    onIsOpenChange: ({ isOpen: isOpen2, inputValue: inputValue2 }) => {
      if (isOpen2 && inputValue2 === "") {
        updateOptions(inputValue2);
      }
    },
    onStateChange: ({ inputValue: newInputValue, type, selectedItem: newSelectedItem }) => {
      switch (type) {
        case useCombobox.stateChangeTypes.InputKeyDownEnter:
        case useCombobox.stateChangeTypes.ItemClick:
          if ((newSelectedItem == null ? undefined : newSelectedItem.value) === ALL_OPTION_VALUE) {
            const isAllFilteredSelected = selectedItems.length === options.length - 1;
            const realOptions = options.slice(1);
            let newSelectedItems = isAllFilteredSelected && inputValue === "" ? [] : realOptions;
            if (!isAllFilteredSelected && inputValue !== "") {
              newSelectedItems = [.../* @__PURE__ */ new Set([...selectedItems, ...realOptions])];
            }
            if (isAllFilteredSelected && inputValue !== "") {
              const filteredSet = new Set(realOptions.map((item) => item.value));
              newSelectedItems = selectedItems.filter((item) => !filteredSet.has(item.value));
            }
            setSelectedItems(newSelectedItems);
          } else if (newSelectedItem && isOptionSelected(newSelectedItem)) {
            removeSelectedItem(newSelectedItem);
          } else if (newSelectedItem) {
            addSelectedItem(newSelectedItem);
          }
          break;
        case useCombobox.stateChangeTypes.InputChange:
          setInputValue(newInputValue != null ? newInputValue : "");
          updateOptions(newInputValue != null ? newInputValue : "");
          break;
      }
    }
  });
  const { inputRef: containerRef, floatingRef, floatStyles, scrollRef } = useComboboxFloat(options, isOpen);
  const multiStyles = useStyles2(
    getMultiComboboxStyles,
    isOpen,
    invalid,
    disabled,
    width,
    minWidth,
    maxWidth,
    isClearable
  );
  const virtualizerOptions = {
    count: options.length,
    getScrollElement: () => scrollRef.current,
    estimateSize: (index) => {
      const firstGroupItem = isNewGroup(options[index], index > 0 ? options[index - 1] : undefined);
      const hasDescription = "description" in options[index];
      let itemHeight = MENU_OPTION_HEIGHT;
      if (hasDescription) {
        itemHeight = MENU_OPTION_HEIGHT_DESCRIPTION;
      }
      if (firstGroupItem) {
        itemHeight += MENU_OPTION_HEIGHT;
      }
      return itemHeight;
    },
    overscan: VIRTUAL_OVERSCAN_ITEMS
  };
  const rowVirtualizer = useVirtualizer(virtualizerOptions);
  const visibleItems = isOpen ? selectedItems.slice(0, MAX_SHOWN_ITEMS) : selectedItems.slice(0, shownItems);
  const { inputRef, inputWidth } = useMultiInputAutoSize(inputValue);
  return /* @__PURE__ */ jsxs("div", { className: multiStyles.container, ref: containerRef, children: [
    /* @__PURE__ */ jsx("div", { className: cx(multiStyles.wrapper, { [multiStyles.disabled]: disabled }), ref: measureRef, children: /* @__PURE__ */ jsxs("span", { className: multiStyles.pillWrapper, children: [
      visibleItems.map((item, index) => /* @__PURE__ */ jsx(
        ValuePill,
        {
          disabled,
          onRemove: () => {
            removeSelectedItem(item);
          },
          ...getSelectedItemProps({ selectedItem: item, index }),
          children: itemToString(item)
        },
        `${item.value}${index}`
      )),
      selectedItems.length > visibleItems.length && /* @__PURE__ */ jsxs(Box, { display: "flex", direction: "row", marginLeft: 0.5, gap: 1, ref: counterMeasureRef, children: [
        /* @__PURE__ */ jsx(Text, { children: "..." }),
        /* @__PURE__ */ jsx(
          Tooltip,
          {
            interactive: true,
            content: /* @__PURE__ */ jsx(Fragment, { children: selectedItems.slice(visibleItems.length).map((item) => /* @__PURE__ */ jsx("div", { children: itemToString(item) }, item.value)) }),
            children: /* @__PURE__ */ jsx("div", { className: multiStyles.restNumber, children: selectedItems.length - shownItems })
          }
        )
      ] }),
      /* @__PURE__ */ jsx(
        "input",
        {
          className: multiStyles.input,
          ...getInputProps(
            getDropdownProps({
              disabled,
              preventKeyAction: isOpen,
              placeholder: visibleItems.length === 0 ? placeholder : "",
              ref: inputRef,
              style: { width: inputWidth }
            })
          )
        }
      ),
      /* @__PURE__ */ jsxs("div", { className: multiStyles.suffix, ref: suffixMeasureRef, ...getToggleButtonProps(), children: [
        isClearable && selectedItems.length > 0 && /* @__PURE__ */ jsx(
          Icon,
          {
            name: "times",
            className: styles.clear,
            title: t("multicombobox.clear.title", "Clear all"),
            tabIndex: 0,
            role: "button",
            onClick: (e) => {
              e.stopPropagation();
              reset();
            },
            onKeyDown: (e) => {
              if (e.key === "Enter" || e.key === " ") {
                reset();
              }
            }
          }
        ),
        /* @__PURE__ */ jsx(SuffixIcon, { isLoading: loading || false, isOpen })
      ] })
    ] }) }),
    /* @__PURE__ */ jsx(Portal, { children: /* @__PURE__ */ jsx(
      "div",
      {
        className: cx(styles.menu, !isOpen && styles.menuClosed),
        style: { ...floatStyles },
        ...getMenuProps({ ref: floatingRef }),
        children: isOpen && /* @__PURE__ */ jsxs(ScrollContainer, { showScrollIndicators: true, maxHeight: "inherit", ref: scrollRef, padding: 0.5, children: [
          /* @__PURE__ */ jsx("ul", { style: { height: rowVirtualizer.getTotalSize() }, className: styles.menuUlContainer, children: rowVirtualizer.getVirtualItems().map((virtualRow) => {
            var _a, _b, _c, _d;
            const startingNewGroup = isNewGroup(options[virtualRow.index], options[virtualRow.index - 1]);
            const index = virtualRow.index;
            const item = options[index];
            const itemProps = getItemProps({ item, index });
            const isSelected = isOptionSelected(item);
            const id = "multicombobox-option-" + item.value.toString();
            const isAll = item.value === ALL_OPTION_VALUE;
            const allItemsSelected = ((_a = options[0]) == null ? undefined : _a.value) === ALL_OPTION_VALUE && selectedItems.length === options.length - 1;
            return /* @__PURE__ */ jsx(
              "li",
              {
                "data-index": index,
                ...itemProps,
                className: styles.optionBasic,
                style: { height: virtualRow.size, transform: `translateY(${virtualRow.start}px)` },
                children: /* @__PURE__ */ jsxs(Stack, { direction: "column", justifyContent: "space-between", width: "100%", height: "100%", gap: 0, children: [
                  startingNewGroup && /* @__PURE__ */ jsx("div", { className: styles.optionGroup, children: /* @__PURE__ */ jsx(
                    OptionListItem,
                    {
                      label: (_b = item.group) != null ? _b : t("combobox.group.undefined", "No group"),
                      id,
                      isGroup: true
                    }
                  ) }),
                  /* @__PURE__ */ jsx(
                    "div",
                    {
                      className: cx(styles.option, {
                        [styles.optionFocused]: highlightedIndex === index
                      }),
                      children: /* @__PURE__ */ jsxs(Stack, { direction: "row", alignItems: "center", children: [
                        /* @__PURE__ */ jsx(
                          Checkbox,
                          {
                            value: allItemsSelected || isSelected,
                            indeterminate: isAll && selectedItems.length > 0 && !allItemsSelected,
                            "aria-labelledby": id,
                            onClick: (e) => {
                              e.stopPropagation();
                            }
                          },
                          id
                        ),
                        /* @__PURE__ */ jsx(
                          OptionListItem,
                          {
                            label: isAll ? ((_c = item.label) != null ? _c : item.value.toString()) + (isAll && inputValue !== "" ? ` (${options.length - 1})` : "") : (_d = item.label) != null ? _d : item.value.toString(),
                            description: item == null ? undefined : item.description,
                            id
                          }
                        )
                      ] })
                    }
                  )
                ] })
              },
              `${item.value}-${index}`
            );
          }) }),
          /* @__PURE__ */ jsx("div", { "aria-live": "polite", children: options.length === 0 && /* @__PURE__ */ jsx(NotFoundError, {}) })
        ] })
      }
    ) })
  ] });
};
function getSelectedItemsFromValue(value, options) {
  if (isComboboxOptions(value)) {
    return value;
  }
  const valueMap = new Map(value.map((val, index) => [val, index]));
  const resultingItems = [];
  for (const option of options) {
    const index = valueMap.get(option.value);
    if (index !== undefined) {
      resultingItems[index] = option;
      valueMap.delete(option.value);
    }
    if (valueMap.size === 0) {
      break;
    }
  }
  for (const [val, index] of valueMap) {
    resultingItems[index] = { value: val };
  }
  return resultingItems;
}
function isComboboxOptions(value) {
  return typeof value[0] === "object";
}
const isNewGroup = (option, prevOption) => {
  const currentGroup = option.group;
  if (!currentGroup) {
    return (prevOption == null ? undefined : prevOption.group) ? true : false;
  }
  if (!prevOption) {
    return true;
  }
  return prevOption.group !== currentGroup;
};

export { MultiCombobox };
//# sourceMappingURL=MultiCombobox.mjs.map
