import { map } from 'rxjs/operators';
import { guessFieldTypeForField } from '../../dataframe/processDataFrame.js';
import { getFieldDisplayName } from '../../field/fieldState.js';
import { FieldType } from '../../types/dataFrame.js';
import { ReducerID, fieldReducers, reduceField } from '../fieldReducer.js';
import { getFieldMatcher } from '../matchers.js';
import { alwaysFieldMatcher, notTimeFieldMatcher } from '../matchers/predicates.js';
import { DataTransformerID } from './ids.js';

const reduceTransformer = {
  id: DataTransformerID.reduce,
  name: "Reduce",
  description: "Reduce all rows or data points to a single value using a function like max, min, mean or last.",
  defaultOptions: {
    reducers: [ReducerID.max]
  },
  /**
   * Return a modified copy of the series. If the transform is not or should not
   * be applied, just return the input series
   */
  operator: (options) => (source) => source.pipe(
    map((data) => {
      var _a;
      if (!((_a = options == null ? void 0 : options.reducers) == null ? void 0 : _a.length)) {
        return data;
      }
      const matcher = options.fields ? getFieldMatcher(options.fields) : options.includeTimeField && options.mode === "reduceFields" /* ReduceFields */ ? alwaysFieldMatcher : notTimeFieldMatcher;
      if (options.mode === "reduceFields" /* ReduceFields */) {
        return reduceFields(data, matcher, options.reducers);
      }
      const res = reduceSeriesToRows(data, matcher, options.reducers, options.labelsToFields);
      return res ? [{ ...res, refId: `${DataTransformerID.reduce}-${data.map((frame) => frame.refId).join("-")}` }] : [];
    })
  )
};
function reduceSeriesToRows(data, matcher, reducerId, labelsToFields) {
  const calculators = fieldReducers.list(reducerId);
  const reducers = calculators.map((c) => c.id);
  const processed = [];
  const distinctLabels = labelsToFields ? getDistinctLabelKeys(data) : [];
  for (const series of data) {
    const source = series.fields.filter((f) => matcher(f, series, data));
    const size = source.length;
    const fields = [];
    const names = new Array(size);
    fields.push({
      name: "Field",
      type: FieldType.string,
      values: names,
      config: {}
    });
    const labels = {};
    if (labelsToFields) {
      for (const key of distinctLabels) {
        labels[key] = new Array(size);
        fields.push({
          name: key,
          type: FieldType.string,
          values: labels[key],
          config: {}
        });
      }
    }
    const calcs = {};
    for (const info of calculators) {
      calcs[info.id] = new Array(size);
      fields.push({
        name: info.name,
        type: FieldType.other,
        // UNKNOWN until after we call the functions
        values: calcs[info.id],
        config: {}
      });
    }
    for (let i = 0; i < source.length; i++) {
      const field = source[i];
      const results = reduceField({
        field,
        reducers
      });
      if (labelsToFields) {
        names[i] = field.name;
        if (field.labels) {
          for (const key in field.labels) {
            labels[key][i] = field.labels[key];
          }
        }
      } else {
        names[i] = getFieldDisplayName(field, series, data);
      }
      for (const info of calculators) {
        const v = results[info.id];
        if (v === null) {
          calcs[info.id][i] = NaN;
        } else {
          calcs[info.id][i] = v;
        }
      }
    }
    for (const f of fields) {
      if (f.type === FieldType.other) {
        const t = guessFieldTypeForField(f);
        if (t) {
          f.type = t;
        }
      }
    }
    processed.push({
      ...series,
      // Same properties, different fields
      fields,
      length: size
    });
  }
  return mergeResults(processed);
}
function getDistinctLabelKeys(frames) {
  const keys = /* @__PURE__ */ new Set();
  for (const frame of frames) {
    for (const field of frame.fields) {
      if (field.labels) {
        for (const k of Object.keys(field.labels)) {
          keys.add(k);
        }
      }
    }
  }
  return [...keys];
}
function mergeResults(data) {
  if (!(data == null ? void 0 : data.length)) {
    return void 0;
  }
  const baseFrame = data[0];
  for (let seriesIndex = 1; seriesIndex < data.length; seriesIndex++) {
    const series = data[seriesIndex];
    for (let baseIndex = 0; baseIndex < baseFrame.fields.length; baseIndex++) {
      const baseField = baseFrame.fields[baseIndex];
      for (let fieldIndex = 0; fieldIndex < series.fields.length; fieldIndex++) {
        const field = series.fields[fieldIndex];
        const isFirstField = baseIndex === 0 && fieldIndex === 0;
        const isSameField = baseField.type === field.type && baseField.name === field.name;
        if (isFirstField || isSameField) {
          const baseValues = baseField.values;
          const values = field.values;
          baseField.values = baseValues.concat(values);
        }
      }
    }
  }
  baseFrame.name = void 0;
  baseFrame.length = baseFrame.fields[0].values.length;
  return baseFrame;
}
function reduceFields(data, matcher, reducerId) {
  const calculators = fieldReducers.list(reducerId);
  const reducers = calculators.map((c) => c.id);
  const processed = [];
  for (const series of data) {
    const fields = [];
    for (const field of series.fields) {
      if (matcher(field, series, data)) {
        const results = reduceField({
          field,
          reducers
        });
        for (const reducer of reducers) {
          const value = results[reducer];
          const copy = {
            ...field,
            type: getFieldType(reducer, field),
            values: [value]
          };
          copy.state = void 0;
          if (reducers.length > 1) {
            if (!copy.labels) {
              copy.labels = {};
            }
            copy.labels["reducer"] = fieldReducers.get(reducer).name;
          }
          fields.push(copy);
        }
      }
    }
    if (fields.length) {
      processed.push({
        ...series,
        fields,
        length: 1
        // always one row
      });
    }
  }
  return processed;
}
function getFieldType(reducer, field) {
  switch (reducer) {
    case ReducerID.allValues:
    case ReducerID.uniqueValues:
      return FieldType.other;
    case ReducerID.first:
    case ReducerID.firstNotNull:
    case ReducerID.last:
    case ReducerID.lastNotNull:
      return field.type;
    default:
      return FieldType.number;
  }
}

export { reduceFields, reduceTransformer };
//# sourceMappingURL=reduce.js.map
