import { merge } from 'lodash';
import { emphasize, getContrastRatio, darken, lighten, alpha } from './colorManipulator.js';
import { palette } from './palette.js';

class DarkColors {
  constructor() {
    this.mode = "dark";
    // Used to get more white opacity colors
    this.whiteBase = "204, 204, 220";
    this.border = {
      weak: `rgba(${this.whiteBase}, 0.12)`,
      medium: `rgba(${this.whiteBase}, 0.20)`,
      strong: `rgba(${this.whiteBase}, 0.30)`
    };
    this.text = {
      primary: `rgb(${this.whiteBase})`,
      secondary: `rgba(${this.whiteBase}, 0.65)`,
      disabled: `rgba(${this.whiteBase}, 0.6)`,
      link: palette.blueDarkText,
      maxContrast: palette.white
    };
    this.primary = {
      main: palette.blueDarkMain,
      text: palette.blueDarkText,
      border: palette.blueDarkText
    };
    this.secondary = {
      main: `rgba(${this.whiteBase}, 0.10)`,
      shade: `rgba(${this.whiteBase}, 0.14)`,
      transparent: `rgba(${this.whiteBase}, 0.08)`,
      text: this.text.primary,
      contrastText: `rgb(${this.whiteBase})`,
      border: `rgba(${this.whiteBase}, 0.08)`
    };
    this.info = this.primary;
    this.error = {
      main: palette.redDarkMain,
      text: palette.redDarkText
    };
    this.success = {
      main: palette.greenDarkMain,
      text: palette.greenDarkText
    };
    this.warning = {
      main: palette.orangeDarkMain,
      text: palette.orangeDarkText
    };
    this.background = {
      canvas: palette.gray05,
      primary: palette.gray10,
      secondary: palette.gray15
    };
    this.action = {
      hover: `rgba(${this.whiteBase}, 0.16)`,
      selected: `rgba(${this.whiteBase}, 0.12)`,
      selectedBorder: palette.orangeDarkMain,
      focus: `rgba(${this.whiteBase}, 0.16)`,
      hoverOpacity: 0.08,
      disabledText: this.text.disabled,
      disabledBackground: `rgba(${this.whiteBase}, 0.04)`,
      disabledOpacity: 0.38
    };
    this.gradients = {
      brandHorizontal: "linear-gradient(270deg, #F55F3E 0%, #FF8833 100%)",
      brandVertical: "linear-gradient(0.01deg, #F55F3E 0.01%, #FF8833 99.99%)"
    };
    this.contrastThreshold = 3;
    this.hoverFactor = 0.03;
    this.tonalOffset = 0.15;
  }
}
class LightColors {
  constructor() {
    this.mode = "light";
    this.blackBase = "36, 41, 46";
    this.primary = {
      main: palette.blueLightMain,
      border: palette.blueLightText,
      text: palette.blueLightText
    };
    this.text = {
      primary: `rgba(${this.blackBase}, 1)`,
      secondary: `rgba(${this.blackBase}, 0.75)`,
      disabled: `rgba(${this.blackBase}, 0.64)`,
      link: this.primary.text,
      maxContrast: palette.black
    };
    this.border = {
      weak: `rgba(${this.blackBase}, 0.12)`,
      medium: `rgba(${this.blackBase}, 0.30)`,
      strong: `rgba(${this.blackBase}, 0.40)`
    };
    this.secondary = {
      main: `rgba(${this.blackBase}, 0.08)`,
      shade: `rgba(${this.blackBase}, 0.15)`,
      transparent: `rgba(${this.blackBase}, 0.08)`,
      contrastText: `rgba(${this.blackBase},  1)`,
      text: this.text.primary,
      border: this.border.weak
    };
    this.info = {
      main: palette.blueLightMain,
      text: palette.blueLightText
    };
    this.error = {
      main: palette.redLightMain,
      text: palette.redLightText,
      border: palette.redLightText
    };
    this.success = {
      main: palette.greenLightMain,
      text: palette.greenLightText
    };
    this.warning = {
      main: palette.orangeLightMain,
      text: palette.orangeLightText
    };
    this.background = {
      canvas: palette.gray90,
      primary: palette.white,
      secondary: palette.gray100
    };
    this.action = {
      hover: `rgba(${this.blackBase}, 0.12)`,
      selected: `rgba(${this.blackBase}, 0.08)`,
      selectedBorder: palette.orangeLightMain,
      hoverOpacity: 0.08,
      focus: `rgba(${this.blackBase}, 0.12)`,
      disabledBackground: `rgba(${this.blackBase}, 0.04)`,
      disabledText: this.text.disabled,
      disabledOpacity: 0.38
    };
    this.gradients = {
      brandHorizontal: "linear-gradient(90deg, #FF8833 0%, #F53E4C 100%)",
      brandVertical: "linear-gradient(0.01deg, #F53E4C -31.2%, #FF8833 113.07%)"
    };
    this.contrastThreshold = 3;
    this.hoverFactor = 0.03;
    this.tonalOffset = 0.2;
  }
}
function createColors(colors) {
  var _a;
  const dark = new DarkColors();
  const light = new LightColors();
  const base = ((_a = colors.mode) != null ? _a : "dark") === "dark" ? dark : light;
  const {
    primary = base.primary,
    secondary = base.secondary,
    info = base.info,
    warning = base.warning,
    success = base.success,
    error = base.error,
    tonalOffset = base.tonalOffset,
    hoverFactor = base.hoverFactor,
    contrastThreshold = base.contrastThreshold,
    ...other
  } = colors;
  function getContrastText(background, threshold = contrastThreshold) {
    const contrastText = getContrastRatio(dark.text.maxContrast, background, base.background.primary) >= threshold ? dark.text.maxContrast : light.text.maxContrast;
    return contrastText;
  }
  const getRichColor = ({ color, name }) => {
    color = { ...color, name };
    if (!color.main) {
      throw new Error(`Missing main color for ${name}`);
    }
    if (!color.text) {
      color.text = color.main;
    }
    if (!color.border) {
      color.border = color.text;
    }
    if (!color.shade) {
      color.shade = base.mode === "light" ? darken(color.main, tonalOffset) : lighten(color.main, tonalOffset);
    }
    if (!color.transparent) {
      color.transparent = alpha(color.main, 0.15);
    }
    if (!color.contrastText) {
      color.contrastText = getContrastText(color.main);
    }
    if (!color.borderTransparent) {
      color.borderTransparent = alpha(color.border, 0.25);
    }
    return color;
  };
  return merge(
    {
      ...base,
      primary: getRichColor({ color: primary, name: "primary" }),
      secondary: getRichColor({ color: secondary, name: "secondary" }),
      info: getRichColor({ color: info, name: "info" }),
      error: getRichColor({ color: error, name: "error" }),
      success: getRichColor({ color: success, name: "success" }),
      warning: getRichColor({ color: warning, name: "warning" }),
      getContrastText,
      emphasize: (color, factor) => {
        return emphasize(color, factor != null ? factor : hoverFactor);
      }
    },
    other
  );
}

export { createColors };
//# sourceMappingURL=createColors.js.map
