import { size } from 'lodash';
import { buildVisualQueryFromString } from './querybuilder/parsing.js';

var __defProp = Object.defineProperty;
var __defProps = Object.defineProperties;
var __getOwnPropDescs = Object.getOwnPropertyDescriptors;
var __getOwnPropSymbols = Object.getOwnPropertySymbols;
var __hasOwnProp = Object.prototype.hasOwnProperty;
var __propIsEnum = Object.prototype.propertyIsEnumerable;
var __defNormalProp = (obj, key, value) => key in obj ? __defProp(obj, key, { enumerable: true, configurable: true, writable: true, value }) : obj[key] = value;
var __spreadValues = (a, b) => {
  for (var prop in b || (b = {}))
    if (__hasOwnProp.call(b, prop))
      __defNormalProp(a, prop, b[prop]);
  if (__getOwnPropSymbols)
    for (var prop of __getOwnPropSymbols(b)) {
      if (__propIsEnum.call(b, prop))
        __defNormalProp(a, prop, b[prop]);
    }
  return a;
};
var __spreadProps = (a, b) => __defProps(a, __getOwnPropDescs(b));
const SUM_HINT_THRESHOLD_COUNT = 20;
function getQueryHints(query, series, datasource) {
  var _a;
  const hints = [];
  const metricsMetadata = (_a = datasource == null ? void 0 : datasource.languageProvider) == null ? void 0 : _a.metricsMetadata;
  const oldHistogramMetric = query.trim().match(/^\w+_bucket$|^\w+_bucket{.*}$/);
  if (oldHistogramMetric) {
    const label = "Selected metric has buckets.";
    hints.push({
      type: "HISTOGRAM_QUANTILE",
      label,
      fix: {
        label: "Consider calculating aggregated quantile by adding histogram_quantile().",
        action: {
          type: "ADD_HISTOGRAM_QUANTILE",
          query
        }
      }
    });
  } else if (metricsMetadata && simpleQueryCheck(query)) {
    const queryTokens = getQueryTokens(query);
    const { nameMetric } = checkMetricType(queryTokens, "histogram", metricsMetadata, false);
    const nativeHistogramNameMetric = nameMetric;
    if (nativeHistogramNameMetric) {
      const label = "Selected metric is a native histogram.";
      hints.push(
        {
          type: "HISTOGRAM_AVG",
          label,
          fix: {
            label: "Consider calculating the arithmetic average of observed values by adding histogram_avg().",
            action: {
              type: "ADD_HISTOGRAM_AVG",
              query
            }
          }
        },
        {
          type: "HISTOGRAM_COUNT",
          label,
          fix: {
            label: "Consider calculating the count of observations by adding histogram_count().",
            action: {
              type: "ADD_HISTOGRAM_COUNT",
              query
            }
          }
        },
        {
          type: "HISTOGRAM_SUM",
          label,
          fix: {
            label: "Consider calculating the sum of observations by adding histogram_sum().",
            action: {
              type: "ADD_HISTOGRAM_SUM",
              query
            }
          }
        },
        {
          type: "HISTOGRAM_FRACTION",
          label,
          fix: {
            label: "Consider calculating the estimated fraction of observations between the provided lower and upper values by adding histogram_fraction().",
            action: {
              type: "ADD_HISTOGRAM_FRACTION",
              query
            }
          }
        },
        {
          type: "HISTOGRAM_STDDEV",
          label,
          fix: {
            label: "Consider calculating the estimated standard deviation of observations by adding histogram_stddev().",
            action: {
              type: "ADD_HISTOGRAM_STDDEV",
              query
            }
          }
        },
        {
          type: "HISTOGRAM_STDVAR",
          label,
          fix: {
            label: "Consider calculating the estimated standard variance of observations by adding histogram_stdvar().",
            action: {
              type: "ADD_HISTOGRAM_STDVAR",
              query
            }
          }
        }
      );
    }
  }
  if (query.indexOf("rate(") === -1 && query.indexOf("increase(") === -1) {
    const nameMatch = query.match(new RegExp("\\b((?<!:)\\w+_(total|sum|count)(?!:))\\b"));
    let counterNameMetric = nameMatch ? nameMatch[1] : "";
    let certain = false;
    if (metricsMetadata) {
      const queryTokens = getQueryTokens(query);
      const metricTypeChecked = checkMetricType(queryTokens, "counter", metricsMetadata, certain);
      counterNameMetric = metricTypeChecked.nameMetric;
      certain = metricTypeChecked.certain;
    }
    if (counterNameMetric) {
      const fixableQuery = simpleQueryCheck(query);
      const verb = certain ? "is" : "looks like";
      let label = `Selected metric ${verb} a counter.`;
      let fix;
      if (fixableQuery) {
        fix = {
          label: "Consider calculating rate of counter by adding rate().",
          action: {
            type: "ADD_RATE",
            query
          }
        };
      } else {
        label = `${label} Consider calculating rate of counter by adding rate().`;
      }
      hints.push({
        type: "APPLY_RATE",
        label,
        fix
      });
    }
  }
  if (datasource && datasource.ruleMappings) {
    const expandQueryHints = getExpandRulesHints(query, datasource.ruleMappings);
    hints.push(...expandQueryHints);
  }
  if (series && series.length >= SUM_HINT_THRESHOLD_COUNT) {
    const simpleMetric = query.trim().match(/^\w+$/);
    if (simpleMetric) {
      hints.push({
        type: "ADD_SUM",
        label: "Many time series results returned.",
        fix: {
          label: "Consider aggregating with sum().",
          action: {
            type: "ADD_SUM",
            query,
            preventSubmit: true
          }
        }
      });
    }
  }
  return hints;
}
function getInitHints(datasource) {
  const hints = [];
  if (datasource.lookupsDisabled) {
    hints.push({
      label: `Labels and metrics lookup was disabled in data source settings.`,
      type: "INFO"
    });
  }
  return hints;
}
function getExpandRulesHints(query, mapping) {
  const hints = [];
  const mappingForQuery = Object.keys(mapping).reduce((acc, ruleName) => {
    if (query.search(ruleName) === -1) {
      return acc;
    }
    if (mapping[ruleName].length > 1) {
      const { idx, expandedQuery, identifier, identifierValue } = getRecordingRuleIdentifierIdx(
        query,
        ruleName,
        mapping[ruleName]
      );
      if (idx === -1) {
        hints.push({
          type: "EXPAND_RULES_WARNING",
          label: "We found multiple recording rules that match in this query. To expand the recording rule, add an identifier label/value."
        });
        return acc;
      } else {
        return __spreadProps(__spreadValues({}, acc), {
          [ruleName]: {
            expandedQuery,
            identifier,
            identifierValue
          }
        });
      }
    } else {
      return __spreadProps(__spreadValues({}, acc), {
        [ruleName]: {
          expandedQuery: mapping[ruleName][0].query
        }
      });
    }
  }, {});
  if (size(mappingForQuery) > 0) {
    const label = "Query contains recording rules.";
    hints.push({
      type: "EXPAND_RULES",
      label,
      fix: {
        label: "Expand rules",
        action: {
          type: "EXPAND_RULES",
          query,
          options: mappingForQuery
        }
      }
    });
  }
  return hints;
}
function getRecordingRuleIdentifierIdx(queryStr, ruleName, mapping) {
  var _a, _b;
  const { query } = buildVisualQueryFromString(queryStr);
  const queryMetricLabels = getQueryLabelsForRuleName(ruleName, query);
  if (queryMetricLabels.length === 0) {
    return { idx: -1, identifier: "", identifierValue: "", expandedQuery: "" };
  }
  let uuidLabel = "";
  let uuidLabelValue = "";
  let uuidLabelIdx = -1;
  queryMetricLabels.forEach((qml) => {
    if (uuidLabelIdx === -1 && qml.label.search("uuid") !== -1) {
      uuidLabel = qml.label;
      uuidLabelValue = qml.value;
    }
  });
  mapping.forEach((mp, idx) => {
    if (mp.labels) {
      Object.entries(mp.labels).forEach(([key, value]) => {
        if (uuidLabelIdx === -1 && key === uuidLabel && value === uuidLabelValue) {
          uuidLabelIdx = idx;
        }
      });
    }
  });
  return {
    idx: uuidLabelIdx,
    identifier: uuidLabel,
    identifierValue: uuidLabelValue,
    expandedQuery: (_b = (_a = mapping[uuidLabelIdx]) == null ? void 0 : _a.query) != null ? _b : ""
  };
}
function getQueryLabelsForRuleName(metricName, query) {
  if (query.metric === metricName) {
    return query.labels;
  } else {
    if (query.binaryQueries) {
      for (let i = 0; i < query.binaryQueries.length; i++) {
        const labels = getQueryLabelsForRuleName(metricName, query.binaryQueries[i].query);
        if (labels && labels.length > 0) {
          return labels;
        }
      }
    }
    return [];
  }
}
function getQueryTokens(query) {
  return Array.from(query.matchAll(/\$?[a-zA-Z_:][a-zA-Z0-9_:]*/g)).map(([match]) => match).filter((token) => !token.startsWith("$")).flatMap((token) => token.split(":"));
}
function checkMetricType(queryTokens, metricType, metricsMetadata, certain) {
  var _a;
  const nameMetric = (_a = queryTokens.find((metricName) => {
    const metadata = metricsMetadata[metricName];
    if (metadata && metadata.type.toLowerCase() === metricType) {
      certain = true;
      return true;
    } else {
      return false;
    }
  })) != null ? _a : "";
  return { nameMetric, certain };
}
function simpleQueryCheck(query) {
  return query.trim().match(/^\w+$|^\w+{.*}$/);
}

export { SUM_HINT_THRESHOLD_COUNT, getExpandRulesHints, getInitHints, getQueryHints, getQueryLabelsForRuleName, getRecordingRuleIdentifierIdx };
//# sourceMappingURL=query_hints.js.map
