'use strict';

Object.defineProperty(exports, '__esModule', { value: true });

var css = require('@emotion/css');
var uFuzzy = require('@leeoniya/ufuzzy');
var React = require('react');
var reactUse = require('react-use');
var ui = require('@grafana/ui');
var data$1 = require('@grafana/data');
var color = require('tinycolor2');
var d3 = require('d3');
var lodash = require('lodash');
var useDebounce = require('react-use/lib/useDebounce');
var usePrevious = require('react-use/lib/usePrevious');
var AutoSizer = require('react-virtualized-auto-sizer');

function _interopDefaultLegacy (e) { return e && typeof e === 'object' && 'default' in e ? e : { 'default': e }; }

var uFuzzy__default = /*#__PURE__*/_interopDefaultLegacy(uFuzzy);
var React__default = /*#__PURE__*/_interopDefaultLegacy(React);
var color__default = /*#__PURE__*/_interopDefaultLegacy(color);
var useDebounce__default = /*#__PURE__*/_interopDefaultLegacy(useDebounce);
var usePrevious__default = /*#__PURE__*/_interopDefaultLegacy(usePrevious);
var AutoSizer__default = /*#__PURE__*/_interopDefaultLegacy(AutoSizer);

const PIXELS_PER_LEVEL = 22 * window.devicePixelRatio;
const MUTE_THRESHOLD = 10 * window.devicePixelRatio;
const HIDE_THRESHOLD = 0.5 * window.devicePixelRatio;
const LABEL_THRESHOLD = 20 * window.devicePixelRatio;
const BAR_BORDER_WIDTH = 0.5 * window.devicePixelRatio;
const BAR_TEXT_PADDING_LEFT = 4 * window.devicePixelRatio;
const GROUP_STRIP_WIDTH = 3 * window.devicePixelRatio;
const GROUP_STRIP_PADDING = 3 * window.devicePixelRatio;
const GROUP_STRIP_MARGIN_LEFT = 4 * window.devicePixelRatio;
const GROUP_TEXT_OFFSET = 2 * window.devicePixelRatio;
const MIN_WIDTH_TO_SHOW_BOTH_TOPTABLE_AND_FLAMEGRAPH = 800;
const TOP_TABLE_COLUMN_WIDTH = 120;

const FlameGraphContextMenu = ({
  data,
  itemData,
  onMenuItemClick,
  onItemFocus,
  onSandwich,
  collapseConfig,
  onExpandGroup,
  onCollapseGroup,
  onExpandAllGroups,
  onCollapseAllGroups,
  getExtraContextMenuButtons,
  collapsing,
  allGroupsExpanded,
  allGroupsCollapsed,
  selectedView,
  search
}) => {
  function renderItems() {
    const extraButtons = (getExtraContextMenuButtons == null ? void 0 : getExtraContextMenuButtons(itemData, data.data, {
      selectedView,
      isDiff: data.isDiffFlamegraph(),
      search,
      collapseConfig
    })) || [];
    return /* @__PURE__ */ React__default["default"].createElement(React__default["default"].Fragment, null, /* @__PURE__ */ React__default["default"].createElement(
      ui.MenuItem,
      {
        label: "Focus block",
        icon: "eye",
        onClick: () => {
          onItemFocus();
          onMenuItemClick();
        }
      }
    ), /* @__PURE__ */ React__default["default"].createElement(
      ui.MenuItem,
      {
        label: "Copy function name",
        icon: "copy",
        onClick: () => {
          navigator.clipboard.writeText(itemData.label).then(() => {
            onMenuItemClick();
          });
        }
      }
    ), /* @__PURE__ */ React__default["default"].createElement(
      ui.MenuItem,
      {
        label: "Sandwich view",
        icon: "gf-show-context",
        onClick: () => {
          onSandwich();
          onMenuItemClick();
        }
      }
    ), extraButtons.map(({ label, icon, onClick }) => {
      return /* @__PURE__ */ React__default["default"].createElement(ui.MenuItem, { label, icon, onClick: () => onClick(), key: label });
    }), collapsing && /* @__PURE__ */ React__default["default"].createElement(ui.MenuGroup, { label: "Grouping" }, collapseConfig ? collapseConfig.collapsed ? /* @__PURE__ */ React__default["default"].createElement(
      ui.MenuItem,
      {
        label: "Expand group",
        icon: "angle-double-down",
        onClick: () => {
          onExpandGroup();
          onMenuItemClick();
        }
      }
    ) : /* @__PURE__ */ React__default["default"].createElement(
      ui.MenuItem,
      {
        label: "Collapse group",
        icon: "angle-double-up",
        onClick: () => {
          onCollapseGroup();
          onMenuItemClick();
        }
      }
    ) : null, !allGroupsExpanded && /* @__PURE__ */ React__default["default"].createElement(
      ui.MenuItem,
      {
        label: "Expand all groups",
        icon: "angle-double-down",
        onClick: () => {
          onExpandAllGroups();
          onMenuItemClick();
        }
      }
    ), !allGroupsCollapsed && /* @__PURE__ */ React__default["default"].createElement(
      ui.MenuItem,
      {
        label: "Collapse all groups",
        icon: "angle-double-up",
        onClick: () => {
          onCollapseAllGroups();
          onMenuItemClick();
        }
      }
    )));
  }
  return /* @__PURE__ */ React__default["default"].createElement("div", { "data-testid": "contextMenu" }, /* @__PURE__ */ React__default["default"].createElement(
    ui.ContextMenu,
    {
      renderMenuItems: renderItems,
      x: itemData.posX + 10,
      y: itemData.posY,
      focusOnOpen: false
    }
  ));
};

const FlameGraphTooltip = ({ data, item, totalTicks, position, collapseConfig }) => {
  const styles = ui.useStyles2(getStyles$6);
  if (!(item && position)) {
    return null;
  }
  let content;
  if (data.isDiffFlamegraph()) {
    const tableData = getDiffTooltipData(data, item, totalTicks);
    content = /* @__PURE__ */ React__default["default"].createElement(
      ui.InteractiveTable,
      {
        className: styles.tooltipTable,
        columns: [
          { id: "label", header: "" },
          { id: "baseline", header: "Baseline" },
          { id: "comparison", header: "Comparison" },
          { id: "diff", header: "Diff" }
        ],
        data: tableData,
        getRowId: (originalRow) => originalRow.rowId
      }
    );
  } else {
    const tooltipData = getTooltipData(data, item, totalTicks);
    content = /* @__PURE__ */ React__default["default"].createElement("p", { className: styles.lastParagraph }, tooltipData.unitTitle, /* @__PURE__ */ React__default["default"].createElement("br", null), "Total: ", /* @__PURE__ */ React__default["default"].createElement("b", null, tooltipData.unitValue), " (", tooltipData.percentValue, "%)", /* @__PURE__ */ React__default["default"].createElement("br", null), "Self: ", /* @__PURE__ */ React__default["default"].createElement("b", null, tooltipData.unitSelf), " (", tooltipData.percentSelf, "%)", /* @__PURE__ */ React__default["default"].createElement("br", null), "Samples: ", /* @__PURE__ */ React__default["default"].createElement("b", null, tooltipData.samples));
  }
  return /* @__PURE__ */ React__default["default"].createElement(ui.Portal, null, /* @__PURE__ */ React__default["default"].createElement(ui.VizTooltipContainer, { className: styles.tooltipContainer, position, offset: { x: 15, y: 0 } }, /* @__PURE__ */ React__default["default"].createElement("div", { className: styles.tooltipContent }, /* @__PURE__ */ React__default["default"].createElement("p", { className: styles.tooltipName }, data.getLabel(item.itemIndexes[0]), collapseConfig && collapseConfig.collapsed ? /* @__PURE__ */ React__default["default"].createElement("span", null, /* @__PURE__ */ React__default["default"].createElement("br", null), "and ", collapseConfig.items.length, " similar items") : ""), content)));
};
const getTooltipData = (data, item, totalTicks) => {
  const displayValue = data.valueDisplayProcessor(item.value);
  const displaySelf = data.getSelfDisplay(item.itemIndexes);
  const percentValue = Math.round(1e4 * (displayValue.numeric / totalTicks)) / 100;
  const percentSelf = Math.round(1e4 * (displaySelf.numeric / totalTicks)) / 100;
  let unitValue = displayValue.text + displayValue.suffix;
  let unitSelf = displaySelf.text + displaySelf.suffix;
  const unitTitle = data.getUnitTitle();
  if (unitTitle === "Count") {
    if (!displayValue.suffix) {
      unitValue = displayValue.text;
    }
    if (!displaySelf.suffix) {
      unitSelf = displaySelf.text;
    }
  }
  return {
    percentValue,
    percentSelf,
    unitTitle,
    unitValue,
    unitSelf,
    samples: displayValue.numeric.toLocaleString()
  };
};
const getDiffTooltipData = (data, item, totalTicks) => {
  const levels = data.getLevels();
  const totalTicksRight = levels[0][0].valueRight;
  const totalTicksLeft = totalTicks - totalTicksRight;
  const valueLeft = item.value - item.valueRight;
  const percentageLeft = Math.round(1e4 * valueLeft / totalTicksLeft) / 100;
  const percentageRight = Math.round(1e4 * item.valueRight / totalTicksRight) / 100;
  const diff = (percentageRight - percentageLeft) / percentageLeft * 100;
  const displayValueLeft = getValueWithUnit(data, data.valueDisplayProcessor(valueLeft));
  const displayValueRight = getValueWithUnit(data, data.valueDisplayProcessor(item.valueRight));
  const shortValFormat = data$1.getValueFormat("short");
  return [
    {
      rowId: "1",
      label: "% of total",
      baseline: percentageLeft + "%",
      comparison: percentageRight + "%",
      diff: shortValFormat(diff).text + "%"
    },
    {
      rowId: "2",
      label: "Value",
      baseline: displayValueLeft,
      comparison: displayValueRight,
      diff: getValueWithUnit(data, data.valueDisplayProcessor(item.valueRight - valueLeft))
    },
    {
      rowId: "3",
      label: "Samples",
      baseline: shortValFormat(valueLeft).text,
      comparison: shortValFormat(item.valueRight).text,
      diff: shortValFormat(item.valueRight - valueLeft).text
    }
  ];
};
function getValueWithUnit(data, displayValue) {
  let unitValue = displayValue.text + displayValue.suffix;
  const unitTitle = data.getUnitTitle();
  if (unitTitle === "Count") {
    if (!displayValue.suffix) {
      unitValue = displayValue.text;
    }
  }
  return unitValue;
}
const getStyles$6 = (theme) => ({
  tooltipContainer: css.css({
    title: "tooltipContainer",
    overflow: "hidden"
  }),
  tooltipContent: css.css({
    title: "tooltipContent",
    fontSize: theme.typography.bodySmall.fontSize,
    width: "100%"
  }),
  tooltipName: css.css({
    title: "tooltipName",
    marginTop: 0,
    wordBreak: "break-all"
  }),
  lastParagraph: css.css({
    title: "lastParagraph",
    marginBottom: 0
  }),
  name: css.css({
    title: "name",
    marginBottom: "10px"
  }),
  tooltipTable: css.css({
    title: "tooltipTable",
    maxWidth: "400px"
  })
});

var SampleUnit = /* @__PURE__ */ ((SampleUnit2) => {
  SampleUnit2["Bytes"] = "bytes";
  SampleUnit2["Short"] = "short";
  SampleUnit2["Nanoseconds"] = "ns";
  return SampleUnit2;
})(SampleUnit || {});
var SelectedView = /* @__PURE__ */ ((SelectedView2) => {
  SelectedView2["TopTable"] = "topTable";
  SelectedView2["FlameGraph"] = "flameGraph";
  SelectedView2["Both"] = "both";
  return SelectedView2;
})(SelectedView || {});
var ColorScheme = /* @__PURE__ */ ((ColorScheme2) => {
  ColorScheme2["ValueBased"] = "valueBased";
  ColorScheme2["PackageBased"] = "packageBased";
  return ColorScheme2;
})(ColorScheme || {});
var ColorSchemeDiff = /* @__PURE__ */ ((ColorSchemeDiff2) => {
  ColorSchemeDiff2["Default"] = "default";
  ColorSchemeDiff2["DiffColorBlind"] = "diffColorBlind";
  return ColorSchemeDiff2;
})(ColorSchemeDiff || {});

function murmurhash3_32_gc(key, seed = 0) {
  let remainder;
  let bytes;
  let h1;
  let h1b;
  let c1;
  let c2;
  let k1;
  let i;
  remainder = key.length & 3;
  bytes = key.length - remainder;
  h1 = seed;
  c1 = 3432918353;
  c2 = 461845907;
  i = 0;
  while (i < bytes) {
    k1 = key.charCodeAt(i) & 255 | (key.charCodeAt(++i) & 255) << 8 | (key.charCodeAt(++i) & 255) << 16 | (key.charCodeAt(++i) & 255) << 24;
    ++i;
    k1 = (k1 & 65535) * c1 + (((k1 >>> 16) * c1 & 65535) << 16) & 4294967295;
    k1 = k1 << 15 | k1 >>> 17;
    k1 = (k1 & 65535) * c2 + (((k1 >>> 16) * c2 & 65535) << 16) & 4294967295;
    h1 ^= k1;
    h1 = h1 << 13 | h1 >>> 19;
    h1b = (h1 & 65535) * 5 + (((h1 >>> 16) * 5 & 65535) << 16) & 4294967295;
    h1 = (h1b & 65535) + 27492 + (((h1b >>> 16) + 58964 & 65535) << 16);
  }
  k1 = 0;
  switch (remainder) {
    case 3:
      k1 ^= (key.charCodeAt(i + 2) & 255) << 16;
    case 2:
      k1 ^= (key.charCodeAt(i + 1) & 255) << 8;
    case 1:
      k1 ^= key.charCodeAt(i) & 255;
    default:
      k1 = (k1 & 65535) * c1 + (((k1 >>> 16) * c1 & 65535) << 16) & 4294967295;
      k1 = k1 << 15 | k1 >>> 17;
      k1 = (k1 & 65535) * c2 + (((k1 >>> 16) * c2 & 65535) << 16) & 4294967295;
      h1 ^= k1;
  }
  h1 ^= key.length;
  h1 ^= h1 >>> 16;
  h1 = (h1 & 65535) * 2246822507 + (((h1 >>> 16) * 2246822507 & 65535) << 16) & 4294967295;
  h1 ^= h1 >>> 13;
  h1 = (h1 & 65535) * 3266489909 + (((h1 >>> 16) * 3266489909 & 65535) << 16) & 4294967295;
  h1 ^= h1 >>> 16;
  return h1 >>> 0;
}

const packageColors = [
  color__default["default"]({ h: 24, s: 69, l: 60 }),
  color__default["default"]({ h: 34, s: 65, l: 65 }),
  color__default["default"]({ h: 194, s: 52, l: 61 }),
  color__default["default"]({ h: 163, s: 45, l: 55 }),
  color__default["default"]({ h: 211, s: 48, l: 60 }),
  color__default["default"]({ h: 246, s: 40, l: 65 }),
  color__default["default"]({ h: 305, s: 63, l: 79 }),
  color__default["default"]({ h: 47, s: 100, l: 73 }),
  color__default["default"]({ r: 183, g: 219, b: 171 }),
  color__default["default"]({ r: 244, g: 213, b: 152 }),
  color__default["default"]({ r: 78, g: 146, b: 249 }),
  color__default["default"]({ r: 249, g: 186, b: 143 }),
  color__default["default"]({ r: 242, g: 145, b: 145 }),
  color__default["default"]({ r: 130, g: 181, b: 216 }),
  color__default["default"]({ r: 229, g: 168, b: 226 }),
  color__default["default"]({ r: 174, g: 162, b: 224 }),
  color__default["default"]({ r: 154, g: 196, b: 138 }),
  color__default["default"]({ r: 242, g: 201, b: 109 }),
  color__default["default"]({ r: 101, g: 197, b: 219 }),
  color__default["default"]({ r: 249, g: 147, b: 78 }),
  color__default["default"]({ r: 234, g: 100, b: 96 }),
  color__default["default"]({ r: 81, g: 149, b: 206 }),
  color__default["default"]({ r: 214, g: 131, b: 206 }),
  color__default["default"]({ r: 128, g: 110, b: 183 })
];
const byValueMinColor = getBarColorByValue(1, 100, 0, 1);
const byValueMaxColor = getBarColorByValue(100, 100, 0, 1);
const byValueGradient = `linear-gradient(90deg, ${byValueMinColor} 0%, ${byValueMaxColor} 100%)`;
const byPackageGradient = `linear-gradient(90deg, ${packageColors[0]} 0%, ${packageColors[2]} 30%, ${packageColors[6]} 50%, ${packageColors[7]} 70%, ${packageColors[8]} 100%)`;
function getBarColorByValue(value, totalTicks, rangeMin, rangeMax) {
  const intensity = Math.min(1, value / totalTicks / (rangeMax - rangeMin));
  const h = 50 - 50 * intensity;
  const l = 65 + 7 * intensity;
  return color__default["default"]({ h, s: 100, l });
}
function getBarColorByPackage(label, theme) {
  const packageName = getPackageName(label);
  const hash = murmurhash3_32_gc(packageName || "", 0);
  const colorIndex = hash % packageColors.length;
  let packageColor = packageColors[colorIndex].clone();
  if (theme.isLight) {
    packageColor = packageColor.brighten(15);
  }
  return packageColor;
}
const diffDefaultColors = ["rgb(0, 170, 0)", "rgb(148, 142, 142)", "rgb(200, 0, 0)"];
const diffDefaultGradient = `linear-gradient(90deg, ${diffDefaultColors[0]} 0%, ${diffDefaultColors[1]} 50%, ${diffDefaultColors[2]} 100%)`;
const diffColorBlindColors = ["rgb(26, 133, 255)", "rgb(148, 142, 142)", "rgb(220, 50, 32)"];
const diffColorBlindGradient = `linear-gradient(90deg, ${diffColorBlindColors[0]} 0%, ${diffColorBlindColors[1]} 50%, ${diffColorBlindColors[2]} 100%)`;
function getBarColorByDiff(ticks, ticksRight, totalTicks, totalTicksRight, colorScheme) {
  const range = colorScheme === ColorSchemeDiff.Default ? diffDefaultColors : diffColorBlindColors;
  const colorScale = d3.scaleLinear().domain([-100, 0, 100]).range(range);
  const ticksLeft = ticks - ticksRight;
  const totalTicksLeft = totalTicks - totalTicksRight;
  if (totalTicksRight === 0 || totalTicksLeft === 0) {
    const rgbString2 = colorScale(0);
    return color__default["default"](rgbString2);
  }
  const percentageLeft = Math.round(1e4 * ticksLeft / totalTicksLeft) / 100;
  const percentageRight = Math.round(1e4 * ticksRight / totalTicksRight) / 100;
  const diff = (percentageRight - percentageLeft) / percentageLeft * 100;
  const rgbString = colorScale(diff);
  return color__default["default"](rgbString);
}
const matchers = [
  ["phpspy", new RegExp("^(?<packageName>([^\\/]*\\/)*)(?<filename>.*\\.php+)(?<line_info>.*)$")],
  ["pyspy", new RegExp("^(?<packageName>([^\\/]*\\/)*)(?<filename>.*\\.py+)(?<line_info>.*)$")],
  ["rbspy", new RegExp("^(?<packageName>([^\\/]*\\/)*)(?<filename>.*\\.rb+)(?<line_info>.*)$")],
  [
    "nodespy",
    new RegExp("^(\\.\\/node_modules\\/)?(?<packageName>[^/]*)(?<filename>.*\\.?(jsx?|tsx?)?):(?<functionName>.*):(?<line_info>.*)$")
  ],
  ["gospy", new RegExp("^(?<packageName>.*?\\/.*?\\.|.*?\\.|.+)(?<functionName>.*)$")],
  // also 'scrape'
  ["javaspy", new RegExp("^(?<packageName>.+\\/)(?<filename>.+\\.)(?<functionName>.+)$")],
  ["dotnetspy", new RegExp("^(?<packageName>.+)\\.(.+)\\.(.+)\\(.*\\)$")],
  ["tracing", new RegExp("^(?<packageName>.+?):.*$")],
  ["pyroscope-rs", new RegExp("^(?<packageName>[^::]+)")],
  ["ebpfspy", new RegExp("^(?<packageName>.+)$")],
  ["unknown", new RegExp("^(?<packageName>.+)$")]
];
function getPackageName(name) {
  var _a;
  for (const [_, matcher] of matchers) {
    const match = name.match(matcher);
    if (match) {
      return ((_a = match.groups) == null ? void 0 : _a.packageName) || "";
    }
  }
  return void 0;
}

function useFlameRender(options) {
  const {
    canvasRef,
    data,
    root,
    depth,
    direction,
    wrapperWidth,
    rangeMin,
    rangeMax,
    matchedLabels,
    textAlign,
    totalViewTicks,
    totalColorTicks,
    totalTicksRight,
    colorScheme,
    focusedItemData,
    collapsedMap
  } = options;
  const ctx = useSetupCanvas(canvasRef, wrapperWidth, depth);
  const theme = ui.useTheme2();
  const mutedColor = React.useMemo(() => {
    const barMutedColor = color__default["default"](theme.colors.background.secondary);
    return theme.isLight ? barMutedColor.darken(10).toHexString() : barMutedColor.lighten(10).toHexString();
  }, [theme]);
  const getBarColor = useColorFunction(
    totalColorTicks,
    totalTicksRight,
    colorScheme,
    theme,
    mutedColor,
    rangeMin,
    rangeMax,
    matchedLabels,
    focusedItemData ? focusedItemData.item.level : 0
  );
  const renderFunc = useRenderFunc(ctx, data, getBarColor, textAlign, collapsedMap);
  React.useEffect(() => {
    if (!ctx) {
      return;
    }
    ctx.clearRect(0, 0, ctx.canvas.width, ctx.canvas.height);
    const mutedPath2D = new Path2D();
    walkTree(
      root,
      direction,
      data,
      totalViewTicks,
      rangeMin,
      rangeMax,
      wrapperWidth,
      collapsedMap,
      (item, x, y, width, height, label, muted) => {
        if (muted) {
          mutedPath2D.rect(x, y, width, height);
        } else {
          renderFunc(item, x, y, width, height, label);
        }
      }
    );
    ctx.fillStyle = mutedColor;
    ctx.fill(mutedPath2D);
  }, [
    ctx,
    data,
    root,
    wrapperWidth,
    rangeMin,
    rangeMax,
    totalViewTicks,
    direction,
    renderFunc,
    collapsedMap,
    mutedColor
  ]);
}
function useRenderFunc(ctx, data, getBarColor, textAlign, collapsedMap) {
  return React.useMemo(() => {
    if (!ctx) {
      return () => {
      };
    }
    const renderFunc = (item, x, y, width, height, label) => {
      ctx.beginPath();
      ctx.rect(x + BAR_BORDER_WIDTH, y, width, height);
      ctx.fillStyle = getBarColor(item, label, false);
      ctx.stroke();
      ctx.fill();
      const collapsedItemConfig = collapsedMap.get(item);
      let finalLabel = label;
      if (collapsedItemConfig && collapsedItemConfig.collapsed) {
        const numberOfCollapsedItems = collapsedItemConfig.items.length;
        finalLabel = `(${numberOfCollapsedItems}) ` + label;
      }
      if (width >= LABEL_THRESHOLD) {
        if (collapsedItemConfig) {
          renderLabel(
            ctx,
            data,
            finalLabel,
            item,
            width,
            textAlign === "left" ? x + GROUP_STRIP_MARGIN_LEFT + GROUP_TEXT_OFFSET : x,
            y,
            textAlign
          );
          renderGroupingStrip(ctx, x, y, height, item, collapsedItemConfig);
        } else {
          renderLabel(ctx, data, finalLabel, item, width, x, y, textAlign);
        }
      }
    };
    return renderFunc;
  }, [ctx, getBarColor, textAlign, data, collapsedMap]);
}
function renderGroupingStrip(ctx, x, y, height, item, collapsedItemConfig) {
  const groupStripX = x + GROUP_STRIP_MARGIN_LEFT;
  ctx.beginPath();
  ctx.rect(x, y, groupStripX - x + GROUP_STRIP_WIDTH + GROUP_STRIP_PADDING, height);
  ctx.fill();
  ctx.beginPath();
  if (collapsedItemConfig.collapsed) {
    ctx.rect(groupStripX, y + height / 4, GROUP_STRIP_WIDTH, height / 2);
  } else {
    if (collapsedItemConfig.items[0] === item) {
      ctx.rect(groupStripX, y + height / 2, GROUP_STRIP_WIDTH, height / 2);
    } else if (collapsedItemConfig.items[collapsedItemConfig.items.length - 1] === item) {
      ctx.rect(groupStripX, y, GROUP_STRIP_WIDTH, height / 2);
    } else {
      ctx.rect(groupStripX, y, GROUP_STRIP_WIDTH, height);
    }
  }
  ctx.fillStyle = "#666";
  ctx.fill();
}
function walkTree(root, direction, data, totalViewTicks, rangeMin, rangeMax, wrapperWidth, collapsedMap, renderFunc) {
  const stack = [];
  stack.push({ item: root, levelOffset: 0 });
  const pixelsPerTick = wrapperWidth * window.devicePixelRatio / totalViewTicks / (rangeMax - rangeMin);
  let collapsedItemRendered = void 0;
  while (stack.length > 0) {
    const { item, levelOffset } = stack.shift();
    let curBarTicks = item.value;
    const muted = curBarTicks * pixelsPerTick <= MUTE_THRESHOLD;
    const width = curBarTicks * pixelsPerTick - (muted ? 0 : BAR_BORDER_WIDTH * 2);
    const height = PIXELS_PER_LEVEL;
    if (width < HIDE_THRESHOLD) {
      continue;
    }
    let offsetModifier = 0;
    let skipRender = false;
    const collapsedItemConfig = collapsedMap.get(item);
    const isCollapsedItem = collapsedItemConfig && collapsedItemConfig.collapsed;
    if (isCollapsedItem) {
      if (collapsedItemRendered === collapsedItemConfig.items[0]) {
        offsetModifier = direction === "children" ? -1 : 1;
        skipRender = true;
      } else {
        collapsedItemRendered = void 0;
      }
    } else {
      collapsedItemRendered = void 0;
    }
    if (!skipRender) {
      const barX = getBarX(item.start, totalViewTicks, rangeMin, pixelsPerTick);
      const barY = (item.level + levelOffset) * PIXELS_PER_LEVEL;
      let label = data.getLabel(item.itemIndexes[0]);
      if (isCollapsedItem) {
        collapsedItemRendered = item;
      }
      renderFunc(item, barX, barY, width, height, label, muted);
    }
    const nextList = direction === "children" ? item.children : item.parents;
    if (nextList) {
      stack.unshift(...nextList.map((c) => ({ item: c, levelOffset: levelOffset + offsetModifier })));
    }
  }
}
function useColorFunction(totalTicks, totalTicksRight, colorScheme, theme, mutedColor, rangeMin, rangeMax, matchedLabels, topLevel) {
  return React.useCallback(
    function getColor(item, label, muted) {
      if (muted && !matchedLabels) {
        return mutedColor;
      }
      const barColor = item.valueRight !== void 0 && (colorScheme === ColorSchemeDiff.Default || colorScheme === ColorSchemeDiff.DiffColorBlind) ? getBarColorByDiff(item.value, item.valueRight, totalTicks, totalTicksRight, colorScheme) : colorScheme === ColorScheme.ValueBased ? getBarColorByValue(item.value, totalTicks, rangeMin, rangeMax) : getBarColorByPackage(label, theme);
      if (matchedLabels) {
        return matchedLabels.has(label) ? barColor.toHslString() : mutedColor;
      }
      return item.level > topLevel - 1 ? barColor.toHslString() : barColor.lighten(15).toHslString();
    },
    [totalTicks, totalTicksRight, colorScheme, theme, rangeMin, rangeMax, matchedLabels, topLevel, mutedColor]
  );
}
function useSetupCanvas(canvasRef, wrapperWidth, numberOfLevels) {
  const [ctx, setCtx] = React.useState();
  React.useEffect(() => {
    if (!(numberOfLevels && canvasRef.current)) {
      return;
    }
    const ctx2 = canvasRef.current.getContext("2d");
    const height = PIXELS_PER_LEVEL * numberOfLevels;
    canvasRef.current.width = Math.round(wrapperWidth * window.devicePixelRatio);
    canvasRef.current.height = Math.round(height);
    canvasRef.current.style.width = `${wrapperWidth}px`;
    canvasRef.current.style.height = `${height / window.devicePixelRatio}px`;
    ctx2.textBaseline = "middle";
    ctx2.font = 12 * window.devicePixelRatio + "px monospace";
    ctx2.strokeStyle = "white";
    setCtx(ctx2);
  }, [canvasRef, setCtx, wrapperWidth, numberOfLevels]);
  return ctx;
}
function renderLabel(ctx, data, label, item, width, x, y, textAlign) {
  ctx.save();
  ctx.clip();
  ctx.fillStyle = "#222";
  const displayValue = data.valueDisplayProcessor(item.value);
  const unit = displayValue.suffix ? displayValue.text + displayValue.suffix : displayValue.text;
  const measure = ctx.measureText(label);
  const spaceForTextInRect = width - BAR_TEXT_PADDING_LEFT;
  let fullLabel = `${label} (${unit})`;
  let labelX = Math.max(x, 0) + BAR_TEXT_PADDING_LEFT;
  if (measure.width > spaceForTextInRect) {
    ctx.textAlign = textAlign;
    if (textAlign === "right") {
      fullLabel = label;
      labelX = x + width - BAR_TEXT_PADDING_LEFT;
    }
  }
  ctx.fillText(fullLabel, labelX, y + PIXELS_PER_LEVEL / 2 + 2);
  ctx.restore();
}
function getBarX(offset, totalTicks, rangeMin, pixelsPerTick) {
  return (offset - totalTicks * rangeMin) * pixelsPerTick;
}

const FlameGraphCanvas = ({
  data,
  rangeMin,
  rangeMax,
  matchedLabels,
  setRangeMin,
  setRangeMax,
  onItemFocused,
  focusedItemData,
  textAlign,
  onSandwich,
  colorScheme,
  totalProfileTicks,
  totalProfileTicksRight,
  totalViewTicks,
  root,
  direction,
  depth,
  showFlameGraphOnly,
  collapsedMap,
  setCollapsedMap,
  collapsing,
  getExtraContextMenuButtons,
  selectedView,
  search
}) => {
  const styles = getStyles$5();
  const [sizeRef, { width: wrapperWidth }] = reactUse.useMeasure();
  const graphRef = React.useRef(null);
  const [tooltipItem, setTooltipItem] = React.useState();
  const [clickedItemData, setClickedItemData] = React.useState();
  useFlameRender({
    canvasRef: graphRef,
    colorScheme,
    data,
    focusedItemData,
    root,
    direction,
    depth,
    rangeMax,
    rangeMin,
    matchedLabels,
    textAlign,
    totalViewTicks,
    // We need this so that if we have a diff profile and are in sandwich view we still show the same diff colors.
    totalColorTicks: data.isDiffFlamegraph() ? totalProfileTicks : totalViewTicks,
    totalTicksRight: totalProfileTicksRight,
    wrapperWidth,
    collapsedMap
  });
  const onGraphClick = React.useCallback(
    (e) => {
      setTooltipItem(void 0);
      const pixelsPerTick = graphRef.current.clientWidth / totalViewTicks / (rangeMax - rangeMin);
      const item = convertPixelCoordinatesToBarCoordinates(
        { x: e.nativeEvent.offsetX, y: e.nativeEvent.offsetY },
        root,
        direction,
        depth,
        pixelsPerTick,
        totalViewTicks,
        rangeMin,
        collapsedMap
      );
      if (item) {
        setClickedItemData({
          posY: e.clientY,
          posX: e.clientX,
          item,
          label: data.getLabel(item.itemIndexes[0])
        });
      } else {
        setClickedItemData(void 0);
      }
    },
    [data, rangeMin, rangeMax, totalViewTicks, root, direction, depth, collapsedMap]
  );
  const [mousePosition, setMousePosition] = React.useState();
  const onGraphMouseMove = React.useCallback(
    (e) => {
      if (clickedItemData === void 0) {
        setTooltipItem(void 0);
        setMousePosition(void 0);
        const pixelsPerTick = graphRef.current.clientWidth / totalViewTicks / (rangeMax - rangeMin);
        const item = convertPixelCoordinatesToBarCoordinates(
          { x: e.nativeEvent.offsetX, y: e.nativeEvent.offsetY },
          root,
          direction,
          depth,
          pixelsPerTick,
          totalViewTicks,
          rangeMin,
          collapsedMap
        );
        if (item) {
          setMousePosition({ x: e.clientX, y: e.clientY });
          setTooltipItem(item);
        }
      }
    },
    [rangeMin, rangeMax, totalViewTicks, clickedItemData, setMousePosition, root, direction, depth, collapsedMap]
  );
  const onGraphMouseLeave = React.useCallback(() => {
    setTooltipItem(void 0);
  }, []);
  React.useEffect(() => {
    const handleOnClick = (e) => {
      var _a;
      if (e.target instanceof HTMLElement && ((_a = e.target.parentElement) == null ? void 0 : _a.id) !== "flameGraphCanvasContainer_clickOutsideCheck") {
        setClickedItemData(void 0);
      }
    };
    window.addEventListener("click", handleOnClick);
    return () => window.removeEventListener("click", handleOnClick);
  }, [setClickedItemData]);
  return /* @__PURE__ */ React__default["default"].createElement("div", { className: styles.graph }, /* @__PURE__ */ React__default["default"].createElement("div", { className: styles.canvasWrapper, id: "flameGraphCanvasContainer_clickOutsideCheck", ref: sizeRef }, /* @__PURE__ */ React__default["default"].createElement(
    "canvas",
    {
      ref: graphRef,
      "data-testid": "flameGraph",
      onClick: onGraphClick,
      onMouseMove: onGraphMouseMove,
      onMouseLeave: onGraphMouseLeave
    }
  )), /* @__PURE__ */ React__default["default"].createElement(
    FlameGraphTooltip,
    {
      position: mousePosition,
      item: tooltipItem,
      data,
      totalTicks: totalViewTicks,
      collapseConfig: tooltipItem ? collapsedMap.get(tooltipItem) : void 0
    }
  ), !showFlameGraphOnly && clickedItemData && /* @__PURE__ */ React__default["default"].createElement(
    FlameGraphContextMenu,
    {
      data,
      itemData: clickedItemData,
      collapsing,
      collapseConfig: collapsedMap.get(clickedItemData.item),
      onMenuItemClick: () => {
        setClickedItemData(void 0);
      },
      onItemFocus: () => {
        setRangeMin(clickedItemData.item.start / totalViewTicks);
        setRangeMax((clickedItemData.item.start + clickedItemData.item.value) / totalViewTicks);
        onItemFocused(clickedItemData);
      },
      onSandwich: () => {
        onSandwich(data.getLabel(clickedItemData.item.itemIndexes[0]));
      },
      onExpandGroup: () => {
        setCollapsedMap(collapsedMap.setCollapsedStatus(clickedItemData.item, false));
      },
      onCollapseGroup: () => {
        setCollapsedMap(collapsedMap.setCollapsedStatus(clickedItemData.item, true));
      },
      onExpandAllGroups: () => {
        setCollapsedMap(collapsedMap.setAllCollapsedStatus(false));
      },
      onCollapseAllGroups: () => {
        setCollapsedMap(collapsedMap.setAllCollapsedStatus(true));
      },
      allGroupsCollapsed: Array.from(collapsedMap.values()).every((i) => i.collapsed),
      allGroupsExpanded: Array.from(collapsedMap.values()).every((i) => !i.collapsed),
      getExtraContextMenuButtons,
      selectedView,
      search
    }
  ));
};
const getStyles$5 = () => ({
  graph: css.css({
    label: "graph",
    overflow: "auto",
    flexGrow: 1,
    flexBasis: "50%"
  }),
  canvasContainer: css.css({
    label: "canvasContainer",
    display: "flex"
  }),
  canvasWrapper: css.css({
    label: "canvasWrapper",
    cursor: "pointer",
    flex: 1,
    overflow: "hidden"
  }),
  sandwichMarker: css.css({
    label: "sandwichMarker",
    writingMode: "vertical-lr",
    transform: "rotate(180deg)",
    overflow: "hidden",
    whiteSpace: "nowrap"
  }),
  sandwichMarkerIcon: css.css({
    label: "sandwichMarkerIcon",
    verticalAlign: "baseline"
  })
});
const convertPixelCoordinatesToBarCoordinates = (pos, root, direction, depth, pixelsPerTick, totalTicks, rangeMin, collapsedMap) => {
  let next = root;
  let currentLevel = direction === "children" ? 0 : depth - 1;
  const levelIndex = Math.floor(pos.y / (PIXELS_PER_LEVEL / window.devicePixelRatio));
  let found = void 0;
  while (next) {
    const node = next;
    next = void 0;
    if (currentLevel === levelIndex) {
      found = node;
      break;
    }
    const nextList = direction === "children" ? node.children : node.parents || [];
    for (const child of nextList) {
      const xStart = getBarX(child.start, totalTicks, rangeMin, pixelsPerTick);
      const xEnd = getBarX(child.start + child.value, totalTicks, rangeMin, pixelsPerTick);
      if (xStart <= pos.x && pos.x < xEnd) {
        next = child;
        const collapsedConfig = collapsedMap.get(child);
        if (!collapsedConfig || !collapsedConfig.collapsed || collapsedConfig.items[0] === child) {
          currentLevel = currentLevel + (direction === "children" ? 1 : -1);
        }
        break;
      }
    }
  }
  return found;
};

const FlameGraphMetadata = React__default["default"].memo(
  ({ data, focusedItem, totalTicks, sandwichedLabel, onFocusPillClick, onSandwichPillClick }) => {
    const styles = ui.useStyles2(getStyles$4);
    const parts = [];
    const ticksVal = data$1.getValueFormat("short")(totalTicks);
    const displayValue = data.valueDisplayProcessor(totalTicks);
    let unitValue = displayValue.text + displayValue.suffix;
    const unitTitle = data.getUnitTitle();
    if (unitTitle === "Count") {
      if (!displayValue.suffix) {
        unitValue = displayValue.text;
      }
    }
    parts.push(
      /* @__PURE__ */ React__default["default"].createElement("div", { className: styles.metadataPill, key: "default" }, unitValue, " | ", ticksVal.text, ticksVal.suffix, " samples (", unitTitle, ")")
    );
    if (sandwichedLabel) {
      parts.push(
        /* @__PURE__ */ React__default["default"].createElement("span", { key: "sandwich" }, /* @__PURE__ */ React__default["default"].createElement(ui.Icon, { size: "sm", name: "angle-right" }), /* @__PURE__ */ React__default["default"].createElement("div", { className: styles.metadataPill }, /* @__PURE__ */ React__default["default"].createElement(ui.Icon, { size: "sm", name: "gf-show-context" }), " ", /* @__PURE__ */ React__default["default"].createElement("span", { className: styles.metadataPillName }, sandwichedLabel.substring(sandwichedLabel.lastIndexOf("/") + 1)), /* @__PURE__ */ React__default["default"].createElement(
          ui.IconButton,
          {
            className: styles.pillCloseButton,
            name: "times",
            size: "sm",
            onClick: onSandwichPillClick,
            tooltip: "Remove sandwich view",
            "aria-label": "Remove sandwich view"
          }
        )))
      );
    }
    if (focusedItem) {
      const percentValue = Math.round(1e4 * (focusedItem.item.value / totalTicks)) / 100;
      parts.push(
        /* @__PURE__ */ React__default["default"].createElement("span", { key: "focus" }, /* @__PURE__ */ React__default["default"].createElement(ui.Icon, { size: "sm", name: "angle-right" }), /* @__PURE__ */ React__default["default"].createElement("div", { className: styles.metadataPill }, /* @__PURE__ */ React__default["default"].createElement(ui.Icon, { size: "sm", name: "eye" }), " ", percentValue, "% of total", /* @__PURE__ */ React__default["default"].createElement(
          ui.IconButton,
          {
            className: styles.pillCloseButton,
            name: "times",
            size: "sm",
            onClick: onFocusPillClick,
            tooltip: "Remove focus",
            "aria-label": "Remove focus"
          }
        )))
      );
    }
    return /* @__PURE__ */ React__default["default"].createElement("div", { className: styles.metadata }, parts);
  }
);
FlameGraphMetadata.displayName = "FlameGraphMetadata";
const getStyles$4 = (theme) => ({
  metadataPill: css.css({
    label: "metadataPill",
    display: "inline-flex",
    alignItems: "center",
    background: theme.colors.background.secondary,
    borderRadius: theme.shape.borderRadius(8),
    padding: theme.spacing(0.5, 1),
    fontSize: theme.typography.bodySmall.fontSize,
    fontWeight: theme.typography.fontWeightMedium,
    lineHeight: theme.typography.bodySmall.lineHeight,
    color: theme.colors.text.secondary
  }),
  pillCloseButton: css.css({
    label: "pillCloseButton",
    verticalAlign: "text-bottom",
    margin: theme.spacing(0, 0.5)
  }),
  metadata: css.css({
    margin: "8px 0",
    textAlign: "center"
  }),
  metadataPillName: css.css({
    label: "metadataPillName",
    maxWidth: "200px",
    overflow: "hidden",
    textOverflow: "ellipsis",
    whiteSpace: "nowrap",
    marginLeft: theme.spacing(0.5)
  })
});

var __defProp$2 = Object.defineProperty;
var __defProps$2 = Object.defineProperties;
var __getOwnPropDescs$2 = Object.getOwnPropertyDescriptors;
var __getOwnPropSymbols$2 = Object.getOwnPropertySymbols;
var __hasOwnProp$2 = Object.prototype.hasOwnProperty;
var __propIsEnum$2 = Object.prototype.propertyIsEnumerable;
var __defNormalProp$2 = (obj, key, value) => key in obj ? __defProp$2(obj, key, { enumerable: true, configurable: true, writable: true, value }) : obj[key] = value;
var __spreadValues$2 = (a, b) => {
  for (var prop in b || (b = {}))
    if (__hasOwnProp$2.call(b, prop))
      __defNormalProp$2(a, prop, b[prop]);
  if (__getOwnPropSymbols$2)
    for (var prop of __getOwnPropSymbols$2(b)) {
      if (__propIsEnum$2.call(b, prop))
        __defNormalProp$2(a, prop, b[prop]);
    }
  return a;
};
var __spreadProps$2 = (a, b) => __defProps$2(a, __getOwnPropDescs$2(b));
const FlameGraph = ({
  data,
  rangeMin,
  rangeMax,
  matchedLabels,
  setRangeMin,
  setRangeMax,
  onItemFocused,
  focusedItemData,
  textAlign,
  onSandwich,
  sandwichItem,
  onFocusPillClick,
  onSandwichPillClick,
  colorScheme,
  showFlameGraphOnly,
  getExtraContextMenuButtons,
  collapsing,
  selectedView,
  search,
  collapsedMap,
  setCollapsedMap
}) => {
  const styles = getStyles$3();
  const [levels, setLevels] = React.useState();
  const [levelsCallers, setLevelsCallers] = React.useState();
  const [totalProfileTicks, setTotalProfileTicks] = React.useState(0);
  const [totalProfileTicksRight, setTotalProfileTicksRight] = React.useState();
  const [totalViewTicks, setTotalViewTicks] = React.useState(0);
  React.useEffect(() => {
    var _a, _b, _c;
    if (data) {
      let levels2 = data.getLevels();
      let totalProfileTicks2 = levels2.length ? levels2[0][0].value : 0;
      let totalProfileTicksRight2 = levels2.length ? levels2[0][0].valueRight : void 0;
      let totalViewTicks2 = totalProfileTicks2;
      let levelsCallers2 = void 0;
      if (sandwichItem) {
        const [callers, callees] = data.getSandwichLevels(sandwichItem);
        levels2 = callees;
        levelsCallers2 = callers;
        totalViewTicks2 = (_c = (_b = (_a = callees[0]) == null ? void 0 : _a[0]) == null ? void 0 : _b.value) != null ? _c : 0;
      }
      setLevels(levels2);
      setLevelsCallers(levelsCallers2);
      setTotalProfileTicks(totalProfileTicks2);
      setTotalProfileTicksRight(totalProfileTicksRight2);
      setTotalViewTicks(totalViewTicks2);
    }
  }, [data, sandwichItem]);
  if (!levels) {
    return null;
  }
  const commonCanvasProps = {
    data,
    rangeMin,
    rangeMax,
    matchedLabels,
    setRangeMin,
    setRangeMax,
    onItemFocused,
    focusedItemData,
    textAlign,
    onSandwich,
    colorScheme,
    totalProfileTicks,
    totalProfileTicksRight,
    totalViewTicks,
    showFlameGraphOnly,
    collapsedMap,
    setCollapsedMap,
    getExtraContextMenuButtons,
    collapsing,
    search,
    selectedView
  };
  const canvas = levelsCallers ? /* @__PURE__ */ React__default["default"].createElement(React__default["default"].Fragment, null, /* @__PURE__ */ React__default["default"].createElement("div", { className: styles.sandwichCanvasWrapper }, /* @__PURE__ */ React__default["default"].createElement("div", { className: styles.sandwichMarker }, "Callers", /* @__PURE__ */ React__default["default"].createElement(ui.Icon, { className: styles.sandwichMarkerIcon, name: "arrow-down" })), /* @__PURE__ */ React__default["default"].createElement(
    FlameGraphCanvas,
    __spreadProps$2(__spreadValues$2({}, commonCanvasProps), {
      root: levelsCallers[levelsCallers.length - 1][0],
      depth: levelsCallers.length,
      direction: "parents",
      collapsing: false
    })
  )), /* @__PURE__ */ React__default["default"].createElement("div", { className: styles.sandwichCanvasWrapper }, /* @__PURE__ */ React__default["default"].createElement("div", { className: css.cx(styles.sandwichMarker, styles.sandwichMarkerCalees) }, /* @__PURE__ */ React__default["default"].createElement(ui.Icon, { className: styles.sandwichMarkerIcon, name: "arrow-up" }), "Callees"), /* @__PURE__ */ React__default["default"].createElement(
    FlameGraphCanvas,
    __spreadProps$2(__spreadValues$2({}, commonCanvasProps), {
      root: levels[0][0],
      depth: levels.length,
      direction: "children",
      collapsing: false
    })
  ))) : /* @__PURE__ */ React__default["default"].createElement(FlameGraphCanvas, __spreadProps$2(__spreadValues$2({}, commonCanvasProps), { root: levels[0][0], depth: levels.length, direction: "children" }));
  return /* @__PURE__ */ React__default["default"].createElement("div", { className: styles.graph }, /* @__PURE__ */ React__default["default"].createElement(
    FlameGraphMetadata,
    {
      data,
      focusedItem: focusedItemData,
      sandwichedLabel: sandwichItem,
      totalTicks: totalViewTicks,
      onFocusPillClick,
      onSandwichPillClick
    }
  ), canvas);
};
const getStyles$3 = () => ({
  graph: css.css({
    label: "graph",
    overflow: "auto",
    flexGrow: 1,
    flexBasis: "50%"
  }),
  sandwichCanvasWrapper: css.css({
    label: "sandwichCanvasWrapper",
    display: "flex",
    marginBottom: `${PIXELS_PER_LEVEL / window.devicePixelRatio}px`
  }),
  sandwichMarker: css.css({
    label: "sandwichMarker",
    writingMode: "vertical-lr",
    transform: "rotate(180deg)",
    overflow: "hidden",
    whiteSpace: "nowrap"
  }),
  sandwichMarkerCalees: css.css({
    label: "sandwichMarkerCalees",
    textAlign: "right"
  }),
  sandwichMarkerIcon: css.css({
    label: "sandwichMarkerIcon",
    verticalAlign: "baseline"
  })
});

var __defProp$1 = Object.defineProperty;
var __defProps$1 = Object.defineProperties;
var __getOwnPropDescs$1 = Object.getOwnPropertyDescriptors;
var __getOwnPropSymbols$1 = Object.getOwnPropertySymbols;
var __hasOwnProp$1 = Object.prototype.hasOwnProperty;
var __propIsEnum$1 = Object.prototype.propertyIsEnumerable;
var __defNormalProp$1 = (obj, key, value) => key in obj ? __defProp$1(obj, key, { enumerable: true, configurable: true, writable: true, value }) : obj[key] = value;
var __spreadValues$1 = (a, b) => {
  for (var prop in b || (b = {}))
    if (__hasOwnProp$1.call(b, prop))
      __defNormalProp$1(a, prop, b[prop]);
  if (__getOwnPropSymbols$1)
    for (var prop of __getOwnPropSymbols$1(b)) {
      if (__propIsEnum$1.call(b, prop))
        __defNormalProp$1(a, prop, b[prop]);
    }
  return a;
};
var __spreadProps$1 = (a, b) => __defProps$1(a, __getOwnPropDescs$1(b));
function mergeParentSubtrees(roots, data) {
  const newRoots = getParentSubtrees(roots);
  return mergeSubtrees(newRoots, data, "parents");
}
function getParentSubtrees(roots) {
  return roots.map((r) => {
    var _a, _b;
    if (!((_a = r.parents) == null ? void 0 : _a.length)) {
      return r;
    }
    const newRoot = __spreadProps$1(__spreadValues$1({}, r), {
      children: []
    });
    const stack = [
      { child: newRoot, parent: r.parents[0] }
    ];
    while (stack.length) {
      const args = stack.shift();
      const newNode = __spreadProps$1(__spreadValues$1({}, args.parent), {
        children: args.child ? [args.child] : [],
        parents: []
      });
      if (args.child) {
        newNode.value = args.child.value;
        args.child.parents = [newNode];
      }
      if ((_b = args.parent.parents) == null ? void 0 : _b.length) {
        stack.push({ child: newNode, parent: args.parent.parents[0] });
      }
    }
    return newRoot;
  });
}
function mergeSubtrees(roots, data, direction = "children") {
  var _a;
  const oppositeDirection = direction === "parents" ? "children" : "parents";
  const levels = [];
  const stack = [
    { previous: void 0, items: roots, level: 0 }
  ];
  while (stack.length) {
    const args = stack.shift();
    const indexes = args.items.flatMap((i) => i.itemIndexes);
    const newItem = {
      // We use the items value instead of value from the data frame, cause we could have changed it in the process
      value: args.items.reduce((acc, i) => acc + i.value, 0),
      itemIndexes: indexes,
      // these will change later
      children: [],
      parents: [],
      start: 0,
      level: args.level
    };
    levels[args.level] = levels[args.level] || [];
    levels[args.level].push(newItem);
    if (args.previous) {
      newItem[oppositeDirection] = [args.previous];
      const prevSiblingsVal = ((_a = args.previous[direction]) == null ? void 0 : _a.reduce((acc, node) => {
        return acc + node.value;
      }, 0)) || 0;
      newItem.start = args.previous.start + prevSiblingsVal;
      args.previous[direction].push(newItem);
    }
    const nextItems = args.items.flatMap((i) => i[direction] || []);
    const nextGroups = lodash.groupBy(nextItems, (c) => data.getLabel(c.itemIndexes[0]));
    for (const g of Object.values(nextGroups)) {
      stack.push({ previous: newItem, items: g, level: args.level + 1 });
    }
  }
  if (direction === "parents") {
    levels.reverse();
    levels.forEach((level, index) => {
      level.forEach((item) => {
        item.level = index;
      });
    });
  }
  return levels;
}

var __defProp = Object.defineProperty;
var __defProps = Object.defineProperties;
var __getOwnPropDescs = Object.getOwnPropertyDescriptors;
var __getOwnPropSymbols = Object.getOwnPropertySymbols;
var __hasOwnProp = Object.prototype.hasOwnProperty;
var __propIsEnum = Object.prototype.propertyIsEnumerable;
var __defNormalProp = (obj, key, value) => key in obj ? __defProp(obj, key, { enumerable: true, configurable: true, writable: true, value }) : obj[key] = value;
var __spreadValues = (a, b) => {
  for (var prop in b || (b = {}))
    if (__hasOwnProp.call(b, prop))
      __defNormalProp(a, prop, b[prop]);
  if (__getOwnPropSymbols)
    for (var prop of __getOwnPropSymbols(b)) {
      if (__propIsEnum.call(b, prop))
        __defNormalProp(a, prop, b[prop]);
    }
  return a;
};
var __spreadProps = (a, b) => __defProps(a, __getOwnPropDescs(b));
var __publicField = (obj, key, value) => {
  __defNormalProp(obj, typeof key !== "symbol" ? key + "" : key, value);
  return value;
};
function nestedSetToLevels(container, options) {
  const levels = [];
  let offset = 0;
  let parent = void 0;
  const uniqueLabels = {};
  for (let i = 0; i < container.data.length; i++) {
    const currentLevel = container.getLevel(i);
    const prevLevel = i > 0 ? container.getLevel(i - 1) : void 0;
    levels[currentLevel] = levels[currentLevel] || [];
    if (prevLevel && prevLevel >= currentLevel) {
      const lastSibling = levels[currentLevel][levels[currentLevel].length - 1];
      offset = lastSibling.start + container.getValue(lastSibling.itemIndexes[0]) + container.getValueRight(lastSibling.itemIndexes[0]);
      parent = lastSibling.parents[0];
    }
    const newItem = {
      itemIndexes: [i],
      value: container.getValue(i) + container.getValueRight(i),
      valueRight: container.isDiffFlamegraph() ? container.getValueRight(i) : void 0,
      start: offset,
      parents: parent && [parent],
      children: [],
      level: currentLevel
    };
    if (uniqueLabels[container.getLabel(i)]) {
      uniqueLabels[container.getLabel(i)].push(newItem);
    } else {
      uniqueLabels[container.getLabel(i)] = [newItem];
    }
    if (parent) {
      parent.children.push(newItem);
    }
    parent = newItem;
    levels[currentLevel].push(newItem);
  }
  const collapsedMapContainer = new CollapsedMapBuilder(options == null ? void 0 : options.collapsingThreshold);
  if (options == null ? void 0 : options.collapsing) {
    collapsedMapContainer.addTree(levels[0][0]);
  }
  return [levels, uniqueLabels, collapsedMapContainer.getCollapsedMap()];
}
class CollapsedMap {
  constructor(map) {
    // The levelItem used as a key is the item that will always be rendered in the flame graph. The config.items are all
    // the items that are in the group and if the config.collapsed is true they will be hidden.
    __publicField(this, "map", /* @__PURE__ */ new Map());
    this.map = map || /* @__PURE__ */ new Map();
  }
  get(item) {
    return this.map.get(item);
  }
  keys() {
    return this.map.keys();
  }
  values() {
    return this.map.values();
  }
  size() {
    return this.map.size;
  }
  setCollapsedStatus(item, collapsed) {
    const newMap = new Map(this.map);
    const collapsedConfig = this.map.get(item);
    const newConfig = __spreadProps(__spreadValues({}, collapsedConfig), { collapsed });
    for (const item2 of collapsedConfig.items) {
      newMap.set(item2, newConfig);
    }
    return new CollapsedMap(newMap);
  }
  setAllCollapsedStatus(collapsed) {
    const newMap = new Map(this.map);
    for (const item of this.map.keys()) {
      const collapsedConfig = this.map.get(item);
      const newConfig = __spreadProps(__spreadValues({}, collapsedConfig), { collapsed });
      newMap.set(item, newConfig);
    }
    return new CollapsedMap(newMap);
  }
}
class CollapsedMapBuilder {
  constructor(threshold) {
    __publicField(this, "map", /* @__PURE__ */ new Map());
    __publicField(this, "threshold", 0.99);
    if (threshold !== void 0) {
      this.threshold = threshold;
    }
  }
  addTree(root) {
    var _a;
    const stack = [root];
    while (stack.length) {
      const current = stack.shift();
      if ((_a = current.parents) == null ? void 0 : _a.length) {
        this.addItem(current, current.parents[0]);
      }
      if (current.children.length) {
        stack.unshift(...current.children);
      }
    }
  }
  // The heuristics here is pretty simple right now. Just check if it's single child and if we are within threshold.
  // We assume items with small self just aren't too important while we cannot really collapse items with siblings
  // as it's not clear what to do with said sibling.
  addItem(item, parent) {
    if (parent && item.value > parent.value * this.threshold && parent.children.length === 1) {
      if (this.map.has(parent)) {
        const config = this.map.get(parent);
        this.map.set(item, config);
        config.items.push(item);
      } else {
        const config = { items: [parent, item], collapsed: true };
        this.map.set(parent, config);
        this.map.set(item, config);
      }
    }
  }
  getCollapsedMap() {
    return new CollapsedMap(this.map);
  }
}
function getMessageCheckFieldsResult(wrongFields) {
  if (wrongFields.missingFields.length) {
    return `Data is missing fields: ${wrongFields.missingFields.join(", ")}`;
  }
  if (wrongFields.wrongTypeFields.length) {
    return `Data has fields of wrong type: ${wrongFields.wrongTypeFields.map((f) => `${f.name} has type ${f.type} but should be ${f.expectedTypes.join(" or ")}`).join(", ")}`;
  }
  return "";
}
function checkFields(data) {
  const fields = [
    ["label", [data$1.FieldType.string, data$1.FieldType.enum]],
    ["level", [data$1.FieldType.number]],
    ["value", [data$1.FieldType.number]],
    ["self", [data$1.FieldType.number]]
  ];
  const missingFields = [];
  const wrongTypeFields = [];
  for (const field of fields) {
    const [name, types] = field;
    const frameField = data.fields.find((f) => f.name === name);
    if (!frameField) {
      missingFields.push(name);
      continue;
    }
    if (!types.includes(frameField.type)) {
      wrongTypeFields.push({ name, expectedTypes: types, type: frameField.type });
    }
  }
  if (missingFields.length > 0 || wrongTypeFields.length > 0) {
    return {
      wrongTypeFields,
      missingFields
    };
  }
  return void 0;
}
class FlameGraphDataContainer {
  constructor(data, options, theme = data$1.createTheme()) {
    __publicField(this, "data");
    __publicField(this, "options");
    __publicField(this, "labelField");
    __publicField(this, "levelField");
    __publicField(this, "valueField");
    __publicField(this, "selfField");
    // Optional fields for diff view
    __publicField(this, "valueRightField");
    __publicField(this, "selfRightField");
    __publicField(this, "labelDisplayProcessor");
    __publicField(this, "valueDisplayProcessor");
    __publicField(this, "uniqueLabels");
    __publicField(this, "levels");
    __publicField(this, "uniqueLabelsMap");
    __publicField(this, "collapsedMap");
    var _a, _b, _c;
    this.data = data;
    this.options = options;
    const wrongFields = checkFields(data);
    if (wrongFields) {
      throw new Error(getMessageCheckFieldsResult(wrongFields));
    }
    this.labelField = data.fields.find((f) => f.name === "label");
    this.levelField = data.fields.find((f) => f.name === "level");
    this.valueField = data.fields.find((f) => f.name === "value");
    this.selfField = data.fields.find((f) => f.name === "self");
    this.valueRightField = data.fields.find((f) => f.name === "valueRight");
    this.selfRightField = data.fields.find((f) => f.name === "selfRight");
    if ((this.valueField || this.selfField) && !(this.valueField && this.selfField)) {
      throw new Error(
        "Malformed dataFrame: both valueRight and selfRight has to be present if one of them is present."
      );
    }
    const enumConfig = (_c = (_b = (_a = this.labelField) == null ? void 0 : _a.config) == null ? void 0 : _b.type) == null ? void 0 : _c.enum;
    if (enumConfig) {
      this.labelDisplayProcessor = data$1.getDisplayProcessor({ field: this.labelField, theme });
      this.uniqueLabels = enumConfig.text || [];
    } else {
      this.labelDisplayProcessor = (value) => ({
        text: value + "",
        numeric: 0
      });
      this.uniqueLabels = [...new Set(this.labelField.values)];
    }
    this.valueDisplayProcessor = data$1.getDisplayProcessor({
      field: this.valueField,
      theme
    });
  }
  isDiffFlamegraph() {
    return Boolean(this.valueRightField && this.selfRightField);
  }
  getLabel(index) {
    return this.labelDisplayProcessor(this.labelField.values[index]).text;
  }
  getLevel(index) {
    return this.levelField.values[index];
  }
  getValue(index) {
    return fieldAccessor(this.valueField, index);
  }
  getValueRight(index) {
    return fieldAccessor(this.valueRightField, index);
  }
  getSelf(index) {
    return fieldAccessor(this.selfField, index);
  }
  getSelfRight(index) {
    return fieldAccessor(this.selfRightField, index);
  }
  getSelfDisplay(index) {
    return this.valueDisplayProcessor(this.getSelf(index));
  }
  getUniqueLabels() {
    return this.uniqueLabels;
  }
  getUnitTitle() {
    switch (this.valueField.config.unit) {
      case SampleUnit.Bytes:
        return "RAM";
      case SampleUnit.Nanoseconds:
        return "Time";
    }
    return "Count";
  }
  getLevels() {
    this.initLevels();
    return this.levels;
  }
  getSandwichLevels(label) {
    const nodes = this.getNodesWithLabel(label);
    if (!(nodes == null ? void 0 : nodes.length)) {
      return [[], []];
    }
    const callers = mergeParentSubtrees(nodes, this);
    const callees = mergeSubtrees(nodes, this);
    return [callers, callees];
  }
  getNodesWithLabel(label) {
    this.initLevels();
    return this.uniqueLabelsMap[label];
  }
  getCollapsedMap() {
    this.initLevels();
    return this.collapsedMap;
  }
  initLevels() {
    if (!this.levels) {
      const [levels, uniqueLabelsMap, collapsedMap] = nestedSetToLevels(this, this.options);
      this.levels = levels;
      this.uniqueLabelsMap = uniqueLabelsMap;
      this.collapsedMap = collapsedMap;
    }
  }
}
function fieldAccessor(field, index) {
  if (!field) {
    return 0;
  }
  let indexArray = typeof index === "number" ? [index] : index;
  return indexArray.reduce((acc, index2) => {
    return acc + field.values[index2];
  }, 0);
}

const FlameGraphHeader = ({
  search,
  setSearch,
  selectedView,
  setSelectedView,
  containerWidth,
  onReset,
  textAlign,
  onTextAlignChange,
  showResetButton,
  colorScheme,
  onColorSchemeChange,
  stickyHeader,
  extraHeaderElements,
  vertical,
  isDiffMode,
  setCollapsedMap,
  collapsedMap
}) => {
  const styles = ui.useStyles2(getStyles$2);
  const [localSearch, setLocalSearch] = useSearchInput(search, setSearch);
  const suffix = localSearch !== "" ? /* @__PURE__ */ React__default["default"].createElement(
    ui.Button,
    {
      icon: "times",
      fill: "text",
      size: "sm",
      onClick: () => {
        setSearch("");
        setLocalSearch("");
      }
    },
    "Clear"
  ) : null;
  return /* @__PURE__ */ React__default["default"].createElement("div", { className: css.cx(styles.header, { [styles.stickyHeader]: stickyHeader }) }, /* @__PURE__ */ React__default["default"].createElement("div", { className: styles.inputContainer }, /* @__PURE__ */ React__default["default"].createElement(
    ui.Input,
    {
      value: localSearch || "",
      onChange: (v) => {
        setLocalSearch(v.currentTarget.value);
      },
      placeholder: "Search...",
      suffix
    }
  )), /* @__PURE__ */ React__default["default"].createElement("div", { className: styles.rightContainer }, showResetButton && /* @__PURE__ */ React__default["default"].createElement(
    ui.Button,
    {
      variant: "secondary",
      fill: "outline",
      size: "sm",
      icon: "history-alt",
      tooltip: "Reset focus and sandwich state",
      onClick: () => {
        onReset();
      },
      className: styles.buttonSpacing,
      "aria-label": "Reset focus and sandwich state"
    }
  ), /* @__PURE__ */ React__default["default"].createElement(ColorSchemeButton, { value: colorScheme, onChange: onColorSchemeChange, isDiffMode }), /* @__PURE__ */ React__default["default"].createElement(ui.ButtonGroup, { className: styles.buttonSpacing }, /* @__PURE__ */ React__default["default"].createElement(
    ui.Button,
    {
      variant: "secondary",
      fill: "outline",
      size: "sm",
      tooltip: "Expand all groups",
      onClick: () => {
        setCollapsedMap(collapsedMap.setAllCollapsedStatus(false));
      },
      "aria-label": "Expand all groups",
      icon: "angle-double-down",
      disabled: selectedView === SelectedView.TopTable
    }
  ), /* @__PURE__ */ React__default["default"].createElement(
    ui.Button,
    {
      variant: "secondary",
      fill: "outline",
      size: "sm",
      tooltip: "Collapse all groups",
      onClick: () => {
        setCollapsedMap(collapsedMap.setAllCollapsedStatus(true));
      },
      "aria-label": "Collapse all groups",
      icon: "angle-double-up",
      disabled: selectedView === SelectedView.TopTable
    }
  )), /* @__PURE__ */ React__default["default"].createElement(
    ui.RadioButtonGroup,
    {
      size: "sm",
      disabled: selectedView === SelectedView.TopTable,
      options: alignOptions,
      value: textAlign,
      onChange: onTextAlignChange,
      className: styles.buttonSpacing
    }
  ), /* @__PURE__ */ React__default["default"].createElement(
    ui.RadioButtonGroup,
    {
      size: "sm",
      options: getViewOptions(containerWidth, vertical),
      value: selectedView,
      onChange: setSelectedView
    }
  ), extraHeaderElements && /* @__PURE__ */ React__default["default"].createElement("div", { className: styles.extraElements }, extraHeaderElements)));
};
function ColorSchemeButton(props) {
  const styles = ui.useStyles2(getStyles$2);
  let menu = /* @__PURE__ */ React__default["default"].createElement(ui.Menu, null, /* @__PURE__ */ React__default["default"].createElement(ui.Menu.Item, { label: "By package name", onClick: () => props.onChange(ColorScheme.PackageBased) }), /* @__PURE__ */ React__default["default"].createElement(ui.Menu.Item, { label: "By value", onClick: () => props.onChange(ColorScheme.ValueBased) }));
  const colorDotStyle = {
    [ColorScheme.ValueBased]: styles.colorDotByValue,
    [ColorScheme.PackageBased]: styles.colorDotByPackage,
    [ColorSchemeDiff.DiffColorBlind]: styles.colorDotDiffColorBlind,
    [ColorSchemeDiff.Default]: styles.colorDotDiffDefault
  }[props.value] || styles.colorDotByValue;
  let contents = /* @__PURE__ */ React__default["default"].createElement("span", { className: css.cx(styles.colorDot, colorDotStyle) });
  if (props.isDiffMode) {
    menu = /* @__PURE__ */ React__default["default"].createElement(ui.Menu, null, /* @__PURE__ */ React__default["default"].createElement(ui.Menu.Item, { label: "Default (green to red)", onClick: () => props.onChange(ColorSchemeDiff.Default) }), /* @__PURE__ */ React__default["default"].createElement(ui.Menu.Item, { label: "Color blind (blue to red)", onClick: () => props.onChange(ColorSchemeDiff.DiffColorBlind) }));
    contents = /* @__PURE__ */ React__default["default"].createElement("div", { className: css.cx(styles.colorDotDiff, colorDotStyle) }, /* @__PURE__ */ React__default["default"].createElement("div", null, "-100% (removed)"), /* @__PURE__ */ React__default["default"].createElement("div", null, "0%"), /* @__PURE__ */ React__default["default"].createElement("div", null, "+100% (added)"));
  }
  return /* @__PURE__ */ React__default["default"].createElement(ui.Dropdown, { overlay: menu }, /* @__PURE__ */ React__default["default"].createElement(
    ui.Button,
    {
      variant: "secondary",
      fill: "outline",
      size: "sm",
      tooltip: "Change color scheme",
      onClick: () => {
      },
      className: styles.buttonSpacing,
      "aria-label": "Change color scheme"
    },
    contents
  ));
}
const alignOptions = [
  { value: "left", description: "Align text left", icon: "align-left" },
  { value: "right", description: "Align text right", icon: "align-right" }
];
function getViewOptions(width, vertical) {
  let viewOptions = [
    { value: SelectedView.TopTable, label: "Top Table", description: "Only show top table" },
    { value: SelectedView.FlameGraph, label: "Flame Graph", description: "Only show flame graph" }
  ];
  if (width >= MIN_WIDTH_TO_SHOW_BOTH_TOPTABLE_AND_FLAMEGRAPH || vertical) {
    viewOptions.push({
      value: SelectedView.Both,
      label: "Both",
      description: "Show both the top table and flame graph"
    });
  }
  return viewOptions;
}
function useSearchInput(search, setSearch) {
  const [localSearchState, setLocalSearchState] = React.useState(search);
  const prevSearch = usePrevious__default["default"](search);
  useDebounce__default["default"](
    () => {
      setSearch(localSearchState);
    },
    250,
    [localSearchState]
  );
  React.useEffect(() => {
    if (prevSearch !== search && search !== localSearchState) {
      setLocalSearchState(search);
    }
  }, [search, prevSearch, localSearchState]);
  return [localSearchState, setLocalSearchState];
}
const getStyles$2 = (theme) => ({
  header: css.css({
    label: "header",
    display: "flex",
    flexWrap: "wrap",
    justifyContent: "space-between",
    width: "100%",
    top: 0,
    gap: theme.spacing(1),
    marginTop: theme.spacing(1)
  }),
  stickyHeader: css.css({
    zIndex: theme.zIndex.navbarFixed,
    position: "sticky",
    background: theme.colors.background.primary
  }),
  inputContainer: css.css({
    label: "inputContainer",
    flexGrow: 1,
    minWidth: "150px",
    maxWidth: "350px"
  }),
  rightContainer: css.css({
    label: "rightContainer",
    display: "flex",
    alignItems: "flex-start",
    flexWrap: "wrap"
  }),
  buttonSpacing: css.css({
    label: "buttonSpacing",
    marginRight: theme.spacing(1)
  }),
  resetButton: css.css({
    label: "resetButton",
    display: "flex",
    marginRight: theme.spacing(2)
  }),
  resetButtonIconWrapper: css.css({
    label: "resetButtonIcon",
    padding: "0 5px",
    color: theme.colors.text.disabled
  }),
  colorDot: css.css({
    label: "colorDot",
    display: "inline-block",
    width: "10px",
    height: "10px",
    // eslint-disable-next-line @grafana/no-border-radius-literal
    borderRadius: "50%"
  }),
  colorDotDiff: css.css({
    label: "colorDotDiff",
    display: "flex",
    width: "200px",
    height: "12px",
    color: "white",
    fontSize: 9,
    lineHeight: 1.3,
    fontWeight: 300,
    justifyContent: "space-between",
    padding: "0 2px",
    // We have a specific sizing for this so probably makes sense to use hardcoded value here
    // eslint-disable-next-line @grafana/no-border-radius-literal
    borderRadius: "2px"
  }),
  colorDotByValue: css.css({
    label: "colorDotByValue",
    background: byValueGradient
  }),
  colorDotByPackage: css.css({
    label: "colorDotByPackage",
    background: byPackageGradient
  }),
  colorDotDiffDefault: css.css({
    label: "colorDotDiffDefault",
    background: diffDefaultGradient
  }),
  colorDotDiffColorBlind: css.css({
    label: "colorDotDiffColorBlind",
    background: diffColorBlindGradient
  }),
  extraElements: css.css({
    label: "extraElements",
    marginLeft: theme.spacing(1)
  })
});

const FlameGraphTopTableContainer = React__default["default"].memo(
  ({
    data,
    onSymbolClick,
    search,
    matchedLabels,
    onSearch,
    sandwichItem,
    onSandwich,
    onTableSort,
    colorScheme
  }) => {
    const table = React.useMemo(() => {
      let filteredTable = {};
      for (let i = 0; i < data.data.length; i++) {
        const value = data.getValue(i);
        const valueRight = data.getValueRight(i);
        const self = data.getSelf(i);
        const label = data.getLabel(i);
        if (!matchedLabels || matchedLabels.has(label)) {
          filteredTable[label] = filteredTable[label] || {};
          filteredTable[label].self = filteredTable[label].self ? filteredTable[label].self + self : self;
          filteredTable[label].total = filteredTable[label].total ? filteredTable[label].total + value : value;
          filteredTable[label].totalRight = filteredTable[label].totalRight ? filteredTable[label].totalRight + valueRight : valueRight;
        }
      }
      return filteredTable;
    }, [data, matchedLabels]);
    const styles = ui.useStyles2(getStyles$1);
    const theme = ui.useTheme2();
    const [sort, setSort] = React.useState([{ displayName: "Self", desc: true }]);
    return /* @__PURE__ */ React__default["default"].createElement("div", { className: styles.topTableContainer, "data-testid": "topTable" }, /* @__PURE__ */ React__default["default"].createElement(AutoSizer__default["default"], { style: { width: "100%" } }, ({ width, height }) => {
      if (width < 3 || height < 3) {
        return null;
      }
      const frame = buildTableDataFrame(
        data,
        table,
        width,
        onSymbolClick,
        onSearch,
        onSandwich,
        theme,
        colorScheme,
        search,
        sandwichItem
      );
      return /* @__PURE__ */ React__default["default"].createElement(
        ui.Table,
        {
          initialSortBy: sort,
          onSortByChange: (s) => {
            if (s && s.length) {
              onTableSort == null ? void 0 : onTableSort(s[0].displayName + "_" + (s[0].desc ? "desc" : "asc"));
            }
            setSort(s);
          },
          data: frame,
          width,
          height
        }
      );
    }));
  }
);
FlameGraphTopTableContainer.displayName = "FlameGraphTopTableContainer";
function buildTableDataFrame(data, table, width, onSymbolClick, onSearch, onSandwich, theme, colorScheme, search, sandwichItem) {
  const actionField = createActionField(onSandwich, onSearch, search, sandwichItem);
  const symbolField = {
    type: data$1.FieldType.string,
    name: "Symbol",
    values: [],
    config: {
      custom: { width: width - actionColumnWidth - TOP_TABLE_COLUMN_WIDTH * 2 },
      links: [
        {
          title: "Highlight symbol",
          url: "",
          onClick: (e) => {
            const field = e.origin.field;
            const value = field.values[e.origin.rowIndex];
            onSymbolClick(value);
          }
        }
      ]
    }
  };
  let frame;
  if (data.isDiffFlamegraph()) {
    symbolField.config.custom.width = width - actionColumnWidth - TOP_TABLE_COLUMN_WIDTH * 3;
    const baselineField = createNumberField("Baseline", "percent");
    const comparisonField = createNumberField("Comparison", "percent");
    const diffField = createNumberField("Diff", "percent");
    diffField.config.custom.cellOptions.type = ui.TableCellDisplayMode.ColorText;
    const [removeColor, addColor] = colorScheme === ColorSchemeDiff.DiffColorBlind ? [diffColorBlindColors[0], diffColorBlindColors[2]] : [diffDefaultColors[0], diffDefaultColors[2]];
    diffField.config.mappings = [
      { type: data$1.MappingType.ValueToText, options: { [Infinity]: { text: "new", color: addColor } } },
      { type: data$1.MappingType.ValueToText, options: { [-100]: { text: "removed", color: removeColor } } },
      { type: data$1.MappingType.RangeToText, options: { from: 0, to: Infinity, result: { color: addColor } } },
      { type: data$1.MappingType.RangeToText, options: { from: -Infinity, to: 0, result: { color: removeColor } } }
    ];
    const levels = data.getLevels();
    const totalTicks = levels.length ? levels[0][0].value : 0;
    const totalTicksRight = levels.length ? levels[0][0].valueRight : void 0;
    for (let key in table) {
      actionField.values.push(null);
      symbolField.values.push(key);
      const ticksLeft = table[key].total;
      const ticksRight = table[key].totalRight;
      const totalTicksLeft = totalTicks - totalTicksRight;
      const percentageLeft = Math.round(1e4 * ticksLeft / totalTicksLeft) / 100;
      const percentageRight = Math.round(1e4 * ticksRight / totalTicksRight) / 100;
      const diff = (percentageRight - percentageLeft) / percentageLeft * 100;
      diffField.values.push(diff);
      baselineField.values.push(percentageLeft);
      comparisonField.values.push(percentageRight);
    }
    frame = {
      fields: [actionField, symbolField, baselineField, comparisonField, diffField],
      length: symbolField.values.length
    };
  } else {
    const selfField = createNumberField("Self", data.selfField.config.unit);
    const totalField = createNumberField("Total", data.valueField.config.unit);
    for (let key in table) {
      actionField.values.push(null);
      symbolField.values.push(key);
      selfField.values.push(table[key].self);
      totalField.values.push(table[key].total);
    }
    frame = { fields: [actionField, symbolField, selfField, totalField], length: symbolField.values.length };
  }
  const dataFrames = data$1.applyFieldOverrides({
    data: [frame],
    fieldConfig: {
      defaults: {},
      overrides: []
    },
    replaceVariables: (value) => value,
    theme
  });
  return dataFrames[0];
}
function createNumberField(name, unit) {
  const tableFieldOptions = {
    width: TOP_TABLE_COLUMN_WIDTH,
    align: "auto",
    inspect: false,
    cellOptions: { type: ui.TableCellDisplayMode.Auto }
  };
  return {
    type: data$1.FieldType.number,
    name,
    values: [],
    config: {
      unit,
      custom: tableFieldOptions
    }
  };
}
const actionColumnWidth = 61;
function createActionField(onSandwich, onSearch, search, sandwichItem) {
  const options = {
    type: ui.TableCellDisplayMode.Custom,
    cellComponent: (props) => {
      return /* @__PURE__ */ React__default["default"].createElement(
        ActionCell,
        {
          frame: props.frame,
          onSandwich,
          onSearch,
          search,
          sandwichItem,
          rowIndex: props.rowIndex
        }
      );
    }
  };
  const actionFieldTableConfig = {
    filterable: false,
    width: actionColumnWidth,
    hideHeader: true,
    inspect: false,
    align: "auto",
    cellOptions: options
  };
  return {
    type: data$1.FieldType.number,
    name: "actions",
    values: [],
    config: {
      custom: actionFieldTableConfig
    }
  };
}
function ActionCell(props) {
  var _a;
  const styles = getStylesActionCell();
  const symbol = (_a = props.frame.fields.find((f) => f.name === "Symbol")) == null ? void 0 : _a.values[props.rowIndex];
  const isSearched = props.search === symbol;
  const isSandwiched = props.sandwichItem === symbol;
  return /* @__PURE__ */ React__default["default"].createElement("div", { className: styles.actionCellWrapper }, /* @__PURE__ */ React__default["default"].createElement(
    ui.IconButton,
    {
      className: styles.actionCellButton,
      name: "search",
      variant: isSearched ? "primary" : "secondary",
      tooltip: isSearched ? "Clear from search" : "Search for symbol",
      "aria-label": isSearched ? "Clear from search" : "Search for symbol",
      onClick: () => {
        props.onSearch(isSearched ? "" : symbol);
      }
    }
  ), /* @__PURE__ */ React__default["default"].createElement(
    ui.IconButton,
    {
      className: styles.actionCellButton,
      name: "gf-show-context",
      tooltip: isSandwiched ? "Remove from sandwich view" : "Show in sandwich view",
      variant: isSandwiched ? "primary" : "secondary",
      "aria-label": isSandwiched ? "Remove from sandwich view" : "Show in sandwich view",
      onClick: () => {
        props.onSandwich(isSandwiched ? void 0 : symbol);
      }
    }
  ));
}
const getStyles$1 = (theme) => {
  return {
    topTableContainer: css.css({
      label: "topTableContainer",
      padding: theme.spacing(1),
      backgroundColor: theme.colors.background.secondary,
      height: "100%"
    })
  };
};
const getStylesActionCell = () => {
  return {
    actionCellWrapper: css.css({
      label: "actionCellWrapper",
      display: "flex",
      height: "24px"
    }),
    actionCellButton: css.css({
      label: "actionCellButton",
      marginRight: 0,
      width: "24px"
    })
  };
};

const ufuzzy = new uFuzzy__default["default"]();
const FlameGraphContainer = ({
  data,
  onTableSymbolClick,
  onViewSelected,
  onTextAlignSelected,
  onTableSort,
  getTheme,
  stickyHeader,
  extraHeaderElements,
  vertical,
  showFlameGraphOnly,
  disableCollapsing,
  getExtraContextMenuButtons
}) => {
  const [focusedItemData, setFocusedItemData] = React.useState();
  const [rangeMin, setRangeMin] = React.useState(0);
  const [rangeMax, setRangeMax] = React.useState(1);
  const [search, setSearch] = React.useState("");
  const [selectedView, setSelectedView] = React.useState(SelectedView.Both);
  const [sizeRef, { width: containerWidth }] = reactUse.useMeasure();
  const [textAlign, setTextAlign] = React.useState("left");
  const [sandwichItem, setSandwichItem] = React.useState();
  const [collapsedMap, setCollapsedMap] = React.useState(new CollapsedMap());
  const theme = React.useMemo(() => getTheme(), [getTheme]);
  const dataContainer = React.useMemo(() => {
    if (!data) {
      return;
    }
    const container = new FlameGraphDataContainer(data, { collapsing: !disableCollapsing }, theme);
    setCollapsedMap(container.getCollapsedMap());
    return container;
  }, [data, theme, disableCollapsing]);
  const [colorScheme, setColorScheme] = useColorScheme(dataContainer);
  const styles = getStyles(theme);
  const matchedLabels = useLabelSearch(search, dataContainer);
  React.useEffect(() => {
    if (containerWidth > 0 && containerWidth < MIN_WIDTH_TO_SHOW_BOTH_TOPTABLE_AND_FLAMEGRAPH && selectedView === SelectedView.Both && !vertical) {
      setSelectedView(SelectedView.FlameGraph);
    }
  }, [selectedView, setSelectedView, containerWidth, vertical]);
  const resetFocus = React.useCallback(() => {
    setFocusedItemData(void 0);
    setRangeMin(0);
    setRangeMax(1);
  }, [setFocusedItemData, setRangeMax, setRangeMin]);
  function resetSandwich() {
    setSandwichItem(void 0);
  }
  React.useEffect(() => {
    resetFocus();
    resetSandwich();
  }, [data, resetFocus]);
  const onSymbolClick = React.useCallback(
    (symbol) => {
      if (search === symbol) {
        setSearch("");
      } else {
        onTableSymbolClick == null ? void 0 : onTableSymbolClick(symbol);
        setSearch(symbol);
        resetFocus();
      }
    },
    [setSearch, resetFocus, onTableSymbolClick, search]
  );
  if (!dataContainer) {
    return null;
  }
  const flameGraph = /* @__PURE__ */ React__default["default"].createElement(
    FlameGraph,
    {
      data: dataContainer,
      rangeMin,
      rangeMax,
      matchedLabels,
      setRangeMin,
      setRangeMax,
      onItemFocused: (data2) => setFocusedItemData(data2),
      focusedItemData,
      textAlign,
      sandwichItem,
      onSandwich: (label) => {
        resetFocus();
        setSandwichItem(label);
      },
      onFocusPillClick: resetFocus,
      onSandwichPillClick: resetSandwich,
      colorScheme,
      showFlameGraphOnly,
      collapsing: !disableCollapsing,
      getExtraContextMenuButtons,
      selectedView,
      search,
      collapsedMap,
      setCollapsedMap
    }
  );
  const table = /* @__PURE__ */ React__default["default"].createElement(
    FlameGraphTopTableContainer,
    {
      data: dataContainer,
      onSymbolClick,
      search,
      matchedLabels,
      sandwichItem,
      onSandwich: setSandwichItem,
      onSearch: setSearch,
      onTableSort,
      colorScheme
    }
  );
  let body;
  if (showFlameGraphOnly || selectedView === SelectedView.FlameGraph) {
    body = flameGraph;
  } else if (selectedView === SelectedView.TopTable) {
    body = /* @__PURE__ */ React__default["default"].createElement("div", { className: styles.tableContainer }, table);
  } else if (selectedView === SelectedView.Both) {
    if (vertical) {
      body = /* @__PURE__ */ React__default["default"].createElement("div", null, /* @__PURE__ */ React__default["default"].createElement("div", { className: styles.verticalGraphContainer }, flameGraph), /* @__PURE__ */ React__default["default"].createElement("div", { className: styles.verticalTableContainer }, table));
    } else {
      body = /* @__PURE__ */ React__default["default"].createElement("div", { className: styles.horizontalContainer }, /* @__PURE__ */ React__default["default"].createElement("div", { className: styles.horizontalTableContainer }, table), /* @__PURE__ */ React__default["default"].createElement("div", { className: styles.horizontalGraphContainer }, flameGraph));
    }
  }
  return (
    // We add the theme context to bridge the gap if this is rendered in non grafana environment where the context
    // isn't already provided.
    /* @__PURE__ */ React__default["default"].createElement(ui.ThemeContext.Provider, { value: theme }, /* @__PURE__ */ React__default["default"].createElement("div", { ref: sizeRef, className: styles.container }, !showFlameGraphOnly && /* @__PURE__ */ React__default["default"].createElement(
      FlameGraphHeader,
      {
        search,
        setSearch,
        selectedView,
        setSelectedView: (view) => {
          setSelectedView(view);
          onViewSelected == null ? void 0 : onViewSelected(view);
        },
        containerWidth,
        onReset: () => {
          resetFocus();
          resetSandwich();
        },
        textAlign,
        onTextAlignChange: (align) => {
          setTextAlign(align);
          onTextAlignSelected == null ? void 0 : onTextAlignSelected(align);
        },
        showResetButton: Boolean(focusedItemData || sandwichItem),
        colorScheme,
        onColorSchemeChange: setColorScheme,
        stickyHeader: Boolean(stickyHeader),
        extraHeaderElements,
        vertical,
        isDiffMode: dataContainer.isDiffFlamegraph(),
        setCollapsedMap,
        collapsedMap
      }
    ), /* @__PURE__ */ React__default["default"].createElement("div", { className: styles.body }, body)))
  );
};
function useColorScheme(dataContainer) {
  const defaultColorScheme = (dataContainer == null ? void 0 : dataContainer.isDiffFlamegraph()) ? ColorSchemeDiff.Default : ColorScheme.PackageBased;
  const [colorScheme, setColorScheme] = React.useState(defaultColorScheme);
  React.useEffect(() => {
    setColorScheme(defaultColorScheme);
  }, [defaultColorScheme]);
  return [colorScheme, setColorScheme];
}
function useLabelSearch(search, data) {
  return React.useMemo(() => {
    if (search && data) {
      const foundLabels = /* @__PURE__ */ new Set();
      let idxs = ufuzzy.filter(data.getUniqueLabels(), search);
      if (idxs) {
        for (let idx of idxs) {
          foundLabels.add(data.getUniqueLabels()[idx]);
        }
      }
      return foundLabels;
    }
    return void 0;
  }, [search, data]);
}
function getStyles(theme) {
  return {
    container: css.css({
      label: "container",
      overflow: "auto",
      height: "100%",
      display: "flex",
      flex: "1 1 0",
      flexDirection: "column",
      minHeight: 0,
      gap: theme.spacing(1)
    }),
    body: css.css({
      label: "body",
      flexGrow: 1
    }),
    tableContainer: css.css({
      // This is not ideal for dashboard panel where it creates a double scroll. In a panel it should be 100% but then
      // in explore we need a specific height.
      height: 800
    }),
    horizontalContainer: css.css({
      label: "horizontalContainer",
      display: "flex",
      minHeight: 0,
      flexDirection: "row",
      columnGap: theme.spacing(1),
      width: "100%"
    }),
    horizontalGraphContainer: css.css({
      flexBasis: "50%"
    }),
    horizontalTableContainer: css.css({
      flexBasis: "50%",
      maxHeight: 800
    }),
    verticalGraphContainer: css.css({
      marginBottom: theme.spacing(1)
    }),
    verticalTableContainer: css.css({
      height: 800
    })
  };
}

const data = {
  name: "response",
  refId: "A",
  // @ts-ignore
  meta: { preferredVisualisationType: "flamegraph" },
  fields: [
    {
      name: "level",
      values: [
        0,
        1,
        2,
        3,
        4,
        5,
        6,
        7,
        8,
        9,
        10,
        11,
        12,
        11,
        12,
        13,
        14,
        15,
        16,
        10,
        11,
        12,
        13,
        9,
        10,
        9,
        9,
        7,
        7,
        6,
        5,
        6,
        7,
        8,
        8,
        9,
        10,
        11,
        12,
        13,
        8,
        9,
        10,
        6,
        7,
        8,
        9,
        10,
        11,
        12,
        13,
        14,
        15,
        4,
        5,
        6,
        7,
        8,
        9,
        9,
        10,
        11,
        12,
        11,
        12,
        13,
        8,
        8,
        7,
        8,
        8,
        7,
        6,
        7,
        8,
        9,
        9,
        10,
        11,
        12,
        13,
        14,
        15,
        16,
        17,
        18,
        9,
        8,
        8,
        7,
        8,
        6,
        7,
        8,
        8,
        7,
        6,
        7,
        8,
        8,
        8,
        9,
        10,
        11,
        12,
        4,
        5,
        6,
        7,
        8,
        9,
        9,
        10,
        11,
        9,
        10,
        7,
        4,
        5,
        6,
        7,
        8,
        8,
        7,
        8,
        9,
        10,
        11,
        12,
        13,
        10,
        10,
        11,
        9,
        10,
        10,
        11,
        12,
        13,
        14,
        15,
        16,
        17,
        8,
        9,
        5,
        6,
        7,
        6,
        7,
        6,
        5,
        6,
        6,
        6,
        6,
        5,
        6,
        7,
        6,
        5,
        6,
        7,
        8,
        4,
        5,
        6,
        5,
        6,
        5,
        4,
        5,
        5,
        3,
        4,
        5,
        6,
        7,
        7,
        8,
        7,
        8,
        9,
        8,
        9,
        8,
        7,
        8,
        9,
        1,
        2,
        3,
        4,
        5,
        6,
        7,
        8,
        9,
        10,
        11,
        12,
        13,
        14,
        15,
        16,
        17,
        18,
        19,
        20,
        21,
        22,
        23,
        24,
        24,
        24,
        25,
        26,
        24,
        25,
        22,
        23,
        23,
        24,
        25,
        26,
        27,
        28,
        29,
        30,
        31,
        32,
        21,
        22,
        23,
        24,
        25,
        26,
        27,
        26,
        24,
        25,
        26,
        25,
        24,
        24,
        25,
        26,
        27,
        28,
        29,
        30,
        31,
        32,
        33,
        34,
        35,
        25,
        26,
        27,
        25,
        26,
        24,
        23,
        24,
        24,
        25,
        23,
        23,
        22,
        23,
        24,
        24,
        23,
        24,
        25,
        26,
        24,
        25,
        23,
        24,
        24,
        25,
        23,
        23,
        24,
        24,
        23,
        23,
        23,
        23,
        23,
        24,
        25,
        22,
        23,
        24,
        25,
        22,
        23,
        24,
        25,
        26,
        27,
        28,
        25,
        26,
        27,
        28,
        29,
        24,
        22,
        23,
        24,
        25,
        26,
        27,
        28,
        29,
        30,
        31,
        32,
        33,
        21,
        22,
        23,
        23,
        24,
        24,
        24,
        22,
        23,
        24,
        25,
        24,
        23,
        24,
        24,
        23,
        22,
        23,
        24,
        25,
        25,
        26,
        27,
        24,
        22,
        23,
        24,
        25,
        26,
        27,
        28,
        29,
        30,
        31,
        32,
        24,
        21,
        22,
        23,
        24,
        25,
        26,
        27,
        28,
        21,
        22,
        19,
        20,
        21,
        22,
        23,
        24,
        25,
        26,
        27,
        28,
        29,
        30,
        31,
        32,
        33,
        34,
        35,
        24,
        25,
        26,
        27,
        28,
        23,
        24,
        25,
        26,
        27,
        28,
        19,
        20,
        21,
        22,
        23,
        24,
        25,
        26,
        27,
        28,
        29,
        30,
        31,
        32,
        33,
        34,
        17,
        18,
        19,
        20,
        21,
        22,
        23,
        24,
        25,
        26,
        27,
        28,
        29,
        30,
        31,
        32,
        33,
        34,
        35,
        36,
        37,
        38,
        39,
        40,
        41,
        42,
        43,
        44,
        45,
        46,
        47,
        48,
        49,
        31,
        30,
        28,
        29,
        30,
        31,
        32,
        33,
        34,
        35,
        36,
        37,
        29,
        30,
        31,
        26,
        27,
        28,
        28,
        28,
        29,
        27,
        26,
        27,
        26,
        25,
        26,
        27,
        28,
        29,
        30,
        31,
        32,
        32,
        33,
        34,
        30,
        31,
        32,
        33,
        34,
        33,
        34,
        35,
        36,
        37,
        38,
        39,
        29,
        30,
        31,
        32,
        32,
        27,
        28,
        28,
        27,
        28,
        29,
        30,
        31,
        32,
        33,
        34,
        35,
        36,
        37,
        38,
        39,
        40,
        27,
        28,
        27,
        28,
        29,
        30,
        31,
        32,
        33,
        34,
        35,
        36,
        28,
        27,
        28,
        28,
        29,
        27,
        27,
        28,
        27,
        27,
        26,
        27,
        28,
        28,
        29,
        29,
        29,
        26,
        27,
        27,
        28,
        29,
        30,
        29,
        30,
        30,
        29,
        30,
        31,
        28,
        29,
        30,
        28,
        28,
        28,
        28,
        29,
        30,
        29,
        30,
        27,
        28,
        28,
        26,
        27,
        26,
        27,
        27,
        28,
        29,
        26,
        27,
        28,
        29,
        28,
        27,
        28,
        29,
        30,
        30,
        28,
        29,
        30,
        31,
        32,
        32,
        27,
        28,
        26,
        25,
        25,
        26,
        27,
        28,
        29,
        30,
        29,
        30,
        31,
        32,
        33,
        34,
        35,
        36,
        37,
        38,
        39,
        40,
        41,
        42,
        43,
        44,
        45,
        46,
        47,
        48,
        31,
        30,
        31,
        32,
        30,
        31,
        32,
        33,
        34,
        35,
        36,
        37,
        38,
        39,
        40,
        41,
        42,
        43,
        44,
        45,
        46,
        47,
        48,
        29,
        26,
        27,
        28,
        29,
        30,
        30,
        31,
        32,
        33,
        34,
        35,
        36,
        37,
        38,
        39,
        40,
        41,
        42,
        43,
        44,
        45,
        31,
        30,
        29,
        29,
        29,
        25,
        26,
        27,
        28,
        26,
        26,
        27,
        28,
        27,
        28,
        28,
        26,
        27,
        28,
        28,
        27,
        28,
        25,
        26,
        27,
        28,
        29,
        30,
        31,
        32,
        33,
        34,
        25,
        25,
        26,
        27,
        28,
        25,
        26,
        24,
        25,
        22,
        23,
        24,
        25,
        26,
        27,
        28,
        29,
        30,
        31,
        32,
        33,
        33,
        34,
        34,
        32,
        33,
        34,
        34,
        32,
        28,
        29,
        30,
        31,
        27,
        28,
        29,
        26,
        27,
        28,
        29,
        30,
        31,
        32,
        33,
        34,
        26,
        27,
        28,
        29,
        24,
        25,
        26,
        27,
        28,
        29,
        30,
        27,
        28,
        26,
        27,
        25,
        26,
        27,
        28,
        29,
        26,
        25,
        26,
        27,
        26,
        27,
        27,
        28,
        29,
        30,
        31,
        32,
        33,
        34,
        35,
        36,
        37,
        38,
        38,
        39,
        30,
        31,
        32,
        26,
        27,
        28,
        27,
        28,
        29,
        25,
        26,
        27,
        28,
        29,
        30,
        31,
        32,
        28,
        29,
        28,
        29,
        28,
        29,
        27,
        28,
        29,
        30,
        31,
        29,
        30,
        31,
        32,
        33,
        34,
        35,
        27,
        28,
        29,
        30,
        31,
        32,
        33,
        34,
        35,
        36,
        37,
        38,
        28,
        29,
        30,
        31,
        28,
        29,
        30,
        31,
        26,
        25,
        26,
        27,
        28,
        29,
        27,
        28,
        29,
        30,
        25,
        26,
        15,
        14,
        15,
        16,
        17,
        13,
        14,
        15,
        16,
        15,
        16,
        17,
        18,
        19,
        20,
        12,
        13,
        14,
        15,
        16,
        17,
        10,
        11,
        12,
        13,
        14,
        12,
        13,
        10,
        11,
        12,
        13,
        14,
        15,
        2,
        3,
        4,
        5,
        6,
        7,
        8,
        4,
        5,
        6,
        7,
        8,
        9,
        2,
        3,
        1,
        2,
        3,
        4,
        5,
        6,
        6,
        6,
        6,
        7,
        8,
        8,
        9,
        6,
        7,
        6,
        6,
        6,
        6,
        5,
        5,
        6,
        5,
        5,
        5,
        5,
        5,
        6,
        7,
        4,
        2,
        3,
        4,
        5,
        6,
        4,
        5,
        6,
        7,
        8,
        9,
        1,
        2,
        3,
        4,
        5,
        6,
        7,
        8,
        2,
        3,
        4,
        5,
        6,
        7,
        8,
        9,
        2,
        3,
        4,
        5,
        6,
        1,
        2,
        3,
        3,
        4,
        3,
        2,
        1,
        2,
        3,
        4,
        5,
        6,
        5,
        6,
        7,
        8,
        5,
        5,
        6,
        7,
        4,
        5,
        6,
        4,
        2,
        3,
        4,
        3,
        4,
        5,
        6,
        5,
        5,
        6,
        4,
        1,
        2,
        3,
        4,
        5,
        6,
        2,
        3,
        4,
        5,
        6,
        7,
        8,
        9,
        10,
        11,
        12,
        13,
        2,
        3,
        4,
        5,
        6,
        7,
        8,
        9,
        10,
        11,
        1,
        2,
        3,
        4,
        5,
        6,
        7,
        8,
        9,
        10,
        11,
        12,
        13,
        14,
        15,
        16,
        17,
        18,
        19,
        20,
        21,
        22,
        23,
        16,
        17,
        18,
        19,
        16,
        17,
        18,
        19,
        20,
        21,
        22,
        23,
        8,
        9,
        10,
        11,
        12,
        13,
        14,
        15,
        16,
        17,
        18,
        19,
        20,
        21,
        22,
        23,
        8,
        9,
        10,
        11,
        12,
        13,
        14,
        1,
        2,
        3,
        4,
        5,
        6,
        7,
        8,
        9,
        10,
        11,
        12,
        1,
        2,
        3,
        4
      ]
    },
    {
      name: "value",
      values: [
        1646e7,
        41e8,
        41e8,
        367e7,
        113e7,
        106e7,
        102e7,
        15e7,
        15e7,
        5e7,
        3e7,
        2e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        8e7,
        4e7,
        1e7,
        1e7,
        53e7,
        8e7,
        4e7,
        7e7,
        5e7,
        5e7,
        3e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        2e7,
        2e7,
        2e7,
        2e7,
        2e7,
        2e7,
        2e7,
        2e7,
        2e7,
        1e7,
        39e7,
        39e7,
        18e7,
        7e7,
        5e7,
        1e7,
        2e7,
        2e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        2e7,
        1e7,
        1e7,
        1e7,
        11e7,
        8e7,
        6e7,
        1e7,
        2e7,
        2e7,
        2e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        2e7,
        1e7,
        1e7,
        3e7,
        2e7,
        4e7,
        3e7,
        2e7,
        1e7,
        1e7,
        4e7,
        4e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        21e7,
        21e7,
        21e7,
        13e7,
        13e7,
        1e7,
        5e7,
        4e7,
        4e7,
        7e7,
        1e7,
        4e7,
        184e7,
        24e7,
        24e7,
        5e7,
        1e7,
        1e7,
        19e7,
        13e7,
        5e7,
        2e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        8e7,
        2e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        6e7,
        1e7,
        133e7,
        129e7,
        32e7,
        1e7,
        1e7,
        1e7,
        13e7,
        1e7,
        1e7,
        6e7,
        3e7,
        1e8,
        4e7,
        1e7,
        1e7,
        4e7,
        3e7,
        1e7,
        1e7,
        7e7,
        1e7,
        1e7,
        5e7,
        4e7,
        1e7,
        3e7,
        2e7,
        1e7,
        43e7,
        43e7,
        43e7,
        43e7,
        1e7,
        4e7,
        2e7,
        37e7,
        2e7,
        1e7,
        18e7,
        1e7,
        3e7,
        1e7,
        1e7,
        1e7,
        563e7,
        558e7,
        558e7,
        558e7,
        558e7,
        558e7,
        558e7,
        558e7,
        558e7,
        554e7,
        554e7,
        553e7,
        551e7,
        55e8,
        549e7,
        549e7,
        161e7,
        161e7,
        154e7,
        154e7,
        19e7,
        17e7,
        17e7,
        1e7,
        2e7,
        12e7,
        1e7,
        1e7,
        2e7,
        2e7,
        2e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        85e7,
        35e7,
        3e8,
        12e7,
        11e7,
        8e7,
        2e7,
        1e7,
        8e7,
        4e7,
        2e7,
        1e7,
        1e7,
        5e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        2e7,
        2e7,
        2e7,
        2e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        37e7,
        4e7,
        2e7,
        1e7,
        1e8,
        1e7,
        1e7,
        1e7,
        3e7,
        3e7,
        8e7,
        2e7,
        1e7,
        1e7,
        1e7,
        4e7,
        1e7,
        1e7,
        1e7,
        2e7,
        1e7,
        2e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        11e7,
        11e7,
        6e7,
        2e7,
        2e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        48e7,
        15e7,
        1e7,
        13e7,
        1e7,
        1e7,
        5e7,
        14e7,
        8e7,
        4e7,
        1e7,
        1e7,
        2e7,
        1e7,
        1e7,
        2e7,
        12e7,
        6e7,
        4e7,
        1e7,
        1e7,
        1e7,
        1e7,
        2e7,
        3e7,
        3e7,
        2e7,
        2e7,
        2e7,
        2e7,
        2e7,
        2e7,
        2e7,
        2e7,
        2e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        6e7,
        6e7,
        6e7,
        6e7,
        3e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        2e7,
        2e7,
        2e7,
        2e7,
        2e7,
        3e7,
        3e7,
        3e7,
        3e7,
        3e7,
        3e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        388e7,
        388e7,
        388e7,
        388e7,
        387e7,
        323e7,
        323e7,
        321e7,
        32e7,
        7e7,
        7e7,
        4e7,
        4e7,
        2e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        2e7,
        3e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        2e7,
        2e7,
        2e7,
        7e7,
        6e7,
        2e7,
        3e7,
        1e7,
        1e7,
        1e7,
        1e8,
        2e7,
        5e7,
        185e7,
        41e7,
        11e7,
        11e7,
        9e7,
        6e7,
        6e7,
        4e7,
        2e7,
        2e7,
        2e7,
        3e7,
        3e7,
        3e7,
        1e7,
        1e7,
        2e7,
        2e7,
        2e7,
        2e7,
        2e7,
        2e7,
        1e7,
        2e7,
        2e7,
        2e7,
        1e7,
        1e7,
        3e7,
        2e7,
        1e7,
        4e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        4e7,
        1e7,
        8e7,
        3e7,
        3e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        2e7,
        6e7,
        3e7,
        3e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        16e7,
        12e7,
        2e7,
        1e8,
        5e7,
        1e7,
        1e7,
        64e7,
        1e7,
        55e7,
        41e7,
        8e7,
        6e7,
        26e7,
        2e7,
        17e7,
        6e7,
        3e7,
        2e7,
        2e7,
        2e7,
        1e7,
        1e7,
        1e7,
        2e7,
        5e7,
        4e7,
        1e7,
        1e7,
        1e7,
        7e7,
        3e7,
        4e7,
        29e7,
        9e7,
        3e7,
        1e7,
        1e7,
        1e7,
        1e7,
        21e7,
        2e7,
        1e7,
        1e7,
        1e7,
        12e7,
        9e7,
        9e7,
        6e7,
        1e7,
        3e7,
        3e7,
        3e7,
        3e7,
        1e7,
        2e7,
        5e7,
        1e7,
        1e7,
        1e7,
        28e7,
        15e7,
        15e7,
        15e7,
        6e7,
        3e7,
        8e7,
        5e7,
        2e7,
        2e7,
        2e7,
        2e7,
        2e7,
        2e7,
        2e7,
        2e7,
        2e7,
        2e7,
        2e7,
        2e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        2e7,
        2e7,
        2e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        13e7,
        13e7,
        13e7,
        7e7,
        3e7,
        3e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        4e7,
        1e7,
        59e7,
        2e7,
        2e7,
        1e7,
        4e7,
        15e7,
        3e7,
        3e7,
        12e7,
        1e7,
        8e7,
        37e7,
        26e7,
        4e7,
        15e7,
        11e7,
        9e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        2e7,
        2e7,
        2e7,
        2e7,
        2e7,
        7e7,
        6e7,
        2e7,
        2e7,
        64e7,
        64e7,
        25e7,
        25e7,
        23e7,
        18e7,
        17e7,
        17e7,
        17e7,
        17e7,
        5e7,
        1e7,
        4e7,
        1e7,
        1e7,
        6e7,
        6e7,
        1e7,
        3e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        5e7,
        5e7,
        2e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        39e7,
        6e7,
        4e7,
        3e7,
        2e7,
        2e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        3e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        9e7,
        1e7,
        1e7,
        6e7,
        1e7,
        4e7,
        4e7,
        4e7,
        3e7,
        3e7,
        3e7,
        3e7,
        3e7,
        3e7,
        3e7,
        3e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        2e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        18e7,
        17e7,
        5e7,
        2e7,
        2e7,
        2e7,
        2e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        3e7,
        3e7,
        2e7,
        2e7,
        2e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        9e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        7e7,
        7e7,
        7e7,
        7e7,
        1e7,
        2e7,
        2e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        2e7,
        2e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        2e7,
        2e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        2e7,
        2e7,
        2e7,
        2e7,
        2e7,
        1e7,
        4e7,
        4e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        3e7,
        3e7,
        3e7,
        3e7,
        3e7,
        3e7,
        1e7,
        1e7,
        619e7,
        617e7,
        617e7,
        607e7,
        446e7,
        1e7,
        5e7,
        44e7,
        54e7,
        2e7,
        1e7,
        1e7,
        1e7,
        98e7,
        1e7,
        5e7,
        48e7,
        54e7,
        5e7,
        21e7,
        3e7,
        3e7,
        6e7,
        18e7,
        1e7,
        21e7,
        2e7,
        2e7,
        2e7,
        1e8,
        2e7,
        2e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        4e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        2e7,
        2e7,
        2e7,
        2e7,
        2e7,
        2e7,
        2e7,
        2e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        17e7,
        16e7,
        7e7,
        3e7,
        1e7,
        4e7,
        1e7,
        17e7,
        9e7,
        9e7,
        7e7,
        2e7,
        1e7,
        3e7,
        3e7,
        3e7,
        3e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        8e7,
        1e7,
        1e7,
        7e7,
        6e7,
        1e7,
        1e7,
        2e7,
        2e7,
        2e7,
        1e7,
        8e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        2e7,
        2e7,
        2e7,
        2e7,
        2e7,
        2e7,
        2e7,
        2e7,
        2e7,
        2e7,
        2e7,
        2e7,
        5e7,
        5e7,
        5e7,
        5e7,
        5e7,
        5e7,
        5e7,
        1e7,
        1e7,
        1e7,
        6e7,
        6e7,
        6e7,
        6e7,
        6e7,
        6e7,
        6e7,
        4e7,
        4e7,
        3e7,
        3e7,
        3e7,
        3e7,
        3e7,
        3e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7,
        1e7
      ],
      config: {
        unit: "short"
      }
    },
    {
      name: "self",
      values: [
        16460,
        4100,
        4100,
        3670,
        1130,
        1060,
        1020,
        150,
        150,
        50,
        30,
        20,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        80,
        40,
        10,
        10,
        530,
        80,
        40,
        70,
        50,
        50,
        30,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        20,
        20,
        20,
        20,
        20,
        20,
        20,
        20,
        20,
        10,
        390,
        390,
        180,
        70,
        50,
        10,
        20,
        20,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        20,
        10,
        10,
        10,
        110,
        80,
        60,
        10,
        20,
        20,
        20,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        20,
        10,
        10,
        30,
        20,
        40,
        30,
        20,
        10,
        10,
        40,
        40,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        210,
        210,
        210,
        130,
        130,
        10,
        50,
        40,
        40,
        70,
        10,
        40,
        1840,
        240,
        240,
        50,
        10,
        10,
        190,
        130,
        50,
        20,
        10,
        10,
        10,
        10,
        10,
        10,
        80,
        20,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        60,
        10,
        1330,
        1290,
        320,
        10,
        10,
        10,
        130,
        10,
        10,
        60,
        30,
        100,
        40,
        10,
        10,
        40,
        30,
        10,
        10,
        70,
        10,
        10,
        50,
        40,
        10,
        30,
        20,
        10,
        430,
        430,
        430,
        430,
        10,
        40,
        20,
        370,
        20,
        10,
        180,
        10,
        30,
        10,
        10,
        10,
        5630,
        5580,
        5580,
        5580,
        5580,
        5580,
        5580,
        5580,
        5580,
        5540,
        5540,
        5530,
        5510,
        5500,
        5490,
        5490,
        1610,
        1610,
        1540,
        1540,
        190,
        170,
        170,
        10,
        20,
        120,
        10,
        10,
        20,
        20,
        20,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        850,
        350,
        300,
        120,
        110,
        80,
        20,
        10,
        80,
        40,
        20,
        10,
        10,
        50,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        20,
        20,
        20,
        20,
        10,
        10,
        10,
        10,
        10,
        370,
        40,
        20,
        10,
        100,
        10,
        10,
        10,
        30,
        30,
        80,
        20,
        10,
        10,
        10,
        40,
        10,
        10,
        10,
        20,
        10,
        20,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        110,
        110,
        60,
        20,
        20,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        480,
        150,
        10,
        130,
        10,
        10,
        50,
        140,
        80,
        40,
        10,
        10,
        20,
        10,
        10,
        20,
        120,
        60,
        40,
        10,
        10,
        10,
        10,
        20,
        30,
        30,
        20,
        20,
        20,
        20,
        20,
        20,
        20,
        20,
        20,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        60,
        60,
        60,
        60,
        30,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        20,
        20,
        20,
        20,
        20,
        30,
        30,
        30,
        30,
        30,
        30,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        3880,
        3880,
        3880,
        3880,
        3870,
        3230,
        3230,
        3210,
        320,
        70,
        70,
        40,
        40,
        20,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        20,
        30,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        20,
        20,
        20,
        70,
        60,
        20,
        30,
        10,
        10,
        10,
        100,
        20,
        50,
        1850,
        410,
        110,
        110,
        90,
        60,
        60,
        40,
        20,
        20,
        20,
        30,
        30,
        30,
        10,
        10,
        20,
        20,
        20,
        20,
        20,
        20,
        10,
        20,
        20,
        20,
        10,
        10,
        30,
        20,
        10,
        40,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        40,
        10,
        80,
        30,
        30,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        20,
        60,
        30,
        30,
        10,
        10,
        10,
        10,
        10,
        10,
        160,
        120,
        20,
        100,
        50,
        10,
        10,
        640,
        10,
        550,
        410,
        80,
        60,
        260,
        20,
        170,
        60,
        30,
        20,
        20,
        20,
        10,
        10,
        10,
        20,
        50,
        40,
        10,
        10,
        10,
        70,
        30,
        40,
        290,
        90,
        30,
        10,
        10,
        10,
        10,
        210,
        20,
        10,
        10,
        10,
        120,
        90,
        90,
        60,
        10,
        30,
        30,
        30,
        30,
        10,
        20,
        50,
        10,
        10,
        10,
        280,
        150,
        150,
        150,
        60,
        30,
        80,
        50,
        20,
        20,
        20,
        20,
        20,
        20,
        20,
        20,
        20,
        20,
        20,
        20,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        20,
        20,
        20,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        130,
        130,
        130,
        70,
        30,
        30,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        40,
        10,
        590,
        20,
        20,
        10,
        40,
        150,
        30,
        30,
        120,
        10,
        80,
        370,
        260,
        40,
        150,
        110,
        90,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        20,
        20,
        20,
        20,
        20,
        70,
        60,
        20,
        20,
        640,
        640,
        250,
        250,
        230,
        180,
        170,
        170,
        170,
        170,
        50,
        10,
        40,
        10,
        10,
        60,
        60,
        10,
        30,
        10,
        10,
        10,
        10,
        10,
        50,
        50,
        20,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        390,
        60,
        40,
        30,
        20,
        20,
        10,
        10,
        10,
        10,
        10,
        30,
        10,
        10,
        10,
        10,
        10,
        90,
        10,
        10,
        60,
        10,
        40,
        40,
        40,
        30,
        30,
        30,
        30,
        30,
        30,
        30,
        30,
        10,
        10,
        10,
        10,
        10,
        10,
        20,
        10,
        10,
        10,
        10,
        10,
        180,
        170,
        50,
        20,
        20,
        20,
        20,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        30,
        30,
        20,
        20,
        20,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        90,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        70,
        70,
        70,
        70,
        10,
        20,
        20,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        20,
        20,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        20,
        20,
        10,
        10,
        10,
        10,
        10,
        20,
        20,
        20,
        20,
        20,
        10,
        40,
        40,
        10,
        10,
        10,
        10,
        10,
        30,
        30,
        30,
        30,
        30,
        30,
        10,
        10,
        6190,
        6170,
        6170,
        6070,
        4460,
        10,
        50,
        440,
        540,
        20,
        10,
        10,
        10,
        980,
        10,
        50,
        480,
        540,
        50,
        210,
        30,
        30,
        60,
        180,
        10,
        210,
        20,
        20,
        20,
        100,
        20,
        20,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        40,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        20,
        20,
        20,
        20,
        20,
        20,
        20,
        20,
        10,
        10,
        10,
        10,
        10,
        170,
        160,
        70,
        30,
        10,
        40,
        10,
        170,
        90,
        90,
        70,
        20,
        10,
        30,
        30,
        30,
        30,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        80,
        10,
        10,
        70,
        60,
        10,
        10,
        20,
        20,
        20,
        10,
        80,
        10,
        10,
        10,
        10,
        10,
        20,
        20,
        20,
        20,
        20,
        20,
        20,
        20,
        20,
        20,
        20,
        20,
        50,
        50,
        50,
        50,
        50,
        50,
        50,
        10,
        10,
        10,
        60,
        60,
        60,
        60,
        60,
        60,
        60,
        40,
        40,
        30,
        30,
        30,
        30,
        30,
        30,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10,
        10
      ],
      config: {
        unit: "short"
      }
    },
    {
      name: "label",
      values: [
        "total",
        "test/pkg/agent.(*Target).start.func1",
        "test/pkg/agent.(*Target).scrape",
        "test/pkg/distributor.(*Distributor).Push",
        "compress/gzip.(*Writer).Write",
        "compress/flate.(*compressor).write",
        "compress/flate.(*compressor).deflate",
        "compress/flate.(*compressor).writeBlock",
        "compress/flate.(*huffmanBitWriter).writeBlock",
        "compress/flate.(*huffmanBitWriter).indexTokens",
        "compress/flate.(*huffmanEncoder).generate",
        "compress/flate.(*huffmanEncoder).bitCounts",
        "runtime.memmove",
        "compress/flate.(*huffmanEncoder).assignEncodingAndSize",
        "sort.Sort",
        "sort.quickSort",
        "sort.quickSort",
        "sort.quickSort",
        "sort.insertionSort",
        "compress/flate.(*byFreq).sort",
        "sort.Sort",
        "sort.quickSort",
        "sort.doPivot",
        "compress/flate.(*huffmanBitWriter).writeTokens",
        "compress/flate.(*huffmanBitWriter).writeCode",
        "compress/flate.offsetCode",
        "compress/flate.(*huffmanBitWriter).dynamicSize",
        "compress/flate.(*compressor).findMatch",
        "compress/flate.matchLen",
        "compress/flate.hash4",
        "compress/flate.NewWriter",
        "runtime.newobject",
        "runtime.mallocgc",
        "runtime.memclrNoHeapPointers",
        "runtime.gcAssistAlloc",
        "runtime.systemstack",
        "runtime.gcAssistAlloc.func1",
        "runtime.gcAssistAlloc1",
        "runtime.gcDrainN",
        "runtime.scanobject",
        "runtime.heapBitsSetType",
        "runtime.heapBitsSetTypeGCProg",
        "runtime.runGCProg",
        "compress/flate.(*compressor).initDeflate",
        "runtime.makeslice",
        "runtime.mallocgc",
        "runtime.gcAssistAlloc",
        "runtime.systemstack",
        "runtime.gcAssistAlloc.func1",
        "runtime.gcAssistAlloc1",
        "runtime.gcDrainN",
        "runtime.scanobject",
        "runtime.pageIndexOf",
        "test/pkg/pprof.OpenRaw",
        "test/pkg/gen/google/v1.(*Profile).UnmarshalVT",
        "test/pkg/gen/google/v1.(*Sample).UnmarshalVT",
        "runtime.makeslice",
        "runtime.mallocgc",
        "runtime.memclrNoHeapPointers",
        "runtime.(*mcache).nextFree",
        "runtime.(*mcache).refill",
        "runtime.(*mcentral).uncacheSpan",
        "runtime.(*spanSet).push",
        "runtime.(*mcentral).cacheSpan",
        "runtime.(*mcentral).grow",
        "runtime.(*mheap).alloc",
        "runtime.getMCache",
        "runtime.nextFreeFast",
        "runtime.newobject",
        "runtime.nextFreeFast",
        "runtime.mallocgc",
        "runtime.growslice",
        "test/pkg/gen/google/v1.(*Location).UnmarshalVT",
        "runtime.newobject",
        "runtime.mallocgc",
        "runtime.heapBitsForAddr",
        "runtime.(*mcache).nextFree",
        "runtime.(*mcache).refill",
        "runtime.(*mcentral).cacheSpan",
        "runtime.(*mcentral).grow",
        "runtime.(*mheap).alloc",
        "runtime.systemstack",
        "runtime.(*mheap).alloc.func1",
        "runtime.(*mheap).allocSpan",
        "runtime.(*mheap).allocMSpanLocked",
        "runtime.(*fixalloc).alloc",
        "runtime.heapBitsSetType",
        "runtime.nextFreeFast",
        "runtime.getMCache",
        "runtime.growslice",
        "runtime.mallocgc",
        "runtime.growslice",
        "runtime.mallocgc",
        "runtime.heapBitsSetType",
        "runtime.memclrNoHeapPointers",
        "runtime.memmove",
        "runtime.newobject",
        "runtime.mallocgc",
        "runtime.heapBitsSetType",
        "runtime.memclrNoHeapPointers",
        "runtime.(*mcache).nextFree",
        "runtime.(*mcache).refill",
        "runtime.(*mcentral).uncacheSpan",
        "runtime.(*spanSet).push",
        "runtime.(*headTailIndex).incTail",
        "compress/gzip.(*Writer).Close",
        "compress/flate.(*compressor).close",
        "compress/flate.(*compressor).deflate",
        "compress/flate.(*compressor).writeBlock",
        "compress/flate.(*huffmanBitWriter).writeBlock",
        "compress/flate.offsetCode",
        "compress/flate.(*huffmanBitWriter).indexTokens",
        "compress/flate.(*huffmanEncoder).generate",
        "compress/flate.(*huffmanEncoder).bitCounts",
        "compress/flate.(*huffmanBitWriter).writeTokens",
        "compress/flate.(*huffmanBitWriter).writeCode",
        "compress/flate.(*compressor).findMatch",
        "test/pkg/pprof.(*Profile).Normalize",
        "sort.Sort",
        "sort.quickSort",
        "sort.doPivot",
        "test/pkg/pprof.(*sortedSample).Less",
        "test/pkg/pprof.(*sortedSample).Swap",
        "sort.quickSort",
        "sort.quickSort",
        "sort.quickSort",
        "sort.quickSort",
        "sort.quickSort",
        "sort.insertionSort",
        "test/pkg/pprof.(*sortedSample).Less",
        "sort.doPivot",
        "sort.insertionSort",
        "test/pkg/pprof.(*sortedSample).Swap",
        "sort.doPivot",
        "test/pkg/pprof.(*sortedSample).Less",
        "test/pkg/pprof.(*sortedSample).Swap",
        "runtime.gcWriteBarrierR8",
        "runtime.gcWriteBarrier",
        "runtime.wbBufFlush",
        "runtime.systemstack",
        "runtime.wbBufFlush.func1",
        "runtime.wbBufFlush1",
        "runtime.findObject",
        "sort.doPivot",
        "test/pkg/pprof.(*sortedSample).Less",
        "test/pkg/pprof.(*Profile).clearSampleReferences",
        "test/pkg/slices.RemoveInPlace[...]",
        "test/pkg/pprof.(*Profile).clearSampleReferences.func1",
        "runtime.growslice",
        "runtime.memmove",
        "runtime.memmove",
        "test/pkg/pprof.StacktracesHasher.Hashes",
        "github.com/cespare/xxhash/v2.round",
        "github.com/cespare/xxhash/v2.(*Digest).Sum64",
        "github.com/cespare/xxhash/v2.(*Digest).Write",
        "github.com/cespare/xxhash/v2.u64",
        "github.com/samber/lo.Uniq[...]",
        "runtime.mapaccess2_fast64",
        "runtime.memhash64",
        "runtime.bucketMask",
        "test/pkg/slices.RemoveInPlace[...]",
        "test/pkg/pprof.(*Profile).Normalize.func1",
        "test/pkg/slices.RemoveInPlace[...]",
        "test/pkg/pprof.(*Profile).Normalize.func1.1",
        "test/pkg/gen/google/v1.(*Profile).MarshalToVT",
        "test/pkg/gen/google/v1.(*Profile).MarshalToSizedBufferVT",
        "test/pkg/gen/google/v1.(*Sample).MarshalToSizedBufferVT",
        "test/pkg/gen/google/v1.(*Profile).SizeVT",
        "test/pkg/gen/google/v1.(*Sample).SizeVT",
        "test/pkg/gen/google/v1.encodeVarint",
        "test/pkg/gen/google/v1.(*Profile).SizeVT",
        "test/pkg/gen/google/v1.(*Location).SizeVT",
        "test/pkg/gen/google/v1.(*Sample).SizeVT",
        "io/ioutil.ReadAll",
        "io.ReadAll",
        "compress/gzip.(*Reader).Read",
        "compress/flate.(*decompressor).Read",
        "runtime.memmove",
        "compress/flate.(*dictDecoder).tryWriteCopy",
        "runtime.memmove",
        "compress/flate.(*decompressor).huffmanBlock",
        "compress/flate.(*dictDecoder).writeCopy",
        "runtime.memmove",
        "compress/flate.(*decompressor).huffSym",
        "bytes.(*Reader).ReadByte",
        "compress/flate.(*decompressor).moreBits",
        "compress/flate.(*decompressor).nextBlock",
        "compress/flate.(*decompressor).huffmanBlock",
        "compress/flate.(*decompressor).moreBits",
        "net/http.(*conn).serve",
        "net/http.serverHandler.ServeHTTP",
        "net/http.HandlerFunc.ServeHTTP",
        "test/pkg/util.glob..func1.1",
        "golang.org/x/net/http2/h2c.h2cHandler.ServeHTTP",
        "net/http.HandlerFunc.ServeHTTP",
        "test/pkg/create.(*create).initServer.func2.1",
        "net/http.HandlerFunc.ServeHTTP",
        "github.com/opentracing-contrib/go-stdlib/nethttp.MiddlewareFunc.func5",
        "net/http.HandlerFunc.ServeHTTP",
        "github.com/weaveworks/common/middleware.Log.Wrap.func1",
        "net/http.HandlerFunc.ServeHTTP",
        "github.com/weaveworks/common/middleware.Instrument.Wrap.func1",
        "github.com/felixge/httpsnoop.(*Metrics).CaptureMetrics",
        "github.com/weaveworks/common/middleware.Instrument.Wrap.func1.2",
        "github.com/gorilla/mux.(*Router).ServeHTTP",
        "github.com/bufbuild/connect-go.(*Handler).ServeHTTP",
        "github.com/bufbuild/connect-go.NewUnaryHandler[...].func1",
        "github.com/bufbuild/connect-go.NewUnaryHandler[...].func1.1",
        "test/pkg/ingester.(*Ingester).Push",
        "io.ReadAll",
        "github.com/klauspost/compress/gzip.(*Reader).Read",
        "github.com/klauspost/compress/flate.(*decompressor).Read",
        "github.com/klauspost/compress/flate.(*dictDecoder).tryWriteCopy",
        "github.com/klauspost/compress/flate.(*dictDecoder).writeByte",
        "github.com/klauspost/compress/flate.(*decompressor).huffmanBytesReader",
        "github.com/klauspost/compress/flate.(*dictDecoder).writeCopy",
        "runtime.memmove",
        "github.com/klauspost/compress/flate.(*decompressor).nextBlock",
        "github.com/klauspost/compress/flate.(*decompressor).huffmanBytesReader",
        "runtime.growslice",
        "runtime.memmove",
        "runtime.mallocgc",
        "runtime.(*mcache).allocLarge",
        "runtime.(*mheap).alloc",
        "runtime.systemstack",
        "runtime.(*mheap).alloc.func1",
        "runtime.(*mheap).allocSpan",
        "runtime.(*pageAlloc).alloc",
        "runtime.(*pageAlloc).allocRange",
        "runtime.(*pageAlloc).update",
        "runtime.(*pallocBits).summarize",
        "test/pkg/create.(*Head).Ingest",
        "test/pkg/create.(*Head).convertSamples",
        "test/pkg/create.(*deduplicatingSlice[...]).ingest",
        "test/pkg/create.(*stacktracesHelper).rewrite",
        "test/pkg/create.idConversionTable.rewriteUint64",
        "runtime.mapaccess2_fast64",
        "runtime.memhash64",
        "runtime.add",
        "test/pkg/create.(*stacktracesHelper).key",
        "github.com/cespare/xxhash/v2.(*Digest).Write",
        "runtime.memmove",
        "github.com/cespare/xxhash/v2.round",
        "runtime.mapaccess2_fast64",
        "runtime.mapassign_fast64",
        "runtime.hashGrow",
        "runtime.makeBucketArray",
        "runtime.newarray",
        "runtime.mallocgc",
        "runtime.profilealloc",
        "runtime.mProf_Malloc",
        "runtime.callers",
        "runtime.systemstack",
        "runtime.callers.func1",
        "runtime.gentraceback",
        "runtime.findfunc",
        "runtime.(*hmap).newoverflow",
        "runtime.growslice",
        "runtime.releasem",
        "runtime.growWork_fast64",
        "runtime.evacuate_fast64",
        "encoding/binary.littleEndian.PutUint64",
        "runtime.newobject",
        "runtime.acquirem",
        "runtime.mallocgc",
        "runtime.heapBitsSetType",
        "test/pkg/create.(*labelCache).rewriteLabels",
        "runtime.mapaccess1_fast64",
        "test/pkg/create.(*deduplicatingSlice[...]).ingest",
        "runtime.mapaccess2_faststr",
        "aeshashbody",
        "memeqbody",
        "runtime.mapassign_fast64",
        "runtime.(*hmap).newoverflow",
        "runtime.growslice",
        "runtime.nextFreeFast",
        "runtime.growWork_fast64",
        "runtime.evacuate_fast64",
        "runtime.mapaccess2",
        "memeqbody",
        "runtime.memhash_varlen",
        "aeshashbody",
        "runtime.evacuated",
        "test/pkg/create.(*locationsHelper).key",
        "github.com/cespare/xxhash/v2.(*Digest).Sum64",
        "github.com/cespare/xxhash/v2.(*Digest).Write",
        "runtime.bucketShift",
        "test/pkg/create.(*stringsHelper).addToRewriter",
        "encoding/binary.littleEndian.PutUint64",
        "runtime.add",
        "test/pkg/create.(*locationsHelper).rewrite",
        "test/pkg/create.idConversionTable.rewriteUint64",
        "runtime.mapaccess2_fast64",
        "github.com/prometheus/client_golang/prometheus.(*CounterVec).WithLabelValues",
        "github.com/prometheus/client_golang/prometheus.(*MetricVec).GetMetricWithLabelValues",
        "github.com/prometheus/client_golang/prometheus.(*MetricVec).hashLabelValues",
        "github.com/prometheus/client_golang/prometheus.hashAdd",
        "test/pkg/create.(*deltaProfiles).computeDelta",
        "test/pkg/create.deltaSamples",
        "runtime.mapassign_fast64",
        "runtime.growWork_fast64",
        "runtime.evacuate_fast64",
        "runtime.typedmemmove",
        "runtime.bulkBarrierPreWrite",
        "runtime.hashGrow",
        "runtime.makeBucketArray",
        "runtime.newarray",
        "runtime.mallocgc",
        "runtime.heapBitsSetType",
        "runtime.mapaccess2_fast64",
        "test/pkg/create.copySlice[...]",
        "runtime.mallocgc",
        "runtime.(*mcache).nextFree",
        "runtime.(*mcache).refill",
        "runtime.(*mcentral).cacheSpan",
        "runtime.(*mcentral).grow",
        "runtime.(*mheap).alloc",
        "runtime.systemstack",
        "runtime.(*mheap).alloc.func1",
        "runtime.(*mheap).allocSpan",
        "runtime.(*mheap).allocMSpanLocked",
        "runtime.(*fixalloc).alloc",
        "test/pkg/gen/google/v1.(*Profile).UnmarshalVT",
        "runtime.newobject",
        "runtime.nextFreeFast",
        "runtime.mallocgc",
        "runtime.add1",
        "runtime.heapBitsForAddr",
        "runtime.heapBitsSetType",
        "test/pkg/gen/google/v1.(*Location).UnmarshalVT",
        "runtime.growslice",
        "runtime.mallocgc",
        "runtime.heapBitsSetType",
        "runtime.nextFreeFast",
        "runtime.newobject",
        "runtime.getMCache",
        "runtime.mallocgc",
        "runtime.roundupsize",
        "test/pkg/gen/google/v1.(*Sample).UnmarshalVT",
        "runtime.makeslice",
        "runtime.mallocgc",
        "runtime.memclrNoHeapPointers",
        "runtime.(*mcache).nextFree",
        "runtime.(*mcache).refill",
        "runtime.(*mcentral).cacheSpan",
        "runtime.acquirem",
        "runtime.growslice",
        "runtime.mallocgc",
        "runtime.profilealloc",
        "runtime.mProf_Malloc",
        "runtime.callers",
        "runtime.systemstack",
        "runtime.callers.func1",
        "runtime.gentraceback",
        "runtime.pcdatavalue",
        "runtime.pcvalue",
        "runtime.step",
        "runtime.memclrNoHeapPointers",
        "github.com/go-kit/log.(*context).Log",
        "github.com/go-kit/log.bindValues",
        "github.com/go-kit/log.Caller.func1",
        "runtime.Caller",
        "runtime.(*Frames).Next",
        "runtime.funcline1",
        "runtime.pcvalue",
        "runtime.readvarint",
        "test/pkg/gen/google/v1.(*Profile).ReturnToVTPool",
        "test/pkg/gen/google/v1.(*Profile).ResetVT",
        "github.com/bufbuild/connect-go.(*errorTranslatingSender).Send",
        "github.com/bufbuild/connect-go.(*connectUnaryHandlerSender).Send",
        "github.com/bufbuild/connect-go.(*connectUnaryMarshaler).Marshal",
        "github.com/bufbuild/connect-go.(*compressionPool).Compress",
        "github.com/bufbuild/connect-go.(*compressionPool).putCompressor",
        "compress/gzip.(*Writer).Close",
        "compress/gzip.(*Writer).Write",
        "compress/flate.NewWriter",
        "runtime.newobject",
        "runtime.mallocgc",
        "runtime.gcAssistAlloc",
        "runtime.systemstack",
        "runtime.gcAssistAlloc.func1",
        "runtime.gcAssistAlloc1",
        "runtime.gcDrainN",
        "runtime.scanobject",
        "runtime.spanOf",
        "compress/gzip.(*Writer).Reset",
        "compress/gzip.(*Writer).init",
        "compress/flate.(*Writer).Reset",
        "compress/flate.(*compressor).reset",
        "runtime.memclrNoHeapPointers",
        "github.com/bufbuild/connect-go.(*compressionPool).getCompressor",
        "compress/gzip.(*Writer).Reset",
        "compress/gzip.(*Writer).init",
        "compress/flate.(*Writer).Reset",
        "compress/flate.(*compressor).reset",
        "runtime.memclrNoHeapPointers",
        "github.com/bufbuild/connect-go.receiveUnaryRequest[...]",
        "github.com/bufbuild/connect-go.(*errorTranslatingReceiver).Receive",
        "github.com/bufbuild/connect-go.(*connectUnaryHandlerReceiver).Receive",
        "github.com/bufbuild/connect-go.(*connectUnaryUnmarshaler).UnmarshalFunc",
        "bytes.(*Buffer).ReadFrom",
        "github.com/weaveworks/common/middleware.(*reqBody).Read",
        "net/http.(*body).Read",
        "net/http.(*body).readLocked",
        "net/http/internal.(*chunkedReader).Read",
        "bufio.(*Reader).Read",
        "net/http.(*connReader).Read",
        "net.(*conn).Read",
        "net.(*netFD).Read",
        "syscall.Read",
        "syscall.read",
        "syscall.Syscall",
        "net/http.(*ServeMux).ServeHTTP",
        "net/http.HandlerFunc.ServeHTTP",
        "net/http/pprof.Index",
        "net/http/pprof.handler.ServeHTTP",
        "runtime/pprof.(*Profile).WriteTo",
        "runtime/pprof.writeAlloc",
        "runtime/pprof.writeHeapInternal",
        "runtime/pprof.writeHeapProto",
        "runtime/pprof.(*profileBuilder).pbSample",
        "runtime/pprof.(*profileBuilder).flush",
        "compress/gzip.(*Writer).Write",
        "compress/flate.(*compressor).write",
        "compress/flate.(*compressor).encSpeed",
        "compress/flate.(*huffmanBitWriter).writeBlockDynamic",
        "compress/flate.(*huffmanBitWriter).writeTokens",
        "compress/flate.(*huffmanBitWriter).writeCode",
        "github.com/felixge/httpsnoop.(*rw).Write",
        "github.com/felixge/httpsnoop.(*Metrics).CaptureMetrics.func2.1",
        "github.com/weaveworks/common/middleware.(*badResponseLoggingWriter).Write",
        "github.com/opentracing-contrib/go-stdlib/nethttp.(*statusCodeTracker).Write",
        "net/http.(*response).Write",
        "net/http.(*response).write",
        "bufio.(*Writer).Write",
        "bufio.(*Writer).Flush",
        "net/http.(*chunkWriter).Write",
        "bufio.(*Writer).Write",
        "bufio.(*Writer).Flush",
        "net/http.checkConnErrorWriter.Write",
        "net.(*conn).Write",
        "net.(*netFD).Write",
        "internal/poll.(*FD).Write",
        "syscall.write",
        "syscall.Syscall",
        "compress/flate.token.length",
        "compress/flate.(*deflateFast).encode",
        "compress/flate.NewWriter",
        "compress/flate.(*compressor).init",
        "runtime.makeslice",
        "runtime.mallocgc",
        "runtime.gcAssistAlloc",
        "runtime.systemstack",
        "runtime.gcAssistAlloc.func1",
        "runtime.gcAssistAlloc1",
        "runtime.gcDrainN",
        "runtime.scanobject",
        "runtime.newobject",
        "runtime.mallocgc",
        "runtime.memclrNoHeapPointers",
        "runtime/pprof.writeHeapProto.func1",
        "runtime/pprof.(*profileBuilder).pbLabel",
        "runtime.mapaccess2_faststr",
        "runtime/pprof.(*protobuf).int64Opt",
        "runtime/pprof.(*protobuf).endMessage",
        "runtime.memmove",
        "runtime/pprof.(*protobuf).startMessage",
        "runtime/pprof.(*protobuf).uint64s",
        "runtime.memmove",
        "runtime/pprof.(*protobuf).int64s",
        "runtime/pprof.(*profileBuilder).appendLocsForStack",
        "runtime/pprof.(*profileBuilder).emitLocation",
        "runtime/pprof.(*profileBuilder).flush",
        "compress/gzip.(*Writer).Write",
        "compress/flate.NewWriter",
        "runtime.newobject",
        "runtime.mallocgc",
        "runtime.memclrNoHeapPointers",
        "runtime.heapBitsSetType",
        "runtime.heapBitsSetTypeGCProg",
        "runtime.runGCProg",
        "compress/flate.(*compressor).init",
        "runtime.makeslice",
        "runtime.mallocgc",
        "runtime.memclrNoHeapPointersChunked",
        "runtime.memclrNoHeapPointers",
        "runtime.gcAssistAlloc",
        "runtime.systemstack",
        "runtime.gcAssistAlloc.func1",
        "runtime.gcAssistAlloc1",
        "runtime.gcDrainN",
        "runtime.scanobject",
        "runtime.pageIndexOf",
        "compress/flate.(*compressor).write",
        "compress/flate.(*compressor).encSpeed",
        "compress/flate.(*huffmanBitWriter).writeBlockDynamic",
        "compress/flate.(*huffmanBitWriter).writeTokens",
        "compress/flate.(*huffmanBitWriter).indexTokens",
        "runtime.growslice",
        "runtime.mallocgc",
        "runtime.acquirem",
        "runtime.mapassign_faststr",
        "runtime.hashGrow",
        "runtime.makeBucketArray",
        "runtime.newarray",
        "runtime.mallocgc",
        "runtime.profilealloc",
        "runtime.mProf_Malloc",
        "runtime.callers",
        "runtime.systemstack",
        "runtime.callers.func1",
        "runtime.gentraceback",
        "runtime.pcdatavalue",
        "runtime.pcvalue",
        "runtime.step",
        "runtime.mapaccess1_faststr",
        "aeshashbody",
        "runtime.mapassign_fast64",
        "runtime.growWork_fast64",
        "runtime.evacuate_fast64",
        "runtime.memclrHasPointers",
        "runtime.bulkBarrierPreWrite",
        "runtime.wbBufFlush",
        "runtime.systemstack",
        "runtime.wbBufFlush.func1",
        "runtime.wbBufFlush1",
        "runtime.spanOf",
        "runtime.memhash64",
        "runtime/pprof.(*profileBuilder).pbLine",
        "runtime/pprof.(*protobuf).int64Opt",
        "runtime/pprof.(*protobuf).endMessage",
        "runtime.memmove",
        "runtime.duffcopy",
        "runtime/pprof.(*protobuf).endMessage",
        "runtime.memmove",
        "runtime.duffzero",
        "runtime/pprof.(*protobuf).int64Opt",
        "runtime.CallersFrames",
        "runtime.newobject",
        "runtime.nextFreeFast",
        "runtime.mallocgc",
        "runtime.heapBitsSetType",
        "runtime.memclrNoHeapPointers",
        "runtime.heapBitsForAddr",
        "runtime/pprof.allFrames",
        "runtime.duffcopy",
        "runtime.(*Frames).Next",
        "runtime.funcline1",
        "runtime.funcInfo.entry",
        "runtime.(*moduledata).textAddr",
        "runtime.pcvalue",
        "runtime.readvarint",
        "runtime.step",
        "runtime.funcfile",
        "runtime.findnull",
        "indexbytebody",
        "runtime.funcname",
        "runtime.findnull",
        "indexbytebody",
        "runtime.findfunc",
        "runtime.duffzero",
        "runtime.(*Func).Entry",
        "runtime.pcdatavalue1",
        "runtime.pcvalue",
        "runtime.step",
        "runtime.funcInfo.entry",
        "runtime.(*moduledata).textAddr",
        "runtime.newobject",
        "runtime.nextFreeFast",
        "runtime.mallocgc",
        "runtime.mapaccess2_fast64",
        "runtime.memhash64",
        "runtime/pprof.runtime_expandFinalInlineFrame",
        "runtime.findfunc",
        "runtime.pcdatavalue1",
        "runtime.funcInfo.entry",
        "runtime.(*moduledata).textAddr",
        "runtime/pprof.(*profileBuilder).stringIndex",
        "runtime.growslice",
        "runtime.mallocgc",
        "runtime.heapBitsSetType",
        "runtime.memmove",
        "runtime.mapassign_faststr",
        "runtime.growWork_faststr",
        "runtime.evacuate_faststr",
        "aeshashbody",
        "runtime.advanceEvacuationMark",
        "runtime.hashGrow",
        "runtime.makeBucketArray",
        "runtime.newarray",
        "runtime.mallocgc",
        "runtime.memclrNoHeapPointers",
        "runtime.heapBitsSetType",
        "runtime.mapaccess2_faststr",
        "aeshashbody",
        "runtime/pprof.(*protobuf).uint64Opt",
        "runtime.funcdata",
        "runtime/pprof.(*profileBuilder).build",
        "compress/gzip.(*Writer).Write",
        "compress/flate.(*compressor).write",
        "compress/flate.(*compressor).encSpeed",
        "compress/flate.(*deflateFast).encode",
        "compress/flate.(*deflateFast).matchLen",
        "compress/flate.(*huffmanBitWriter).writeBlockDynamic",
        "compress/flate.(*huffmanBitWriter).writeTokens",
        "compress/flate.(*huffmanBitWriter).writeCode",
        "github.com/felixge/httpsnoop.(*rw).Write",
        "github.com/felixge/httpsnoop.(*Metrics).CaptureMetrics.func2.1",
        "github.com/weaveworks/common/middleware.(*badResponseLoggingWriter).Write",
        "github.com/opentracing-contrib/go-stdlib/nethttp.(*statusCodeTracker).Write",
        "net/http.(*response).Write",
        "net/http.(*response).write",
        "bufio.(*Writer).Write",
        "bufio.(*Writer).Flush",
        "net/http.(*chunkWriter).Write",
        "bufio.(*Writer).Write",
        "bufio.(*Writer).Flush",
        "net/http.checkConnErrorWriter.Write",
        "net.(*conn).Write",
        "net.(*netFD).Write",
        "internal/poll.(*FD).Write",
        "syscall.write",
        "syscall.Syscall",
        "compress/flate.(*huffmanBitWriter).writeBits",
        "compress/flate.(*huffmanBitWriter).indexTokens",
        "compress/flate.(*huffmanEncoder).generate",
        "compress/flate.(*huffmanEncoder).bitCounts",
        "compress/flate.(*huffmanBitWriter).writeDynamicHeader",
        "compress/flate.(*huffmanBitWriter).writeCode",
        "github.com/felixge/httpsnoop.(*rw).Write",
        "github.com/felixge/httpsnoop.(*Metrics).CaptureMetrics.func2.1",
        "github.com/weaveworks/common/middleware.(*badResponseLoggingWriter).Write",
        "github.com/opentracing-contrib/go-stdlib/nethttp.(*statusCodeTracker).Write",
        "net/http.(*response).Write",
        "net/http.(*response).write",
        "bufio.(*Writer).Write",
        "bufio.(*Writer).Flush",
        "net/http.(*chunkWriter).Write",
        "bufio.(*Writer).Write",
        "bufio.(*Writer).Flush",
        "net/http.checkConnErrorWriter.Write",
        "net.(*conn).Write",
        "net.(*netFD).Write",
        "internal/poll.(*FD).Write",
        "syscall.write",
        "syscall.Syscall",
        "compress/flate.hash",
        "compress/gzip.(*Writer).Close",
        "compress/flate.(*compressor).close",
        "compress/flate.(*compressor).encSpeed",
        "compress/flate.(*huffmanBitWriter).writeBlockDynamic",
        "compress/flate.(*huffmanBitWriter).indexTokens",
        "compress/flate.(*huffmanBitWriter).writeTokens",
        "compress/flate.(*huffmanBitWriter).writeBits",
        "github.com/felixge/httpsnoop.(*rw).Write",
        "github.com/felixge/httpsnoop.(*Metrics).CaptureMetrics.func2.1",
        "github.com/weaveworks/common/middleware.(*badResponseLoggingWriter).Write",
        "github.com/opentracing-contrib/go-stdlib/nethttp.(*statusCodeTracker).Write",
        "net/http.(*response).Write",
        "net/http.(*response).write",
        "bufio.(*Writer).Write",
        "bufio.(*Writer).Flush",
        "net/http.(*chunkWriter).Write",
        "fmt.Fprintf",
        "fmt.(*pp).doPrintf",
        "fmt.(*pp).printArg",
        "fmt.(*pp).fmtInteger",
        "fmt.(*fmt).fmtInteger",
        "compress/flate.(*huffmanBitWriter).writeCode",
        "compress/flate.token.offset",
        "compress/flate.hash",
        "compress/flate.(*deflateFast).encode",
        "compress/flate.load64",
        "runtime.FuncForPC",
        "runtime.newobject",
        "runtime.mallocgc",
        "runtime.heapBitsSetType",
        "runtime.findfunc",
        "runtime.pcdatavalue1",
        "runtime.funcInfo.entry",
        "runtime.(*moduledata).textAddr",
        "runtime.pcvalue",
        "runtime.readvarint",
        "runtime.step",
        "runtime.funcline1",
        "runtime.pcvalue",
        "runtime.readvarint",
        "runtime.step",
        "runtime.funcInfo.entry",
        "runtime.(*moduledata).textAddr",
        "runtime/pprof.newProfileBuilder",
        "runtime/pprof.(*profileBuilder).readMapping",
        "os.ReadFile",
        "runtime.growslice",
        "runtime.mallocgc",
        "runtime.gcAssistAlloc",
        "runtime.systemstack",
        "runtime.gcAssistAlloc.func1",
        "runtime.gcAssistAlloc1",
        "runtime.gcDrainN",
        "runtime.duffcopy",
        "runtime.(*Func).Name",
        "runtime.funcname",
        "runtime.findnull",
        "indexbytebody",
        "runtime/pprof.scaleHeapSample",
        "math.archExp",
        "runtime.(*MemProfileRecord).Stack",
        "runtime.duffcopy",
        "runtime/pprof.writeGoroutine",
        "runtime/pprof.writeRuntimeProfile",
        "runtime/pprof.runtime_goroutineProfileWithLabels",
        "runtime.goroutineProfileWithLabels",
        "runtime.forEachGRace",
        "runtime.goroutineProfileWithLabels.func4",
        "runtime.systemstack",
        "runtime.goroutineProfileWithLabels.func4.1",
        "runtime.saveg",
        "runtime.gentraceback",
        "runtime.pcdatavalue",
        "runtime.funcInfo.entry",
        "runtime.pcvalue",
        "runtime.step",
        "runtime.readvarint",
        "runtime.funcspdelta",
        "runtime.pcvalue",
        "runtime.readvarint",
        "runtime.step",
        "runtime.findfunc",
        "runtime.goroutineProfileWithLabels.func1",
        "runtime.isSystemGoroutine",
        "runtime.funcname",
        "runtime.findnull",
        "runtime.goroutineProfileWithLabels.func2",
        "runtime.goroutineProfileWithLabels.func1",
        "runtime.isSystemGoroutine",
        "runtime.startTheWorld",
        "runtime.systemstack",
        "runtime.startTheWorld.func1",
        "runtime.startTheWorldWithSema",
        "runtime.wakep",
        "runtime.startm",
        "runtime.notewakeup",
        "runtime.futexwakeup",
        "runtime.futex",
        "runtime.systemstack",
        "runtime.goroutineProfileWithLabels.func3",
        "runtime.saveg",
        "runtime.gentraceback",
        "runtime/pprof.printCountProfile",
        "runtime/pprof.printCountProfile.func1",
        "fmt.Fprintf",
        "fmt.(*pp).doPrintf",
        "fmt.(*pp).printArg",
        "fmt.(*pp).fmtInteger",
        "fmt.(*fmt).fmtInteger",
        "bytes.(*Buffer).Write",
        "runtime.memmove",
        "runtime.slicebytetostring",
        "runtime.nextFreeFast",
        "runtime.mapassign_faststr",
        "runtime.hashGrow",
        "runtime.makeBucketArray",
        "runtime.newarray",
        "runtime.nextFreeFast",
        "aeshashbody",
        "runtime/pprof.(*profileBuilder).appendLocsForStack",
        "runtime/pprof.(*profileBuilder).stringIndex",
        "runtime.overLoadFactor",
        "runtime/pprof.(*profileBuilder).emitLocation",
        "runtime/pprof.(*profileBuilder).pbLine",
        "runtime/pprof.(*profileBuilder).flush",
        "compress/gzip.(*Writer).Write",
        "compress/flate.NewWriter",
        "compress/flate.(*compressor).init",
        "runtime.makeslice",
        "runtime.mallocgc",
        "runtime.gcAssistAlloc",
        "runtime.systemstack",
        "runtime.gcAssistAlloc.func1",
        "runtime.gcAssistAlloc1",
        "runtime.gcDrainN",
        "runtime.heapBits.bits",
        "runtime.scanobject",
        "runtime.greyobject",
        "runtime.newobject",
        "runtime.mallocgc",
        "runtime.memclrNoHeapPointers",
        "runtime/pprof.allFrames",
        "runtime.newobject",
        "runtime.nextFreeFast",
        "runtime.(*Frames).Next",
        "runtime.funcline1",
        "runtime.pcvalue",
        "runtime/pprof.newProfileBuilder",
        "runtime/pprof.(*profileBuilder).readMapping",
        "runtime/pprof.parseProcSelfMaps",
        "runtime/pprof.elfBuildID",
        "os.OpenFile",
        "syscall.Open",
        "syscall.openat",
        "syscall.Syscall6",
        "runtime/pprof.parseProcSelfMaps.func1",
        "bytes.Cut",
        "runtime.slicebytetostring",
        "runtime.nextFreeFast",
        "runtime/pprof.(*profileBuilder).addMapping",
        "runtime.growslice",
        "os.Open",
        "os.OpenFile",
        "syscall.Open",
        "syscall.openat",
        "syscall.Syscall6",
        "os.openFileNolog",
        "os.newFile",
        "runtime.SetFinalizer",
        "runtime.systemstack",
        "runtime.SetFinalizer.func2",
        "runtime.addfinalizer",
        "runtime.addspecial",
        "os.ReadFile",
        "runtime.growslice",
        "runtime.mallocgc",
        "runtime.(*mcache).nextFree",
        "runtime.(*mcache).refill",
        "runtime.(*mcentral).cacheSpan",
        "runtime.(*mcentral).grow",
        "runtime.(*mheap).alloc",
        "runtime.systemstack",
        "runtime.(*mheap).alloc.func1",
        "runtime.(*mheap).allocSpan",
        "runtime.(*pageAlloc).allocToCache",
        "os.(*File).Stat",
        "internal/poll.(*FD).Fstat",
        "syscall.Fstat",
        "syscall.Syscall",
        "os.(*File).Read",
        "syscall.Read",
        "syscall.read",
        "syscall.Syscall",
        "runtime.newobject",
        "sort.Sort",
        "sort.quickSort",
        "sort.doPivot",
        "runtime/pprof.(*keysByCount).Less",
        "cmpbody",
        "sort.quickSort",
        "sort.insertionSort",
        "runtime/pprof.(*keysByCount).Less",
        "runtime.mapaccess1_faststr",
        "runtime/pprof.(*profileBuilder).pbSample",
        "runtime/pprof.(*protobuf).uint64s",
        "runtime.newobject",
        "github.com/prometheus/client_golang/prometheus.(*GaugeVec).WithLabelValues",
        "runtime.assertI2I",
        "runtime.getitab",
        "runtime.(*itabTableType).find",
        "github.com/weaveworks/common/middleware.Instrument.getRouteName",
        "github.com/weaveworks/common/middleware.getRouteName",
        "github.com/weaveworks/common/middleware.MakeLabelValue",
        "strings.Trim",
        "github.com/gorilla/mux.(*Router).Match",
        "github.com/gorilla/mux.(*Route).Match",
        "github.com/gorilla/mux.(*routeRegexp).Match",
        "regexp.(*Regexp).doExecute",
        "regexp.(*Regexp).backtrack",
        "regexp.(*Regexp).tryBacktrack",
        "github.com/weaveworks/common/logging.gokit.Debugf",
        "github.com/go-kit/log.(*context).Log",
        "github.com/go-kit/log.bindValues",
        "github.com/go-kit/log.Caller.func1",
        "runtime.Caller",
        "runtime.callers",
        "github.com/uber/jaeger-client-go.(*Tracer).Extract",
        "github.com/uber/jaeger-client-go.(*TextMapPropagator).Extract",
        "runtime.assertE2I2",
        "runtime.getitab",
        "runtime.(*itabTableType).find",
        "runtime.newobject",
        "runtime.nextFreeFast",
        "github.com/weaveworks/common/middleware.Tracer.Wrap.func1",
        "github.com/weaveworks/common/middleware.getRouteName",
        "github.com/gorilla/mux.(*Router).Match",
        "github.com/gorilla/mux.(*Route).Match",
        "github.com/gorilla/mux.(*routeRegexp).Match",
        "regexp.(*Regexp).doExecute",
        "net/http.(*response).finishRequest",
        "bufio.(*Writer).Flush",
        "net/http.(*chunkWriter).Write",
        "net/http.(*chunkWriter).writeHeader",
        "net/http.Header.writeSubset",
        "net/http.Header.sortedKeyValues",
        "sync.(*Pool).Get",
        "net/http.checkConnErrorWriter.Write",
        "net.(*conn).Write",
        "net.(*netFD).Write",
        "internal/poll.(*FD).Write",
        "syscall.write",
        "syscall.Syscall",
        "net/http.(*conn).readRequest",
        "runtime.mapiterinit",
        "runtime.gcBgMarkWorker",
        "runtime.systemstack",
        "runtime.gcBgMarkWorker.func2",
        "runtime.gcDrain",
        "runtime.scanobject",
        "runtime.(*gcBits).bytep",
        "runtime.(*mspan).objIndex",
        "runtime.pageIndexOf",
        "runtime.greyobject",
        "runtime.(*gcWork).put",
        "runtime.getempty",
        "runtime.putfull",
        "runtime.(*lfstack).push",
        "runtime.findObject",
        "runtime.spanOf",
        "runtime.(*mspan).markBitsForIndex",
        "runtime.spanOf",
        "runtime.markBits.isMarked",
        "runtime.arenaIndex",
        "runtime.heapBitsForAddr",
        "runtime.(*gcWork).tryGet",
        "runtime.trygetfull",
        "runtime.spanOfUnchecked",
        "runtime.heapBits.next",
        "runtime.gcFlushBgCredit",
        "runtime.heapBits.bits",
        "runtime.(*gcWork).balance",
        "runtime.handoff",
        "runtime.getempty",
        "runtime.(*gcWork).tryGetFast",
        "runtime.gcMarkDone",
        "runtime.gcMarkTermination",
        "runtime.mProf_Flush",
        "runtime.mProf_FlushLocked",
        "runtime.(*bucket).mp",
        "runtime.systemstack",
        "runtime.gcMarkTermination.func4",
        "runtime.forEachP",
        "runtime.gcMarkTermination.func4.1",
        "runtime.(*mcache).prepareForSweep",
        "runtime.(*mcache).releaseAll",
        "net/http.(*persistConn).readLoop",
        "runtime.selectgo",
        "runtime.send",
        "runtime.goready",
        "runtime.systemstack",
        "runtime.goready.func1",
        "runtime.ready",
        "runtime.runqput",
        "bufio.(*Reader).Peek",
        "bufio.(*Reader).fill",
        "net/http.(*persistConn).Read",
        "net.(*conn).Read",
        "net.(*netFD).Read",
        "syscall.Read",
        "syscall.read",
        "syscall.Syscall",
        "net/http.(*persistConn).readResponse",
        "github.com/opentracing-contrib/go-stdlib/nethttp.(*Tracer).gotFirstResponseByte",
        "runtime.newobject",
        "runtime.mallocgc",
        "runtime.memclrNoHeapPointers",
        "runtime.bgsweep",
        "runtime.sweepone",
        "runtime.(*sweepLocker).tryAcquire",
        "runtime.(*sweepLocked).sweep",
        "runtime.(*gcBitsArena).tryAlloc",
        "runtime.spanClass.sizeclass",
        "runtime/internal/atomic.(*Uint32).CompareAndSwap",
        "runtime.mcall",
        "runtime.park_m",
        "runtime.schedule",
        "runtime.findrunnable",
        "runtime.netpoll",
        "runtime.epollwait",
        "runtime.stopm",
        "runtime.notesleep",
        "runtime.futexsleep",
        "runtime.futex",
        "runtime.(*randomOrder).start",
        "runtime.checkTimers",
        "runtime.runtimer",
        "runtime.dodeltimer0",
        "runtime.gcstopm",
        "runtime.lock2",
        "runtime.procyield",
        "runtime.(*lfstack).pop",
        "runtime.gosched_m",
        "runtime.lock",
        "runtime.lock2",
        "runtime.goschedImpl",
        "runtime.schedule",
        "runtime.execute",
        "runtime.casgstatus",
        "runtime.checkTimers",
        "runtime.findrunnable",
        "runtime.checkTimers",
        "runtime.casgstatus",
        "net/http.(*persistConn).writeLoop",
        "net/http.(*Request).write",
        "net/http.newTransferWriter",
        "runtime.convI2I",
        "runtime.getitab",
        "runtime.(*itabTableType).find",
        "runtime.chansend1",
        "runtime.chansend",
        "runtime.send",
        "runtime.goready",
        "runtime.systemstack",
        "runtime.goready.func1",
        "runtime.ready",
        "runtime.wakep",
        "runtime.startm",
        "runtime.notewakeup",
        "runtime.futexwakeup",
        "runtime.futex",
        "bufio.(*Writer).Flush",
        "net/http.persistConnWriter.Write",
        "net.(*conn).Write",
        "net.(*netFD).Write",
        "internal/poll.(*FD).Write",
        "syscall.write",
        "syscall.Syscall",
        "runtime.exitsyscall",
        "runtime.exitsyscallfast",
        "runtime.exitsyscallfast_reacquired",
        "test/pkg/distributor.(*Distributor).Push.func1",
        "test/pkg/distributor.(*Distributor).sendProfiles",
        "test/pkg/distributor.(*Distributor).sendProfilesErr",
        "test/pkg/gen/ingester/v1/ingesterv1connect.(*ingesterServiceClient).Push",
        "github.com/bufbuild/connect-go.(*Client[...]).CallUnary",
        "github.com/bufbuild/connect-go.NewClient[...].func2",
        "github.com/bufbuild/connect-go.NewClient[...].func1",
        "github.com/bufbuild/connect-go.receiveUnaryResponse[...]",
        "github.com/bufbuild/connect-go.(*errorTranslatingReceiver).Receive",
        "github.com/bufbuild/connect-go.(*connectUnaryClientReceiver).Receive",
        "github.com/bufbuild/connect-go.(*connectUnaryUnmarshaler).UnmarshalFunc",
        "github.com/bufbuild/connect-go.(*compressionPool).Decompress",
        "github.com/bufbuild/connect-go.(*compressionPool).getDecompressor",
        "compress/gzip.(*Reader).Reset",
        "compress/gzip.(*Reader).readHeader",
        "compress/flate.(*dictDecoder).init",
        "runtime.makeslice",
        "runtime.mallocgc",
        "runtime.(*mcache).nextFree",
        "runtime.(*mcache).refill",
        "runtime.(*mcentral).cacheSpan",
        "runtime.(*mcentral).grow",
        "runtime.heapBits.initSpan",
        "compress/flate.(*decompressor).Reset",
        "runtime.newstack",
        "runtime.copystack",
        "runtime.gentraceback",
        "compress/flate.NewReader",
        "compress/flate.fixedHuffmanDecoderInit",
        "runtime.newstack",
        "runtime.copystack",
        "runtime.gentraceback",
        "runtime.funcspdelta",
        "runtime.pcvalue",
        "runtime.step",
        "github.com/bufbuild/connect-go.(*errorTranslatingSender).Send",
        "github.com/bufbuild/connect-go.(*connectClientSender).Send",
        "github.com/bufbuild/connect-go.(*connectUnaryMarshaler).Marshal",
        "github.com/bufbuild/connect-go.(*connectUnaryMarshaler).write",
        "github.com/bufbuild/connect-go.(*duplexHTTPCall).Write",
        "io.(*pipe).write",
        "runtime.chanrecv1",
        "runtime.chanrecv",
        "runtime.recv",
        "runtime.gcWriteBarrierR8",
        "runtime.gcWriteBarrier",
        "runtime.wbBufFlush",
        "runtime.systemstack",
        "runtime.wbBufFlush.func1",
        "runtime.wbBufFlush1",
        "runtime.findObject",
        "github.com/bufbuild/connect-go.(*errorTranslatingReceiver).Close",
        "github.com/bufbuild/connect-go.(*connectUnaryClientReceiver).Close",
        "github.com/bufbuild/connect-go.(*duplexHTTPCall).CloseRead",
        "github.com/opentracing-contrib/go-stdlib/nethttp.closeTracker.Close",
        "runtime.newobject",
        "runtime.mallocgc",
        "runtime.heapBitsSetType",
        "runtime/pprof.profileWriter",
        "runtime/pprof.(*profileBuilder).addCPUData",
        "runtime/pprof.(*profMap).lookup",
        "runtime.makeslice",
        "runtime.mallocgc",
        "runtime.gcAssistAlloc",
        "runtime.systemstack",
        "runtime.gcAssistAlloc.func1",
        "runtime.gcAssistAlloc1",
        "runtime.gcDrainN",
        "runtime.scanobject",
        "runtime.pageIndexOf",
        "net/http.(*connReader).backgroundRead",
        "net.(*conn).Read",
        "net.(*netFD).Read",
        "internal/poll.(*FD).Read"
      ]
    },
    {
      name: "fileName",
      values: [
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/net.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/fd_posix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/internal/poll/fd_unix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/zsyscall_linux_amd64.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/memclr_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/asm_linux_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/net.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/fd_posix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/internal/poll/fd_unix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/zsyscall_linux_amd64.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/memclr_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/asm_linux_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/net.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/fd_posix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/internal/poll/fd_unix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/zsyscall_linux_amd64.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/memclr_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/asm_linux_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/net.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/fd_posix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/internal/poll/fd_unix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/zsyscall_linux_amd64.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/memclr_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/asm_linux_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/net.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/fd_posix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/internal/poll/fd_unix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/zsyscall_linux_amd64.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/memclr_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/asm_linux_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/net.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/fd_posix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/internal/poll/fd_unix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/zsyscall_linux_amd64.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/memclr_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/asm_linux_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/net.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/fd_posix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/internal/poll/fd_unix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/zsyscall_linux_amd64.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/memclr_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/asm_linux_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/net.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/fd_posix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/internal/poll/fd_unix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/zsyscall_linux_amd64.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/memclr_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/asm_linux_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/net.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/fd_posix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/internal/poll/fd_unix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/zsyscall_linux_amd64.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/memclr_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/asm_linux_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/net.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/fd_posix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/internal/poll/fd_unix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/zsyscall_linux_amd64.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/memclr_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/asm_linux_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/net.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/fd_posix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/internal/poll/fd_unix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/zsyscall_linux_amd64.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/memclr_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/asm_linux_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/net.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/fd_posix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/internal/poll/fd_unix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/zsyscall_linux_amd64.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/memclr_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/asm_linux_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/net.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/fd_posix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/internal/poll/fd_unix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/zsyscall_linux_amd64.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/memclr_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/asm_linux_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/net.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/fd_posix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/internal/poll/fd_unix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/zsyscall_linux_amd64.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/memclr_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/asm_linux_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/net.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/fd_posix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/internal/poll/fd_unix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/zsyscall_linux_amd64.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/memclr_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/asm_linux_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/net.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/fd_posix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/internal/poll/fd_unix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/zsyscall_linux_amd64.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/memclr_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/asm_linux_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/net.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/fd_posix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/internal/poll/fd_unix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/zsyscall_linux_amd64.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/memclr_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/asm_linux_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/net.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/fd_posix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/internal/poll/fd_unix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/zsyscall_linux_amd64.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/memclr_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/asm_linux_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/net.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/fd_posix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/internal/poll/fd_unix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/zsyscall_linux_amd64.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/memclr_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/asm_linux_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/net.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/fd_posix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/internal/poll/fd_unix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/zsyscall_linux_amd64.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/memclr_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/asm_linux_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/net.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/fd_posix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/internal/poll/fd_unix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/zsyscall_linux_amd64.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/memclr_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/asm_linux_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/net.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/fd_posix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/internal/poll/fd_unix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/zsyscall_linux_amd64.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/memclr_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/asm_linux_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/net.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/fd_posix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/internal/poll/fd_unix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/zsyscall_linux_amd64.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/memclr_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/asm_linux_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/net.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/fd_posix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/internal/poll/fd_unix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/zsyscall_linux_amd64.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/memclr_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/asm_linux_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/net.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/fd_posix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/internal/poll/fd_unix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/zsyscall_linux_amd64.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/memclr_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/asm_linux_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/net.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/fd_posix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/internal/poll/fd_unix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/zsyscall_linux_amd64.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/memclr_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/asm_linux_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/net.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/fd_posix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/internal/poll/fd_unix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/zsyscall_linux_amd64.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/memclr_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/asm_linux_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/net.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/fd_posix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/internal/poll/fd_unix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/zsyscall_linux_amd64.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/memclr_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/asm_linux_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/net.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/fd_posix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/internal/poll/fd_unix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/zsyscall_linux_amd64.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/memclr_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/asm_linux_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/net.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/fd_posix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/internal/poll/fd_unix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/zsyscall_linux_amd64.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/memclr_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/asm_linux_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/net.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/fd_posix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/internal/poll/fd_unix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/zsyscall_linux_amd64.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/memclr_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/asm_linux_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/net.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/fd_posix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/internal/poll/fd_unix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/zsyscall_linux_amd64.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/memclr_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/asm_linux_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/net.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/fd_posix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/internal/poll/fd_unix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/zsyscall_linux_amd64.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/memclr_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/asm_linux_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/net.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/fd_posix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/internal/poll/fd_unix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/zsyscall_linux_amd64.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/memclr_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/asm_linux_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/net.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/fd_posix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/internal/poll/fd_unix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/zsyscall_linux_amd64.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/memclr_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/asm_linux_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/net.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/fd_posix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/internal/poll/fd_unix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/zsyscall_linux_amd64.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/memclr_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/asm_linux_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/net.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/fd_posix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/internal/poll/fd_unix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/zsyscall_linux_amd64.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/memclr_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/asm_linux_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/net.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/fd_posix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/internal/poll/fd_unix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/zsyscall_linux_amd64.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/memclr_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/asm_linux_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/net.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/fd_posix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/internal/poll/fd_unix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/zsyscall_linux_amd64.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/memclr_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/asm_linux_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/net.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/fd_posix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/internal/poll/fd_unix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/zsyscall_linux_amd64.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/memclr_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/asm_linux_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/net.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/fd_posix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/internal/poll/fd_unix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/zsyscall_linux_amd64.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/memclr_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/asm_linux_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/net.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/fd_posix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/internal/poll/fd_unix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/zsyscall_linux_amd64.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/memclr_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/asm_linux_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/net.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/fd_posix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/internal/poll/fd_unix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/zsyscall_linux_amd64.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/memclr_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/asm_linux_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/net.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/fd_posix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/internal/poll/fd_unix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/zsyscall_linux_amd64.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/memclr_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/asm_linux_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/net.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/fd_posix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/internal/poll/fd_unix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/zsyscall_linux_amd64.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/memclr_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/asm_linux_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/net.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/fd_posix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/internal/poll/fd_unix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/zsyscall_linux_amd64.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/memclr_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/asm_linux_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/net.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/fd_posix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/internal/poll/fd_unix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/zsyscall_linux_amd64.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/memclr_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/asm_linux_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/net.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/fd_posix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/internal/poll/fd_unix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/zsyscall_linux_amd64.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/memclr_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/asm_linux_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/net.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/fd_posix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/internal/poll/fd_unix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/zsyscall_linux_amd64.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/memclr_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/asm_linux_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/net.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/fd_posix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/internal/poll/fd_unix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/zsyscall_linux_amd64.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/memclr_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/asm_linux_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/net.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/fd_posix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/internal/poll/fd_unix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/zsyscall_linux_amd64.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/memclr_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/asm_linux_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/net.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/fd_posix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/internal/poll/fd_unix.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/zsyscall_linux_amd64.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/malloc.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/memclr_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/compress/flate/huffman_bit_writer.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/syscall_linux.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/runtime/internal/syscall/asm_linux_amd64.s",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/bufio/bufio.go",
        "/usr/local/Cellar/go/1.19.4/libexec/src/net/http/server.go"
      ]
    },
    {
      name: "line",
      values: [
        0,
        1,
        2,
        3,
        4,
        5,
        6,
        7,
        8,
        9,
        10,
        11,
        12,
        11,
        12,
        13,
        14,
        15,
        16,
        10,
        11,
        12,
        13,
        9,
        10,
        9,
        9,
        7,
        7,
        6,
        5,
        6,
        7,
        8,
        8,
        9,
        10,
        11,
        12,
        13,
        8,
        9,
        10,
        6,
        7,
        8,
        9,
        10,
        11,
        12,
        13,
        14,
        15,
        4,
        5,
        6,
        7,
        8,
        9,
        9,
        10,
        11,
        12,
        11,
        12,
        13,
        8,
        8,
        7,
        8,
        8,
        7,
        6,
        7,
        8,
        9,
        9,
        10,
        11,
        12,
        13,
        14,
        15,
        16,
        17,
        18,
        9,
        8,
        8,
        7,
        8,
        6,
        7,
        8,
        8,
        7,
        6,
        7,
        8,
        8,
        8,
        9,
        10,
        11,
        12,
        4,
        5,
        6,
        7,
        8,
        9,
        9,
        10,
        11,
        9,
        10,
        7,
        4,
        5,
        6,
        7,
        8,
        8,
        7,
        8,
        9,
        10,
        11,
        12,
        13,
        10,
        10,
        11,
        9,
        10,
        10,
        11,
        12,
        13,
        14,
        15,
        16,
        17,
        8,
        9,
        5,
        6,
        7,
        6,
        7,
        6,
        5,
        6,
        6,
        6,
        6,
        5,
        6,
        7,
        6,
        5,
        6,
        7,
        8,
        4,
        5,
        6,
        5,
        6,
        5,
        4,
        5,
        5,
        3,
        4,
        5,
        6,
        7,
        7,
        8,
        7,
        8,
        9,
        8,
        9,
        8,
        7,
        8,
        9,
        1,
        2,
        3,
        4,
        5,
        6,
        7,
        8,
        9,
        10,
        11,
        12,
        13,
        14,
        15,
        16,
        17,
        18,
        19,
        20,
        21,
        22,
        23,
        24,
        24,
        24,
        25,
        26,
        24,
        25,
        22,
        23,
        23,
        24,
        25,
        26,
        27,
        28,
        29,
        30,
        31,
        32,
        21,
        22,
        23,
        24,
        25,
        26,
        27,
        26,
        24,
        25,
        26,
        25,
        24,
        24,
        25,
        26,
        27,
        28,
        29,
        30,
        31,
        32,
        33,
        34,
        35,
        25,
        26,
        27,
        25,
        26,
        24,
        23,
        24,
        24,
        25,
        23,
        23,
        22,
        23,
        24,
        24,
        23,
        24,
        25,
        26,
        24,
        25,
        23,
        24,
        24,
        25,
        23,
        23,
        24,
        24,
        23,
        23,
        23,
        23,
        23,
        24,
        25,
        22,
        23,
        24,
        25,
        22,
        23,
        24,
        25,
        26,
        27,
        28,
        25,
        26,
        27,
        28,
        29,
        24,
        22,
        23,
        24,
        25,
        26,
        27,
        28,
        29,
        30,
        31,
        32,
        33,
        21,
        22,
        23,
        23,
        24,
        24,
        24,
        22,
        23,
        24,
        25,
        24,
        23,
        24,
        24,
        23,
        22,
        23,
        24,
        25,
        25,
        26,
        27,
        24,
        22,
        23,
        24,
        25,
        26,
        27,
        28,
        29,
        30,
        31,
        32,
        24,
        21,
        22,
        23,
        24,
        25,
        26,
        27,
        28,
        21,
        22,
        19,
        20,
        21,
        22,
        23,
        24,
        25,
        26,
        27,
        28,
        29,
        30,
        31,
        32,
        33,
        34,
        35,
        24,
        25,
        26,
        27,
        28,
        23,
        24,
        25,
        26,
        27,
        28,
        19,
        20,
        21,
        22,
        23,
        24,
        25,
        26,
        27,
        28,
        29,
        30,
        31,
        32,
        33,
        34,
        17,
        18,
        19,
        20,
        21,
        22,
        23,
        24,
        25,
        26,
        27,
        28,
        29,
        30,
        31,
        32,
        33,
        34,
        35,
        36,
        37,
        38,
        39,
        40,
        41,
        42,
        43,
        44,
        45,
        46,
        47,
        48,
        49,
        31,
        30,
        28,
        29,
        30,
        31,
        32,
        33,
        34,
        35,
        36,
        37,
        29,
        30,
        31,
        26,
        27,
        28,
        28,
        28,
        29,
        27,
        26,
        27,
        26,
        25,
        26,
        27,
        28,
        29,
        30,
        31,
        32,
        32,
        33,
        34,
        30,
        31,
        32,
        33,
        34,
        33,
        34,
        35,
        36,
        37,
        38,
        39,
        29,
        30,
        31,
        32,
        32,
        27,
        28,
        28,
        27,
        28,
        29,
        30,
        31,
        32,
        33,
        34,
        35,
        36,
        37,
        38,
        39,
        40,
        27,
        28,
        27,
        28,
        29,
        30,
        31,
        32,
        33,
        34,
        35,
        36,
        28,
        27,
        28,
        28,
        29,
        27,
        27,
        28,
        27,
        27,
        26,
        27,
        28,
        28,
        29,
        29,
        29,
        26,
        27,
        27,
        28,
        29,
        30,
        29,
        30,
        30,
        29,
        30,
        31,
        28,
        29,
        30,
        28,
        28,
        28,
        28,
        29,
        30,
        29,
        30,
        27,
        28,
        28,
        26,
        27,
        26,
        27,
        27,
        28,
        29,
        26,
        27,
        28,
        29,
        28,
        27,
        28,
        29,
        30,
        30,
        28,
        29,
        30,
        31,
        32,
        32,
        27,
        28,
        26,
        25,
        25,
        26,
        27,
        28,
        29,
        30,
        29,
        30,
        31,
        32,
        33,
        34,
        35,
        36,
        37,
        38,
        39,
        40,
        41,
        42,
        43,
        44,
        45,
        46,
        47,
        48,
        31,
        30,
        31,
        32,
        30,
        31,
        32,
        33,
        34,
        35,
        36,
        37,
        38,
        39,
        40,
        41,
        42,
        43,
        44,
        45,
        46,
        47,
        48,
        29,
        26,
        27,
        28,
        29,
        30,
        30,
        31,
        32,
        33,
        34,
        35,
        36,
        37,
        38,
        39,
        40,
        41,
        42,
        43,
        44,
        45,
        31,
        30,
        29,
        29,
        29,
        25,
        26,
        27,
        28,
        26,
        26,
        27,
        28,
        27,
        28,
        28,
        26,
        27,
        28,
        28,
        27,
        28,
        25,
        26,
        27,
        28,
        29,
        30,
        31,
        32,
        33,
        34,
        25,
        25,
        26,
        27,
        28,
        25,
        26,
        24,
        25,
        22,
        23,
        24,
        25,
        26,
        27,
        28,
        29,
        30,
        31,
        32,
        33,
        33,
        34,
        34,
        32,
        33,
        34,
        34,
        32,
        28,
        29,
        30,
        31,
        27,
        28,
        29,
        26,
        27,
        28,
        29,
        30,
        31,
        32,
        33,
        34,
        26,
        27,
        28,
        29,
        24,
        25,
        26,
        27,
        28,
        29,
        30,
        27,
        28,
        26,
        27,
        25,
        26,
        27,
        28,
        29,
        26,
        25,
        26,
        27,
        26,
        27,
        27,
        28,
        29,
        30,
        31,
        32,
        33,
        34,
        35,
        36,
        37,
        38,
        38,
        39,
        30,
        31,
        32,
        26,
        27,
        28,
        27,
        28,
        29,
        25,
        26,
        27,
        28,
        29,
        30,
        31,
        32,
        28,
        29,
        28,
        29,
        28,
        29,
        27,
        28,
        29,
        30,
        31,
        29,
        30,
        31,
        32,
        33,
        34,
        35,
        27,
        28,
        29,
        30,
        31,
        32,
        33,
        34,
        35,
        36,
        37,
        38,
        28,
        29,
        30,
        31,
        28,
        29,
        30,
        31,
        26,
        25,
        26,
        27,
        28,
        29,
        27,
        28,
        29,
        30,
        25,
        26,
        15,
        14,
        15,
        16,
        17,
        13,
        14,
        15,
        16,
        15,
        16,
        17,
        18,
        19,
        20,
        12,
        13,
        14,
        15,
        16,
        17,
        10,
        11,
        12,
        13,
        14,
        12,
        13,
        10,
        11,
        12,
        13,
        14,
        15,
        2,
        3,
        4,
        5,
        6,
        7,
        8,
        4,
        5,
        6,
        7,
        8,
        9,
        2,
        3,
        1,
        2,
        3,
        4,
        5,
        6,
        6,
        6,
        6,
        7,
        8,
        8,
        9,
        6,
        7,
        6,
        6,
        6,
        6,
        5,
        5,
        6,
        5,
        5,
        5,
        5,
        5,
        6,
        7,
        4,
        2,
        3,
        4,
        5,
        6,
        4,
        5,
        6,
        7,
        8,
        9,
        1,
        2,
        3,
        4,
        5,
        6,
        7,
        8,
        2,
        3,
        4,
        5,
        6,
        7,
        8,
        9,
        2,
        3,
        4,
        5,
        6,
        1,
        2,
        3,
        3,
        4,
        3,
        2,
        1,
        2,
        3,
        4,
        5,
        6,
        5,
        6,
        7,
        8,
        5,
        5,
        6,
        7,
        4,
        5,
        6,
        4,
        2,
        3,
        4,
        3,
        4,
        5,
        6,
        5,
        5,
        6,
        4,
        1,
        2,
        3,
        4,
        5,
        6,
        2,
        3,
        4,
        5,
        6,
        7,
        8,
        9,
        10,
        11,
        12,
        13,
        2,
        3,
        4,
        5,
        6,
        7,
        8,
        9,
        10,
        11,
        1,
        2,
        3,
        4,
        5,
        6,
        7,
        8,
        9,
        10,
        11,
        12,
        13,
        14,
        15,
        16,
        17,
        18,
        19,
        20,
        21,
        22,
        23,
        16,
        17,
        18,
        19,
        16,
        17,
        18,
        19,
        20,
        21,
        22,
        23,
        8,
        9,
        10,
        11,
        12,
        13,
        14,
        15,
        16,
        17,
        18,
        19,
        20,
        21,
        22,
        23,
        8,
        9,
        10,
        11,
        12,
        13,
        14,
        1,
        2,
        3,
        4,
        5,
        6,
        7,
        8,
        9,
        10,
        11,
        12,
        1,
        2,
        3,
        4
      ]
    }
  ]
};

exports.FlameGraph = FlameGraphContainer;
exports.checkFields = checkFields;
exports.data = data;
exports.getMessageCheckFieldsResult = getMessageCheckFieldsResult;
//# sourceMappingURL=index.js.map
