import { delay, http, HttpResponse } from 'msw';
import { SetupServerApi } from 'msw/lib/node';

import { setDataSourceSrv } from '@grafana/runtime';
import { AlertManagerDataSourceJsonData, AlertManagerImplementation } from 'app/plugins/datasource/alertmanager/types';

import { mockDataSource, MockDataSourceSrv } from '../../../mocks';
import * as config from '../../../utils/config';
import { DataSourceType } from '../../../utils/datasource';

import internalAlertmanagerConfig from './api/alertmanager/grafana/config/api/v1/alerts.json';
import history from './api/alertmanager/grafana/config/history.json';
import vanillaAlertmanagerConfig from './api/alertmanager/vanilla prometheus/api/v2/status.json';
import datasources from './api/datasources.json';
import admin_config from './api/v1/ngalert/admin_config.json';
import alertmanagers from './api/v1/ngalert/alertmanagers.json';

export { datasources as DataSourcesResponse };
export { admin_config as AdminConfigResponse };
export { alertmanagers as AlertmanagersResponse };
export { internalAlertmanagerConfig as InternalAlertmanagerConfiguration };
export { vanillaAlertmanagerConfig as VanillaAlertmanagerConfiguration };
export { history as alertmanagerConfigurationHistory };

export const EXTERNAL_VANILLA_ALERTMANAGER_UID = 'vanilla-alertmanager';
export const PROVISIONED_MIMIR_ALERTMANAGER_UID = 'provisioned-alertmanager';

jest.spyOn(config, 'getAllDataSources');

const mocks = {
  getAllDataSources: jest.mocked(config.getAllDataSources),
};

const mockDataSources = {
  [EXTERNAL_VANILLA_ALERTMANAGER_UID]: mockDataSource<AlertManagerDataSourceJsonData>({
    uid: EXTERNAL_VANILLA_ALERTMANAGER_UID,
    name: EXTERNAL_VANILLA_ALERTMANAGER_UID,
    type: DataSourceType.Alertmanager,
    jsonData: {
      implementation: AlertManagerImplementation.prometheus,
    },
  }),
  [PROVISIONED_MIMIR_ALERTMANAGER_UID]: mockDataSource<AlertManagerDataSourceJsonData>({
    uid: PROVISIONED_MIMIR_ALERTMANAGER_UID,
    name: PROVISIONED_MIMIR_ALERTMANAGER_UID,
    type: DataSourceType.Alertmanager,
    jsonData: {
      // this is a mutable data source type but we're making it readOnly
      implementation: AlertManagerImplementation.mimir,
    },
    readOnly: true,
  }),
};

export function setupGrafanaManagedServer(server: SetupServerApi) {
  server.use(
    createAdminConfigHandler(),
    createExternalAlertmanagersHandler(),
    createAlertmanagerDataSourcesHandler(),
    ...createAlertmanagerConfigurationHandlers(),
    createAlertmanagerHistoryHandler()
  );

  return server;
}

export function setupVanillaAlertmanagerServer(server: SetupServerApi) {
  mocks.getAllDataSources.mockReturnValue(Object.values(mockDataSources));
  setDataSourceSrv(new MockDataSourceSrv(mockDataSources));

  server.use(
    createVanillaAlertmanagerConfigurationHandler(EXTERNAL_VANILLA_ALERTMANAGER_UID),
    ...createAlertmanagerConfigurationHandlers(PROVISIONED_MIMIR_ALERTMANAGER_UID)
  );

  return server;
}

const createAdminConfigHandler = () => http.get('/api/v1/ngalert/admin_config', () => HttpResponse.json(admin_config));

const createExternalAlertmanagersHandler = () => {
  return http.get('/api/v1/ngalert/alertmanagers', () => HttpResponse.json(alertmanagers));
};

const createAlertmanagerConfigurationHandlers = (name = 'grafana') => {
  return [
    http.get(`/api/alertmanager/${name}/config/api/v1/alerts`, () => HttpResponse.json(internalAlertmanagerConfig)),
    http.post(`/api/alertmanager/${name}/config/api/v1/alerts`, async () => {
      await delay(1000); // simulate some time
      return HttpResponse.json({ message: 'configuration created' });
    }),
  ];
};

const createAlertmanagerDataSourcesHandler = () => http.get('/api/datasources', () => HttpResponse.json(datasources));
const createAlertmanagerHistoryHandler = (name = 'grafana') =>
  http.get(`/api/alertmanager/${name}/config/history`, () => HttpResponse.json(history));

const createVanillaAlertmanagerConfigurationHandler = (dataSourceUID: string) =>
  http.get(`/api/alertmanager/${dataSourceUID}/api/v2/status`, () => HttpResponse.json(vanillaAlertmanagerConfig));

export const withExternalOnlySetting = (server: SetupServerApi) => {
  server.use(createAdminConfigHandler());
};
