import { cx, css } from '@emotion/css';
import React, { useState, useEffect } from 'react';
import useDebounce from 'react-use/lib/useDebounce';
import usePrevious from 'react-use/lib/usePrevious';
import { useStyles2, Input, Button, RadioButtonGroup, Menu, Dropdown } from '@grafana/ui';
import { byValueGradient, byPackageGradient, diffDefaultGradient, diffColorBlindGradient } from './FlameGraph/colors.js';
import { MIN_WIDTH_TO_SHOW_BOTH_TOPTABLE_AND_FLAMEGRAPH } from './constants.js';
import { SelectedView, ColorScheme, ColorSchemeDiff } from './types.js';

const FlameGraphHeader = ({
  search,
  setSearch,
  selectedView,
  setSelectedView,
  containerWidth,
  onReset,
  textAlign,
  onTextAlignChange,
  showResetButton,
  colorScheme,
  onColorSchemeChange,
  stickyHeader,
  extraHeaderElements,
  vertical,
  isDiffMode
}) => {
  const styles = useStyles2(getStyles, stickyHeader);
  const [localSearch, setLocalSearch] = useSearchInput(search, setSearch);
  const suffix = localSearch !== "" ? /* @__PURE__ */ React.createElement(
    Button,
    {
      icon: "times",
      fill: "text",
      size: "sm",
      onClick: () => {
        setSearch("");
        setLocalSearch("");
      }
    },
    "Clear"
  ) : null;
  return /* @__PURE__ */ React.createElement("div", { className: styles.header }, /* @__PURE__ */ React.createElement("div", { className: styles.inputContainer }, /* @__PURE__ */ React.createElement(
    Input,
    {
      value: localSearch || "",
      onChange: (v) => {
        setLocalSearch(v.currentTarget.value);
      },
      placeholder: "Search...",
      suffix
    }
  )), /* @__PURE__ */ React.createElement("div", { className: styles.rightContainer }, showResetButton && /* @__PURE__ */ React.createElement(
    Button,
    {
      variant: "secondary",
      fill: "outline",
      size: "sm",
      icon: "history-alt",
      tooltip: "Reset focus and sandwich state",
      onClick: () => {
        onReset();
      },
      className: styles.buttonSpacing,
      "aria-label": "Reset focus and sandwich state"
    }
  ), /* @__PURE__ */ React.createElement(ColorSchemeButton, { value: colorScheme, onChange: onColorSchemeChange, isDiffMode }), /* @__PURE__ */ React.createElement(
    RadioButtonGroup,
    {
      size: "sm",
      disabled: selectedView === SelectedView.TopTable,
      options: alignOptions,
      value: textAlign,
      onChange: onTextAlignChange,
      className: styles.buttonSpacing
    }
  ), /* @__PURE__ */ React.createElement(
    RadioButtonGroup,
    {
      size: "sm",
      options: getViewOptions(containerWidth, vertical),
      value: selectedView,
      onChange: setSelectedView
    }
  ), extraHeaderElements && /* @__PURE__ */ React.createElement("div", { className: styles.extraElements }, extraHeaderElements)));
};
function ColorSchemeButton(props) {
  const styles = useStyles2(getStyles, false);
  let menu = /* @__PURE__ */ React.createElement(Menu, null, /* @__PURE__ */ React.createElement(Menu.Item, { label: "By package name", onClick: () => props.onChange(ColorScheme.PackageBased) }), /* @__PURE__ */ React.createElement(Menu.Item, { label: "By value", onClick: () => props.onChange(ColorScheme.ValueBased) }));
  if (props.isDiffMode) {
    menu = /* @__PURE__ */ React.createElement(Menu, null, /* @__PURE__ */ React.createElement(Menu.Item, { label: "Default (green to red)", onClick: () => props.onChange(ColorSchemeDiff.Default) }), /* @__PURE__ */ React.createElement(Menu.Item, { label: "Color blind (blue to red)", onClick: () => props.onChange(ColorSchemeDiff.DiffColorBlind) }));
  }
  const colorDotStyle = {
    [ColorScheme.ValueBased]: styles.colorDotByValue,
    [ColorScheme.PackageBased]: styles.colorDotByPackage,
    [ColorSchemeDiff.DiffColorBlind]: styles.colorDotDiffColorBlind,
    [ColorSchemeDiff.Default]: styles.colorDotDiffDefault
  }[props.value] || styles.colorDotByValue;
  return /* @__PURE__ */ React.createElement(Dropdown, { overlay: menu }, /* @__PURE__ */ React.createElement(
    Button,
    {
      variant: "secondary",
      fill: "outline",
      size: "sm",
      tooltip: "Change color scheme",
      onClick: () => {
      },
      className: styles.buttonSpacing,
      "aria-label": "Change color scheme"
    },
    /* @__PURE__ */ React.createElement("span", { className: cx(styles.colorDot, colorDotStyle) })
  ));
}
const alignOptions = [
  { value: "left", description: "Align text left", icon: "align-left" },
  { value: "right", description: "Align text right", icon: "align-right" }
];
function getViewOptions(width, vertical) {
  let viewOptions = [
    { value: SelectedView.TopTable, label: "Top Table", description: "Only show top table" },
    { value: SelectedView.FlameGraph, label: "Flame Graph", description: "Only show flame graph" }
  ];
  if (width >= MIN_WIDTH_TO_SHOW_BOTH_TOPTABLE_AND_FLAMEGRAPH || vertical) {
    viewOptions.push({
      value: SelectedView.Both,
      label: "Both",
      description: "Show both the top table and flame graph"
    });
  }
  return viewOptions;
}
function useSearchInput(search, setSearch) {
  const [localSearchState, setLocalSearchState] = useState(search);
  const prevSearch = usePrevious(search);
  useDebounce(
    () => {
      setSearch(localSearchState);
    },
    250,
    [localSearchState]
  );
  useEffect(() => {
    if (prevSearch !== search && search !== localSearchState) {
      setLocalSearchState(search);
    }
  }, [search, prevSearch, localSearchState]);
  return [localSearchState, setLocalSearchState];
}
const getStyles = (theme, sticky) => ({
  header: css`
    label: header;
    display: flex;
    flex-wrap: wrap;
    justify-content: space-between;
    width: 100%;
    top: 0;
    ${sticky ? css`
          z-index: ${theme.zIndex.navbarFixed};
          position: sticky;
          padding-bottom: ${theme.spacing(1)};
          padding-top: ${theme.spacing(1)};
          background: ${theme.colors.background.primary};
        ` : ""};
  `,
  inputContainer: css`
    label: inputContainer;
    margin-right: 20px;
    flex-grow: 1;
    min-width: 150px;
    max-width: 350px;
  `,
  rightContainer: css`
    label: rightContainer;
    display: flex;
    align-items: flex-start;
    flex-wrap: wrap;
  `,
  buttonSpacing: css`
    label: buttonSpacing;
    margin-right: ${theme.spacing(1)};
  `,
  resetButton: css`
    label: resetButton;
    display: flex;
    margin-right: ${theme.spacing(2)};
  `,
  resetButtonIconWrapper: css`
    label: resetButtonIcon;
    padding: 0 5px;
    color: ${theme.colors.text.disabled};
  `,
  colorDot: css`
    label: colorDot;
    display: inline-block;
    width: 10px;
    height: 10px;
    border-radius: 50%;
  `,
  colorDotByValue: css`
    label: colorDotByValue;
    background: ${byValueGradient};
  `,
  colorDotByPackage: css`
    label: colorDotByPackage;
    background: ${byPackageGradient};
  `,
  colorDotDiffDefault: css`
    label: colorDotDiffDefault;
    background: ${diffDefaultGradient};
  `,
  colorDotDiffColorBlind: css`
    label: colorDotDiffColorBlind;
    background: ${diffColorBlindGradient};
  `,
  extraElements: css`
    label: extraElements;
    margin-left: ${theme.spacing(1)};
  `
});

export { FlameGraphHeader as default };
//# sourceMappingURL=FlameGraphHeader.js.map
