/// <reference types="cypress" />
import { Selectors, StringSelector, FunctionSelector, CssSelector, UrlSelector, E2ESelectors } from '@grafana/e2e-selectors';
import { Preferences } from '@grafana/schema/src/raw/preferences/x/preferences_types.gen';

interface DeleteDashboardConfig {
    quick?: boolean;
    title: string;
    uid: string;
}
declare const deleteDashboard: ({ quick, title, uid }: DeleteDashboardConfig) => void;

interface DeleteDataSourceConfig {
    id: string;
    name: string;
    quick?: boolean;
}
declare const deleteDataSource: ({ id, name, quick }: DeleteDataSourceConfig) => void;

interface ScenarioContext {
    addedDashboards: DeleteDashboardConfig[];
    addedDataSources: DeleteDataSourceConfig[];
    lastAddedDashboard: string;
    lastAddedDashboardUid: string;
    lastAddedDataSource: string;
    lastAddedDataSourceId: string;
    hasChangedUserPreferences: boolean;
    [key: string]: any;
}

type VisitFunction = (args?: string, queryParams?: object) => Cypress.Chainable<Window>;
type E2EVisit = {
    visit: VisitFunction;
};
type E2EFunction = ((text?: string, options?: CypressOptions) => Cypress.Chainable<JQuery<HTMLElement>>) & E2EFunctionWithOnlyOptions;
type E2EFunctionWithOnlyOptions = (options?: CypressOptions) => Cypress.Chainable<JQuery<HTMLElement>>;
type TypeSelectors<S> = S extends StringSelector ? E2EFunctionWithOnlyOptions : S extends FunctionSelector ? E2EFunction : S extends CssSelector ? E2EFunction : S extends UrlSelector ? E2EVisit & Omit<E2EFunctions<S>, 'url'> : S extends Record<any, any> ? E2EFunctions<S> : S;
type E2EFunctions<S extends Selectors> = {
    [P in keyof S]: TypeSelectors<S[P]>;
};
type CypressOptions = Partial<Cypress.Loggable & Cypress.Timeoutable & Cypress.Withinable & Cypress.Shadow>;

interface BenchmarkArguments {
    name: string;
    dashboard: {
        folder: string;
        delayAfterOpening: number;
        skipPanelValidation: boolean;
    };
    repeat: number;
    duration: number;
    appStats?: {
        startCollecting?: (window: Window) => void;
        collect: (window: Window) => Record<string, unknown>;
    };
    skipScenario?: boolean;
}

interface TimeRangeConfig {
    from: string;
    to: string;
    zone?: string;
}
declare const setTimeRange: ({ from, to, zone }: TimeRangeConfig) => void;

interface AddAnnotationConfig {
    dataSource: string;
    dataSourceForm?: () => void;
    name: string;
}
interface AddDashboardConfig {
    annotations: AddAnnotationConfig[];
    timeRange: TimeRangeConfig;
    title: string;
    variables: PartialAddVariableConfig[];
}
interface AddVariableDefault {
    hide: string;
    type: string;
}
interface AddVariableOptional {
    constantValue?: string;
    dataSource?: string;
    label?: string;
    query?: string;
    regex?: string;
    variableQueryForm?: (config: AddVariableConfig) => void;
}
interface AddVariableRequired {
    name: string;
}
type PartialAddVariableConfig = Partial<AddVariableDefault> & AddVariableOptional & AddVariableRequired;
type AddVariableConfig = AddVariableDefault & AddVariableOptional & AddVariableRequired;
/**
 * This flow is used to add a dashboard with whatever configuration specified.
 * @param config Configuration object. Currently supports configuring dashboard time range, annotations, and variables (support dependant on type).
 * @see{@link AddDashboardConfig}
 *
 * @example
 * ```
 * // Configuring a simple dashboard
 * addDashboard({
 *    timeRange: {
 *      from: '2022-10-03 00:00:00',
 *      to: '2022-10-03 23:59:59',
 *      zone: 'Coordinated Universal Time',
 *    },
 *    title: 'Test Dashboard',
 * })
 * ```
 *
 * @example
 * ```
 * // Configuring a dashboard with annotations
 * addDashboard({
 *    title: 'Test Dashboard',
 *    annotations: [
 *      {
 *        // This should match the datasource name
 *        dataSource: 'azure-monitor',
 *        name: 'Test Annotation',
 *        dataSourceForm: () => {
 *          // Insert steps to create annotation using datasource form
 *        }
 *      }
 *    ]
 * })
 * ```
 *
 * @see{@link AddAnnotationConfig}
 *
 * @example
 * ```
 * // Configuring a dashboard with variables
 * addDashboard({
 *    title: 'Test Dashboard',
 *    variables: [
 *      {
 *        name: 'test-query-variable',
 *        label: 'Testing Query',
 *        hide: '',
 *        type: e2e.flows.VARIABLE_TYPE_QUERY,
 *        dataSource: 'azure-monitor',
 *        variableQueryForm: () => {
 *          // Insert steps to create variable using datasource form
 *        },
 *      },
 *      {
 *        name: 'test-constant-variable',
 *        label: 'Testing Constant',
 *        type: e2e.flows.VARIABLE_TYPE_CONSTANT,
 *        constantValue: 'constant',
 *      }
 *    ]
 * })
 * ```
 *
 * @see{@link AddVariableConfig}
 *
 * @see{@link https://github.com/grafana/grafana/blob/main/e2e/cloud-plugins-suite/azure-monitor.spec.ts Azure Monitor Tests for full examples}
 */
declare const addDashboard: (config?: Partial<AddDashboardConfig>) => Cypress.Chainable<{
    config: AddDashboardConfig;
    uid: string;
}>;
declare const VARIABLE_HIDE_LABEL = "Label";
declare const VARIABLE_HIDE_NOTHING = "";
declare const VARIABLE_HIDE_VARIABLE = "Variable";
declare const VARIABLE_TYPE_AD_HOC_FILTERS = "Ad hoc filters";
declare const VARIABLE_TYPE_CONSTANT = "Constant";
declare const VARIABLE_TYPE_DATASOURCE = "Datasource";
declare const VARIABLE_TYPE_QUERY = "Query";

interface AddDataSourceConfig {
    basicAuth: boolean;
    basicAuthPassword: string;
    basicAuthUser: string;
    expectedAlertMessage: string | RegExp;
    form: () => void;
    name: string;
    skipTlsVerify: boolean;
    type: string;
    timeout?: number;
    awaitHealth?: boolean;
}
declare const addDataSource: (config?: Partial<AddDataSourceConfig>) => Cypress.Chainable<{
    config: AddDataSourceConfig;
}>;

interface AddPanelOverrides {
    dataSourceName: string;
    queriesForm: (config: AddPanelConfig) => void;
    panelTitle: string;
}
interface EditPanelOverrides {
    queriesForm?: (config: EditPanelConfig) => void;
    panelTitle: string;
}
interface ConfigurePanelDefault {
    chartData: {
        method: string;
        route: string | RegExp;
    };
    dashboardUid: string;
    matchScreenshot: boolean;
    saveDashboard: boolean;
    screenshotName: string;
    visitDashboardAtStart: boolean;
}
interface ConfigurePanelOptional {
    dataSourceName?: string;
    queriesForm?: (config: ConfigurePanelConfig) => void;
    panelTitle?: string;
    timeRange?: TimeRangeConfig;
    visualizationName?: string;
    timeout?: number;
}
interface ConfigurePanelRequired {
    isEdit: boolean;
}
type PartialConfigurePanelConfig = Partial<ConfigurePanelDefault> & ConfigurePanelOptional & ConfigurePanelRequired;
type ConfigurePanelConfig = ConfigurePanelDefault & ConfigurePanelOptional & ConfigurePanelRequired;
type PartialAddPanelConfig = PartialConfigurePanelConfig & AddPanelOverrides;
type AddPanelConfig = ConfigurePanelConfig & AddPanelOverrides;
type PartialEditPanelConfig = PartialConfigurePanelConfig & EditPanelOverrides;
type EditPanelConfig = ConfigurePanelConfig & EditPanelOverrides;
declare const VISUALIZATION_ALERT_LIST = "Alert list";
declare const VISUALIZATION_BAR_GAUGE = "Bar gauge";
declare const VISUALIZATION_CLOCK = "Clock";
declare const VISUALIZATION_DASHBOARD_LIST = "Dashboard list";
declare const VISUALIZATION_GAUGE = "Gauge";
declare const VISUALIZATION_GRAPH = "Graph";
declare const VISUALIZATION_HEAT_MAP = "Heatmap";
declare const VISUALIZATION_LOGS = "Logs";
declare const VISUALIZATION_NEWS = "News";
declare const VISUALIZATION_PIE_CHART = "Pie Chart";
declare const VISUALIZATION_PLUGIN_LIST = "Plugin list";
declare const VISUALIZATION_POLYSTAT = "Polystat";
declare const VISUALIZATION_STAT = "Stat";
declare const VISUALIZATION_TABLE = "Table";
declare const VISUALIZATION_TEXT = "Text";
declare const VISUALIZATION_WORLD_MAP = "Worldmap Panel";

declare const addPanel: (config?: Partial<PartialAddPanelConfig>) => Cypress.Chainable<{
    config: ConfigurePanelConfig | AddPanelConfig | EditPanelConfig;
}>;

declare const assertSuccessNotification: () => void;

declare const editPanel: (config: Partial<PartialEditPanelConfig>) => Cypress.Chainable<{
    config: ConfigurePanelConfig | AddPanelConfig | EditPanelConfig;
}>;

declare const login: (username?: string, password?: string, loginViaApi?: boolean) => void;

interface OpenDashboardDefault {
    uid: string;
}
interface OpenDashboardOptional {
    timeRange?: TimeRangeConfig;
    queryParams?: object;
}
type PartialOpenDashboardConfig = Partial<OpenDashboardDefault> & OpenDashboardOptional;
type OpenDashboardConfig = OpenDashboardDefault & OpenDashboardOptional;
declare const openDashboard: (config?: PartialOpenDashboardConfig) => Cypress.Chainable<{
    config: OpenDashboardConfig;
}>;

declare enum PanelMenuItems {
    Edit = "Edit",
    Inspect = "Inspect",
    More = "More...",
    Extensions = "Extensions"
}
declare const openPanelMenuItem: (menu: PanelMenuItems, panelTitle?: string) => void;
declare const openPanelMenuExtension: (extensionTitle: string, panelTitle?: string) => void;

declare const revertAllChanges: () => void;

declare const saveDashboard: () => void;

interface SelectOptionConfig {
    clickToOpen?: boolean;
    container: any;
    forceClickOption?: boolean;
    optionText: string | RegExp;
}
declare const selectOption: (config: SelectOptionConfig) => any;

type Panel = {
    title: string;
    [key: string]: unknown;
};
type Dashboard = {
    title: string;
    panels: Panel[];
    uid: string;
    [key: string]: unknown;
};
/**
 * Smoke test a particular dashboard by quickly importing a json file and validate that all the panels finish loading
 * @param dashboardToImport a sample dashboard
 * @param queryTimeout a number of ms to wait for the imported dashboard to finish loading
 * @param skipPanelValidation skip panel validation
 */
declare const importDashboard: (dashboardToImport: Dashboard, queryTimeout?: number, skipPanelValidation?: boolean) => void;

/**
 * Smoke test several dashboard json files from a test directory
 * and validate that all the panels in each import finish loading their queries
 * @param dirPath the relative path to a directory which contains json files representing dashboards,
 * for example if your dashboards live in `cypress/testDashboards` you can pass `/testDashboards`
 * @param queryTimeout a number of ms to wait for the imported dashboard to finish loading
 * @param skipPanelValidation skips panel validation
 */
declare const importDashboards: (dirPath: string, queryTimeout?: number, skipPanelValidation?: boolean) => Promise<void>;

declare const defaultUserPreferences: {
    readonly timezone: "";
};
type UserPreferences = Pick<Preferences, keyof typeof defaultUserPreferences>;
declare function setUserPreferences(prefs: UserPreferences): Cypress.Chainable<Cypress.Response<any>>;
declare function setDefaultUserPreferences(): Cypress.Chainable<Cypress.Response<any>>;

declare const flows_VISUALIZATION_ALERT_LIST: typeof VISUALIZATION_ALERT_LIST;
declare const flows_VISUALIZATION_BAR_GAUGE: typeof VISUALIZATION_BAR_GAUGE;
declare const flows_VISUALIZATION_CLOCK: typeof VISUALIZATION_CLOCK;
declare const flows_VISUALIZATION_DASHBOARD_LIST: typeof VISUALIZATION_DASHBOARD_LIST;
declare const flows_VISUALIZATION_GAUGE: typeof VISUALIZATION_GAUGE;
declare const flows_VISUALIZATION_GRAPH: typeof VISUALIZATION_GRAPH;
declare const flows_VISUALIZATION_HEAT_MAP: typeof VISUALIZATION_HEAT_MAP;
declare const flows_VISUALIZATION_LOGS: typeof VISUALIZATION_LOGS;
declare const flows_VISUALIZATION_NEWS: typeof VISUALIZATION_NEWS;
declare const flows_VISUALIZATION_PIE_CHART: typeof VISUALIZATION_PIE_CHART;
declare const flows_VISUALIZATION_PLUGIN_LIST: typeof VISUALIZATION_PLUGIN_LIST;
declare const flows_VISUALIZATION_POLYSTAT: typeof VISUALIZATION_POLYSTAT;
declare const flows_VISUALIZATION_STAT: typeof VISUALIZATION_STAT;
declare const flows_VISUALIZATION_TABLE: typeof VISUALIZATION_TABLE;
declare const flows_VISUALIZATION_TEXT: typeof VISUALIZATION_TEXT;
declare const flows_VISUALIZATION_WORLD_MAP: typeof VISUALIZATION_WORLD_MAP;
type flows_AddAnnotationConfig = AddAnnotationConfig;
type flows_AddDashboardConfig = AddDashboardConfig;
type flows_PartialAddVariableConfig = PartialAddVariableConfig;
type flows_AddVariableConfig = AddVariableConfig;
declare const flows_addDashboard: typeof addDashboard;
declare const flows_VARIABLE_HIDE_LABEL: typeof VARIABLE_HIDE_LABEL;
declare const flows_VARIABLE_HIDE_NOTHING: typeof VARIABLE_HIDE_NOTHING;
declare const flows_VARIABLE_HIDE_VARIABLE: typeof VARIABLE_HIDE_VARIABLE;
declare const flows_VARIABLE_TYPE_AD_HOC_FILTERS: typeof VARIABLE_TYPE_AD_HOC_FILTERS;
declare const flows_VARIABLE_TYPE_CONSTANT: typeof VARIABLE_TYPE_CONSTANT;
declare const flows_VARIABLE_TYPE_DATASOURCE: typeof VARIABLE_TYPE_DATASOURCE;
declare const flows_VARIABLE_TYPE_QUERY: typeof VARIABLE_TYPE_QUERY;
type flows_AddDataSourceConfig = AddDataSourceConfig;
declare const flows_addDataSource: typeof addDataSource;
declare const flows_addPanel: typeof addPanel;
declare const flows_assertSuccessNotification: typeof assertSuccessNotification;
type flows_DeleteDashboardConfig = DeleteDashboardConfig;
declare const flows_deleteDashboard: typeof deleteDashboard;
type flows_DeleteDataSourceConfig = DeleteDataSourceConfig;
declare const flows_deleteDataSource: typeof deleteDataSource;
declare const flows_editPanel: typeof editPanel;
declare const flows_login: typeof login;
type flows_PartialOpenDashboardConfig = PartialOpenDashboardConfig;
type flows_OpenDashboardConfig = OpenDashboardConfig;
declare const flows_openDashboard: typeof openDashboard;
type flows_PanelMenuItems = PanelMenuItems;
declare const flows_PanelMenuItems: typeof PanelMenuItems;
declare const flows_openPanelMenuItem: typeof openPanelMenuItem;
declare const flows_openPanelMenuExtension: typeof openPanelMenuExtension;
declare const flows_revertAllChanges: typeof revertAllChanges;
declare const flows_saveDashboard: typeof saveDashboard;
type flows_SelectOptionConfig = SelectOptionConfig;
declare const flows_selectOption: typeof selectOption;
type flows_TimeRangeConfig = TimeRangeConfig;
declare const flows_setTimeRange: typeof setTimeRange;
type flows_Dashboard = Dashboard;
declare const flows_importDashboard: typeof importDashboard;
declare const flows_importDashboards: typeof importDashboards;
declare const flows_setUserPreferences: typeof setUserPreferences;
declare const flows_setDefaultUserPreferences: typeof setDefaultUserPreferences;
declare namespace flows {
  export {
    flows_VISUALIZATION_ALERT_LIST as VISUALIZATION_ALERT_LIST,
    flows_VISUALIZATION_BAR_GAUGE as VISUALIZATION_BAR_GAUGE,
    flows_VISUALIZATION_CLOCK as VISUALIZATION_CLOCK,
    flows_VISUALIZATION_DASHBOARD_LIST as VISUALIZATION_DASHBOARD_LIST,
    flows_VISUALIZATION_GAUGE as VISUALIZATION_GAUGE,
    flows_VISUALIZATION_GRAPH as VISUALIZATION_GRAPH,
    flows_VISUALIZATION_HEAT_MAP as VISUALIZATION_HEAT_MAP,
    flows_VISUALIZATION_LOGS as VISUALIZATION_LOGS,
    flows_VISUALIZATION_NEWS as VISUALIZATION_NEWS,
    flows_VISUALIZATION_PIE_CHART as VISUALIZATION_PIE_CHART,
    flows_VISUALIZATION_PLUGIN_LIST as VISUALIZATION_PLUGIN_LIST,
    flows_VISUALIZATION_POLYSTAT as VISUALIZATION_POLYSTAT,
    flows_VISUALIZATION_STAT as VISUALIZATION_STAT,
    flows_VISUALIZATION_TABLE as VISUALIZATION_TABLE,
    flows_VISUALIZATION_TEXT as VISUALIZATION_TEXT,
    flows_VISUALIZATION_WORLD_MAP as VISUALIZATION_WORLD_MAP,
    flows_AddAnnotationConfig as AddAnnotationConfig,
    flows_AddDashboardConfig as AddDashboardConfig,
    flows_PartialAddVariableConfig as PartialAddVariableConfig,
    flows_AddVariableConfig as AddVariableConfig,
    flows_addDashboard as addDashboard,
    flows_VARIABLE_HIDE_LABEL as VARIABLE_HIDE_LABEL,
    flows_VARIABLE_HIDE_NOTHING as VARIABLE_HIDE_NOTHING,
    flows_VARIABLE_HIDE_VARIABLE as VARIABLE_HIDE_VARIABLE,
    flows_VARIABLE_TYPE_AD_HOC_FILTERS as VARIABLE_TYPE_AD_HOC_FILTERS,
    flows_VARIABLE_TYPE_CONSTANT as VARIABLE_TYPE_CONSTANT,
    flows_VARIABLE_TYPE_DATASOURCE as VARIABLE_TYPE_DATASOURCE,
    flows_VARIABLE_TYPE_QUERY as VARIABLE_TYPE_QUERY,
    flows_AddDataSourceConfig as AddDataSourceConfig,
    flows_addDataSource as addDataSource,
    flows_addPanel as addPanel,
    flows_assertSuccessNotification as assertSuccessNotification,
    flows_DeleteDashboardConfig as DeleteDashboardConfig,
    flows_deleteDashboard as deleteDashboard,
    flows_DeleteDataSourceConfig as DeleteDataSourceConfig,
    flows_deleteDataSource as deleteDataSource,
    flows_editPanel as editPanel,
    flows_login as login,
    flows_PartialOpenDashboardConfig as PartialOpenDashboardConfig,
    flows_OpenDashboardConfig as OpenDashboardConfig,
    flows_openDashboard as openDashboard,
    flows_PanelMenuItems as PanelMenuItems,
    flows_openPanelMenuItem as openPanelMenuItem,
    flows_openPanelMenuExtension as openPanelMenuExtension,
    flows_revertAllChanges as revertAllChanges,
    flows_saveDashboard as saveDashboard,
    flows_SelectOptionConfig as SelectOptionConfig,
    flows_selectOption as selectOption,
    flows_TimeRangeConfig as TimeRangeConfig,
    flows_setTimeRange as setTimeRange,
    flows_Dashboard as Dashboard,
    flows_importDashboard as importDashboard,
    flows_importDashboards as importDashboards,
    flows_setUserPreferences as setUserPreferences,
    flows_setDefaultUserPreferences as setDefaultUserPreferences,
  };
}

interface ScenarioArguments {
    describeName: string;
    itName: string;
    scenario: Function;
    skipScenario?: boolean;
    addScenarioDataSource?: boolean;
    addScenarioDashBoard?: boolean;
    loginViaApi?: boolean;
}

declare const undo: () => "{cmd}z" | "{ctrl}z";

declare const typings_undo: typeof undo;
declare namespace typings {
  export {
    typings_undo as undo,
  };
}

declare const e2eObject: {
    env: (args: string) => any;
    config: () => Cypress.Config;
    blobToBase64String: (blob: Blob) => Promise<string>;
    imgSrcToBlob: (url: string) => Promise<Blob>;
    scenario: (args: ScenarioArguments) => void;
    benchmark: ({ name, skipScenario, repeat, duration, appStats, dashboard, }: BenchmarkArguments) => void;
    pages: E2EFunctions<E2ESelectors<{
        Login: {
            url: string;
            username: string;
            password: string;
            submit: string;
            skip: string;
        };
        Home: {
            url: string;
        };
        DataSource: {
            name: string;
            delete: string;
            readOnly: string;
            saveAndTest: string;
            alert: string;
        };
        DataSources: {
            url: string;
            dataSources: (dataSourceName: string) => string;
        };
        EditDataSource: {
            url: (dataSourceUid: string) => string;
            settings: string;
        };
        AddDataSource: {
            url: string;
            dataSourcePlugins: (pluginName: string) => string;
            dataSourcePluginsV2: (pluginName: string) => string;
        };
        ConfirmModal: {
            delete: string;
        };
        AddDashboard: {
            url: string;
            itemButton: (title: string) => string;
            addNewPanel: string;
            addNewRow: string;
            addNewPanelLibrary: string;
        };
        Dashboard: {
            url: (uid: string) => string;
            DashNav: {
                nav: string;
                navV2: string;
                publicDashboardTag: string;
                shareButton: string;
            };
            SubMenu: {
                submenu: string;
                submenuItem: string;
                submenuItemLabels: (item: string) => string;
                submenuItemValueDropDownValueLinkTexts: (item: string) => string;
                submenuItemValueDropDownDropDown: string;
                submenuItemValueDropDownOptionTexts: (item: string) => string;
                Annotations: {
                    annotationsWrapper: string;
                    annotationLabel: (label: string) => string;
                    annotationToggle: (label: string) => string;
                };
            };
            Settings: {
                Actions: {
                    close: string;
                };
                General: {
                    deleteDashBoard: string;
                    sectionItems: (item: string) => string;
                    saveDashBoard: string;
                    saveAsDashBoard: string;
                    timezone: string;
                    title: string;
                };
                Annotations: {
                    List: {
                        addAnnotationCTA: string;
                        addAnnotationCTAV2: string;
                        annotations: string;
                    };
                    Settings: {
                        name: string;
                    };
                    NewAnnotation: {
                        panelFilterSelect: string;
                        showInLabel: string;
                        previewInDashboard: string;
                        delete: string;
                        apply: string;
                        enable: string;
                        hide: string;
                    };
                };
                Variables: {
                    List: {
                        addVariableCTA: string;
                        addVariableCTAV2: string;
                        newButton: string;
                        table: string;
                        tableRowNameFields: (variableName: string) => string;
                        tableRowDefinitionFields: (variableName: string) => string;
                        tableRowArrowUpButtons: (variableName: string) => string;
                        tableRowArrowDownButtons: (variableName: string) => string;
                        tableRowDuplicateButtons: (variableName: string) => string;
                        tableRowRemoveButtons: (variableName: string) => string;
                    };
                    Edit: {
                        General: {
                            headerLink: string;
                            modeLabelNew: string;
                            modeLabelEdit: string;
                            generalNameInput: string;
                            generalNameInputV2: string;
                            generalTypeSelect: string;
                            generalTypeSelectV2: string;
                            generalLabelInput: string;
                            generalLabelInputV2: string;
                            generalHideSelect: string;
                            generalHideSelectV2: string;
                            selectionOptionsMultiSwitch: string;
                            selectionOptionsIncludeAllSwitch: string;
                            selectionOptionsCustomAllInput: string;
                            previewOfValuesOption: string;
                            submitButton: string;
                            applyButton: string;
                        };
                        QueryVariable: {
                            queryOptionsDataSourceSelect: string;
                            queryOptionsRefreshSelect: string;
                            queryOptionsRefreshSelectV2: string;
                            queryOptionsRegExInput: string;
                            queryOptionsRegExInputV2: string;
                            queryOptionsSortSelect: string;
                            queryOptionsSortSelectV2: string;
                            queryOptionsQueryInput: string;
                            valueGroupsTagsEnabledSwitch: string;
                            valueGroupsTagsTagsQueryInput: string;
                            valueGroupsTagsTagsValuesQueryInput: string;
                        };
                        ConstantVariable: {
                            constantOptionsQueryInput: string;
                            constantOptionsQueryInputV2: string;
                        };
                        DatasourceVariable: {
                            datasourceSelect: string;
                        };
                        TextBoxVariable: {
                            textBoxOptionsQueryInput: string;
                            textBoxOptionsQueryInputV2: string;
                        };
                        CustomVariable: {
                            customValueInput: string;
                        };
                        IntervalVariable: {
                            intervalsValueInput: string;
                            autoEnabledCheckbox: string;
                            stepCountIntervalSelect: string;
                            minIntervalInput: string;
                        };
                        GroupByVariable: {
                            dataSourceSelect: string;
                            infoText: string;
                            modeToggle: string;
                        };
                        AdHocFiltersVariable: {
                            datasourceSelect: string;
                            infoText: string;
                        };
                    };
                };
            };
            Annotations: {
                marker: string;
            };
            Rows: {
                Repeated: {
                    ConfigSection: {
                        warningMessage: string;
                    };
                };
            };
        };
        Dashboards: {
            url: string;
            dashboards: (title: string) => string;
        };
        SaveDashboardAsModal: {
            newName: string;
            save: string;
        };
        SaveDashboardModal: {
            save: string;
            saveVariables: string;
            saveTimerange: string;
        };
        SharePanelModal: {
            linkToRenderedImage: string;
        };
        ShareDashboardModal: {
            PublicDashboard: {
                Tab: string;
                WillBePublicCheckbox: string;
                LimitedDSCheckbox: string;
                CostIncreaseCheckbox: string;
                PauseSwitch: string;
                EnableAnnotationsSwitch: string;
                CreateButton: string;
                DeleteButton: string;
                CopyUrlInput: string;
                CopyUrlButton: string;
                SettingsDropdown: string;
                TemplateVariablesWarningAlert: string;
                UnsupportedDataSourcesWarningAlert: string;
                NoUpsertPermissionsWarningAlert: string;
                EnableTimeRangeSwitch: string;
                EmailSharingConfiguration: {
                    Container: string;
                    ShareType: string;
                    EmailSharingInput: string;
                    EmailSharingInviteButton: string;
                    EmailSharingList: string;
                    DeleteEmail: string;
                    ReshareLink: string;
                };
            };
        };
        PublicDashboard: {
            page: string;
            NotAvailable: {
                container: string;
                title: string;
                pausedDescription: string;
            };
        };
        RequestViewAccess: {
            form: string;
            recipientInput: string;
            submitButton: string;
        };
        PublicDashboardConfirmAccess: {
            submitButton: string;
        };
        Explore: {
            url: string;
            General: {
                container: string;
                graph: string;
                table: string;
                scrollView: string;
            };
        };
        SoloPanel: {
            url: (page: string) => string;
        };
        PluginsList: {
            page: string;
            list: string;
            listItem: string;
            signatureErrorNotice: string;
        };
        PluginPage: {
            page: string;
            signatureInfo: string;
            disabledInfo: string;
        };
        PlaylistForm: {
            name: string;
            interval: string;
            itemDelete: string;
        };
        BrowseDashboards: {
            table: {
                body: string;
                row: (name: string) => string;
                checkbox: (uid: string) => string;
            };
            NewFolderForm: {
                form: string;
                nameInput: string;
                createButton: string;
            };
        };
        Search: {
            url: string;
            FolderView: {
                url: string;
            };
        };
        PublicDashboards: {
            ListItem: {
                linkButton: string;
                configButton: string;
                trashcanButton: string;
                pauseSwitch: string;
            };
        };
        UserListPage: {
            tabs: {
                allUsers: string;
                orgUsers: string;
                anonUserDevices: string;
                publicDashboardsUsers: string;
                users: string;
            };
            org: {
                url: string;
            };
            admin: {
                url: string;
            };
            publicDashboards: {
                container: string;
            };
            UserListAdminPage: {
                container: string;
            };
            UsersListPage: {
                container: string;
            };
            UserAnonListPage: {
                container: string;
            };
            UsersListPublicDashboardsPage: {
                container: string;
                DashboardsListModal: {
                    listItem: (uid: string) => string;
                };
            };
        };
        ProfilePage: {
            url: string;
        };
    }>>;
    typings: typeof typings;
    components: E2EFunctions<E2ESelectors<{
        RadioButton: {
            container: string;
        };
        Breadcrumbs: {
            breadcrumb: (title: string) => string;
        };
        TimePicker: {
            openButton: string;
            overlayContent: string;
            fromField: string;
            toField: string;
            applyTimeRange: string;
            copyTimeRange: string;
            pasteTimeRange: string;
            calendar: {
                label: string;
                openButton: string;
                closeButton: string;
            };
            absoluteTimeRangeTitle: string;
        };
        DataSourcePermissions: {
            form: () => string;
            roleType: string;
            rolePicker: string;
            permissionLevel: string;
        };
        DateTimePicker: {
            input: string;
        };
        DataSource: {
            TestData: {
                QueryTab: {
                    scenarioSelectContainer: string;
                    scenarioSelect: string;
                    max: string;
                    min: string;
                    noise: string;
                    seriesCount: string;
                    spread: string;
                    startValue: string;
                    drop: string;
                };
            };
            DataSourceHttpSettings: {
                urlInput: string;
            };
            Jaeger: {
                traceIDInput: string;
            };
            Prometheus: {
                configPage: {
                    connectionSettings: string;
                    manageAlerts: string;
                    scrapeInterval: string;
                    queryTimeout: string;
                    defaultEditor: string;
                    disableMetricLookup: string;
                    prometheusType: string;
                    prometheusVersion: string;
                    cacheLevel: string;
                    incrementalQuerying: string;
                    queryOverlapWindow: string;
                    disableRecordingRules: string;
                    customQueryParameters: string;
                    httpMethod: string;
                    exemplarsAddButton: string;
                    internalLinkSwitch: string;
                };
                queryEditor: {
                    explain: string;
                    editorToggle: string;
                    options: string;
                    legend: string;
                    format: string;
                    step: string;
                    type: string;
                    exemplars: string;
                    builder: {
                        metricSelect: string;
                        hints: string;
                        metricsExplorer: string;
                        queryAdvisor: string;
                    };
                    code: {
                        queryField: string;
                        metricsBrowser: {
                            openButton: string;
                            selectMetric: string;
                            metricList: string;
                            labelNamesFilter: string;
                            labelValuesFilter: string;
                            useQuery: string;
                            useAsRateQuery: string;
                            validateSelector: string;
                            clear: string;
                        };
                    };
                };
                exemplarMarker: string;
                variableQueryEditor: {
                    queryType: string;
                    labelnames: {
                        metricRegex: string;
                    };
                    labelValues: {
                        labelSelect: string;
                    };
                    metricNames: {
                        metricRegex: string;
                    };
                    varQueryResult: string;
                    seriesQuery: string;
                    classicQuery: string;
                };
                annotations: {
                    minStep: string;
                    title: string;
                    tags: string;
                    text: string;
                    seriesValueAsTimestamp: string;
                };
            };
        };
        Menu: {
            MenuComponent: (title: string) => string;
            MenuGroup: (title: string) => string;
            MenuItem: (title: string) => string;
            SubMenu: {
                container: string;
                icon: string;
            };
        };
        Panels: {
            Panel: {
                title: (title: string) => string;
                headerItems: (item: string) => string;
                menuItems: (item: string) => string;
                menu: (title: string) => string;
                containerByTitle: (title: string) => string;
                headerCornerInfo: (mode: string) => string;
                status: (status: string) => string;
                loadingBar: () => string;
                HoverWidget: {
                    container: string;
                    dragIcon: string;
                };
                PanelDataErrorMessage: string;
            };
            Visualization: {
                Graph: {
                    container: string;
                    VisualizationTab: {
                        legendSection: string;
                    };
                    Legend: {
                        legendItemAlias: (name: string) => string;
                        showLegendSwitch: string;
                    };
                    xAxis: {
                        labels: () => string;
                    };
                };
                BarGauge: {
                    value: string;
                    valueV2: string;
                };
                PieChart: {
                    svgSlice: string;
                };
                Text: {
                    container: () => string;
                };
                Table: {
                    header: string;
                    footer: string;
                    body: string;
                };
            };
        };
        VizLegend: {
            seriesName: (name: string) => string;
        };
        Drawer: {
            General: {
                title: (title: string) => string;
                expand: string;
                contract: string;
                close: string;
                rcContentWrapper: () => string;
                subtitle: string;
            };
        };
        PanelEditor: {
            General: {
                content: string;
            };
            OptionsPane: {
                content: string;
                select: string;
                fieldLabel: (type: string) => string;
            };
            DataPane: {
                content: string;
            };
            applyButton: string;
            toggleVizPicker: string;
            toggleVizOptions: string;
            toggleTableView: string;
            showZoomField: string;
            showAttributionField: string;
            showScaleField: string;
            showMeasureField: string;
            showDebugField: string;
            measureButton: string;
        };
        PanelInspector: {
            Data: {
                content: string;
            };
            Stats: {
                content: string;
            };
            Json: {
                content: string;
            };
            Query: {
                content: string;
                refreshButton: string;
                jsonObjectKeys: () => string;
            };
        };
        Tab: {
            title: (title: string) => string;
            active: () => string;
        };
        RefreshPicker: {
            runButton: string;
            intervalButton: string;
            runButtonV2: string;
            intervalButtonV2: string;
        };
        QueryTab: {
            content: string;
            queryInspectorButton: string;
            queryHistoryButton: string;
            addQuery: string;
        };
        QueryHistory: {
            queryText: string;
        };
        QueryEditorRows: {
            rows: string;
        };
        QueryEditorRow: {
            actionButton: (title: string) => string;
            title: (refId: string) => string;
            container: (refId: string) => string;
        };
        AlertTab: {
            content: string;
        };
        Alert: {
            alert: (severity: string) => string;
            alertV2: (severity: string) => string;
        };
        TransformTab: {
            content: string;
            newTransform: (name: string) => string;
            transformationEditor: (name: string) => string;
            transformationEditorDebugger: (name: string) => string;
        };
        Transforms: {
            card: (name: string) => string;
            disableTransformationButton: string;
            Reduce: {
                modeLabel: string;
                calculationsLabel: string;
            };
            SpatialOperations: {
                actionLabel: string;
                locationLabel: string;
                location: {
                    autoOption: string;
                    coords: {
                        option: string;
                        latitudeFieldLabel: string;
                        longitudeFieldLabel: string;
                    };
                    geohash: {
                        option: string;
                        geohashFieldLabel: string;
                    };
                    lookup: {
                        option: string;
                        lookupFieldLabel: string;
                        gazetteerFieldLabel: string;
                    };
                };
            };
            searchInput: string;
            noTransformationsMessage: string;
            addTransformationButton: string;
            removeAllTransformationsButton: string;
        };
        NavBar: {
            Configuration: {
                button: string;
            };
            Toggle: {
                button: string;
            };
            Reporting: {
                button: string;
            };
        };
        NavMenu: {
            Menu: string;
            item: string;
        };
        NavToolbar: {
            container: string;
        };
        PageToolbar: {
            container: () => string;
            item: (tooltip: string) => string;
            itemButton: (title: string) => string;
        };
        QueryEditorToolbarItem: {
            button: (title: string) => string;
        };
        BackButton: {
            backArrow: string;
        };
        OptionsGroup: {
            group: (title?: string | undefined) => string;
            toggle: (title?: string | undefined) => string;
        };
        PluginVisualization: {
            item: (title: string) => string;
            current: () => string;
        };
        Select: {
            option: string;
            input: () => string;
            singleValue: () => string;
        };
        FieldConfigEditor: {
            content: string;
        };
        OverridesConfigEditor: {
            content: string;
        };
        FolderPicker: {
            container: string;
            containerV2: string;
            input: string;
        };
        ReadonlyFolderPicker: {
            container: string;
        };
        DataSourcePicker: {
            container: string;
            input: () => string;
            inputV2: string;
            dataSourceList: string;
            advancedModal: {
                dataSourceList: string;
                builtInDataSourceList: string;
            };
        };
        TimeZonePicker: {
            container: string;
            containerV2: string;
        };
        WeekStartPicker: {
            container: string;
            containerV2: string;
            placeholder: string;
        };
        TraceViewer: {
            spanBar: string;
        };
        QueryField: {
            container: string;
        };
        QueryBuilder: {
            queryPatterns: string;
            labelSelect: string;
            valueSelect: string;
            matchOperatorSelect: string;
        };
        ValuePicker: {
            button: (name: string) => string;
            select: (name: string) => string;
        };
        Search: {
            section: string;
            sectionV2: string;
            items: string;
            itemsV2: string;
            cards: string;
            collapseFolder: (sectionId: string) => string;
            expandFolder: (sectionId: string) => string;
            dashboardItem: (item: string) => string;
            dashboardCard: (item: string) => string;
            folderHeader: (folderName: string) => string;
            folderContent: (folderName: string) => string;
            dashboardItems: string;
        };
        DashboardLinks: {
            container: string;
            dropDown: string;
            link: string;
        };
        LoadingIndicator: {
            icon: string;
        };
        CallToActionCard: {
            button: (name: string) => string;
            buttonV2: (name: string) => string;
        };
        DataLinksContextMenu: {
            singleLink: string;
        };
        CodeEditor: {
            container: string;
        };
        DashboardImportPage: {
            textarea: string;
            submit: string;
        };
        ImportDashboardForm: {
            name: string;
            submit: string;
        };
        PanelAlertTabContent: {
            content: string;
        };
        VisualizationPreview: {
            card: (name: string) => string;
        };
        ColorSwatch: {
            name: string;
        };
        DashboardRow: {
            title: (title: string) => string;
        };
        UserProfile: {
            profileSaveButton: string;
            preferencesSaveButton: string;
            orgsTable: string;
            sessionsTable: string;
            extensionPointTabs: string;
            extensionPointTab: (tabId: string) => string;
        };
        FileUpload: {
            inputField: string;
            fileNameSpan: string;
        };
        DebugOverlay: {
            wrapper: string;
        };
        OrgRolePicker: {
            input: string;
        };
        AnalyticsToolbarButton: {
            button: string;
        };
        Variables: {
            variableOption: string;
        };
        Annotations: {
            annotationsTypeInput: string;
            annotationsChoosePanelInput: string;
        };
        Tooltip: {
            container: string;
        };
    }>>;
    flows: typeof flows;
    getScenarioContext: () => Cypress.Chainable<ScenarioContext>;
    setScenarioContext: (newContext: Partial<ScenarioContext>) => Cypress.Chainable<ScenarioContext>;
    getSelectors: <T extends Selectors>(selectors: E2ESelectors<T>) => E2EFunctions<E2ESelectors<T>>;
};
declare const e2e: (() => Cypress.cy) & typeof e2eObject;

export { e2e };
